﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.ComponentModel;
using System.Xml;
using System.Windows.Controls.Primitives;
using System.Windows.Interop;
using System.Runtime.InteropServices;

namespace Nintendo.AudioToolKit.Controls
{
    /// <summary>
    ///
    /// </summary>
    public class TimelineControl : Selector
    {
        public TimelineControl()
        {
            //DefaultStyleKeyProperty.OverrideMetadata
            //    (typeof(TimelineControl), new FrameworkPropertyMetadata(typeof(TimelineControl)));
        }

        /// <summary>
        ///
        /// </summary>
        public double HorizontalScale
        {
            get { return (double)GetValue(HorizontalScaleProperty); }
            set { SetValue(HorizontalScaleProperty, value); }
        }

        public static readonly DependencyProperty HorizontalScaleProperty =
            DependencyProperty.Register("HorizontalScale", typeof(double), typeof(TimelineControl),
            new FrameworkPropertyMetadata(1.0));

        /// <summary>
        ///
        /// </summary>
        public double HorizontalSnap
        {
            get { return (double)GetValue(HorizontalSnapProperty); }
            set { SetValue(HorizontalSnapProperty, value); }
        }

        public static readonly DependencyProperty HorizontalSnapProperty =
            DependencyProperty.Register("HorizontalSnap", typeof(double), typeof(TimelineControl),
            new FrameworkPropertyMetadata(1.0));

        /// <summary>
        ///
        /// </summary>
        /// <param name="item"></param>
        /// <returns></returns>
        protected override bool IsItemItsOwnContainerOverride(object item)
        {
            return item is TimelineControlItem;
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            var control = element as TimelineControlItem;
            if (element != item)
            {
                if (control != null)
                {
                    control.Content = item;
                    control.DataContext = item;
                }
            }
        }

        protected override DependencyObject GetContainerForItemOverride()
        {
            return new TimelineControlItem(this);
        }

        internal void HandleMouseDown(TimelineControlItem item, MouseButtonEventArgs e)
        {
            this.dragItem = item;
            this.dragPosition = e.GetPosition(this);
        }

        private Control dragItem = null;
        private Point dragPosition;
        private DragAdorner dragGhost = null;


        internal void HandleMouseMove(object sender, MouseEventArgs e)
        {
            var item = sender as TimelineControlItem;
            if (e.LeftButton == MouseButtonState.Pressed && dragGhost == null && dragItem == item)
            {
                var position = e.GetPosition(this);
                if (Math.Abs(position.X - dragPosition.X) > SystemParameters.MinimumHorizontalDragDistance ||
                    Math.Abs(position.Y - dragPosition.Y) > SystemParameters.MinimumVerticalDragDistance)
                {
                    this.dragItem.AllowDrop = true;
                    this.AllowDrop = true;

                    var layer = AdornerLayer.GetAdornerLayer(this);
                    dragGhost = new DragAdorner(this, item, 0.5, this.dragPosition);
                    layer.Add(dragGhost);

                    DragDrop.DoDragDrop(item, item, DragDropEffects.Move);

                    layer.Remove(dragGhost);
                    this.dragGhost = null;
                    this.dragItem = null;

                    this.AllowDrop = false;
                    e.Handled = true;
                }
            }
        }

        protected override void OnQueryContinueDrag(QueryContinueDragEventArgs e)
        {
            base.OnQueryContinueDrag(e);

            if (dragGhost != null)
            {
                var position = CursorInfo.GetPosition(this);
                // Avalondockでフローティングの時には問題があります
                // Window.GetWindow()で取得できるのがフローティングウインドウではなくメインウインドウになるのが問題
                Point windowLocation = Window.GetWindow(this).PointFromScreen(this.PointToScreen(new Point(0, 0)));
                dragGhost.LeftOffset = position.X - windowLocation.X;
                dragGhost.TopOffset  = position.Y - windowLocation.Y;
            }
        }

        protected override void OnDrop(DragEventArgs e)
        {
            var dropPosition = e.GetPosition(this);

            var x = (dropPosition.X - this.dragPosition.X) / this.HorizontalScale;
            var y = (dropPosition.Y - this.dragPosition.Y);

            var nx = TimelinePanel.GetLeft(this.dragItem);
            var ny = TimelinePanel.GetTop(this.dragItem);

            nx += (double)x;
            this.Snap(ref nx);
            TimelinePanel.SetLeft(this.dragItem, nx);

            base.OnDrop(e);
        }

        private void Snap(ref double x)
        {
            x = Math.Round(x / this.HorizontalSnap) * this.HorizontalSnap;
        }


        /// <summary>
        ///
        /// </summary>
        public static class CursorInfo
        {
            [DllImport("user32.dll")]
            private static extern void GetCursorPos(out POINT pt);

            [DllImport("user32.dll")]
            private static extern int ScreenToClient(IntPtr hwnd, ref POINT pt);

            private struct POINT
            {
                public UInt32 X;
                public UInt32 Y;
            }

            public static Point GetPosition(Visual v)
            {
                POINT point;
                GetCursorPos(out point);

                var source = HwndSource.FromVisual(v) as HwndSource;
                var hwnd = source.Handle;

                ScreenToClient(hwnd, ref point);
                return new Point(point.X, point.Y);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class DragAdorner : Adorner
        {
            private double leftOffset;
            private double topOffset;
            private UIElement element;
            private double xCenter;
            private double yCenter;

            public DragAdorner(UIElement owner)
                : base(owner)
            {
            }

            public DragAdorner(UIElement owner, UIElement adornerElement, double opacity, Point dragPosition)
                : base(owner)
            {
                var adornerLocation = adornerElement.PointToScreen(new Point(0, 0));
                var ownerLocation = owner.PointToScreen(new Point(0, 0));
                this.xCenter = dragPosition.X - (adornerLocation.X - ownerLocation.X);
                this.yCenter = dragPosition.Y - (adornerLocation.Y - ownerLocation.Y);

                var bounds = VisualTreeHelper.GetDescendantBounds(adornerElement);
                this.element = new Rectangle()
                {
                    Width = bounds.Width,
                    Height = bounds.Height,
                    Fill = new VisualBrush(adornerElement) { Opacity = opacity }
                };
            }

            public double LeftOffset
            {
                get { return this.leftOffset; }
                set
                {
                    this.leftOffset = value;
                    UpdatePosition();
                }
            }

            public double TopOffset
            {
                get { return this.topOffset; }
                set
                {
                    this.topOffset = value;
                    UpdatePosition();
                }
            }

            private void UpdatePosition()
            {
                var adorner = this.Parent as AdornerLayer;
                if (adorner != null)
                {
                    adorner.Update(this.AdornedElement);
                }
            }

            protected override Visual GetVisualChild(int index)
            {
                return this.element;
            }

            protected override int VisualChildrenCount
            {
                get { return 1; }
            }

            protected override Size MeasureOverride(Size finalSize)
            {
                this.element.Measure(finalSize);
                return this.element.DesiredSize;
            }

            protected override Size ArrangeOverride(Size finalSize)
            {

                this.element.Arrange(new Rect(this.element.DesiredSize));
                return finalSize;
            }

            public override GeneralTransform GetDesiredTransform(GeneralTransform transform)
            {
                var x = (int)(this.leftOffset - this.xCenter);
                var y = (int)(this.topOffset - this.yCenter);

                var result = new GeneralTransformGroup();
                result.Children.Add(base.GetDesiredTransform(transform));
                result.Children.Add(new TranslateTransform(x, y));
                return result;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TimelineControlItem : ContentControl
    {
        public static readonly DependencyProperty IsSelectedProperty;

        private TimelineControl parent = null;

        public bool IsSelected
        {
            get { return (bool)base.GetValue(ListBoxItem.IsSelectedProperty); }
            set { base.SetValue(ListBoxItem.IsSelectedProperty, value); }
        }

        public TimelineControlItem(TimelineControl parent)
        {
            this.parent = parent;
        }

        protected override void OnPreviewMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            if (e.Handled == false)
            {
                this.IsSelected = !this.IsSelected;

                this.parent.HandleMouseDown(this, e);
            }

            base.OnPreviewMouseLeftButtonDown(e);
        }

        protected override void OnPreviewMouseMove(MouseEventArgs e)
        {
            this.parent.HandleMouseMove(this, e);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TimelinePanel : Panel
    {
        public static readonly DependencyProperty LeftProperty =
            DependencyProperty.RegisterAttached("Left", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(TimelinePanel.OnPositioningChanged)));

        public static readonly DependencyProperty TopProperty =
            DependencyProperty.RegisterAttached("Top", typeof(int), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0, new PropertyChangedCallback(TimelinePanel.OnPositioningChanged)));

        public static double GetLeft(UIElement element)
        {
            if (element == null) { throw new ArgumentNullException("element"); }
            return (double)element.GetValue(TimelinePanel.LeftProperty);
        }

        public static void SetLeft(UIElement element, double value)
        {
            if (element == null) { throw new ArgumentNullException("element"); }
            element.SetValue(TimelinePanel.LeftProperty, value);
        }

        public static int GetTop(UIElement element)
        {
            if (element == null) { throw new ArgumentNullException("element"); }
            return (int)element.GetValue(TimelinePanel.TopProperty);
        }

        public static void SetTop(UIElement element, int length)
        {
            if (element == null) { throw new ArgumentNullException("element"); }
            element.SetValue(TimelinePanel.TopProperty, length);
        }

        private static void OnPositioningChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            UIElement element = d as UIElement;
            if (element != null)
            {
                TimelinePanel panel = VisualTreeHelper.GetParent(element) as TimelinePanel;
                if (panel != null)
                {
                    panel.InvalidateArrange();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public TimelinePanel()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public double HorizontalScale
        {
            get { return (double)GetValue(HorizontalScaleProperty); }
            set { SetValue(HorizontalScaleProperty, value); }
        }

        public static readonly DependencyProperty HorizontalScaleProperty =
            DependencyProperty.Register("HorizontalScale", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(1.0, new PropertyChangedCallback(OnHorizontalScaleChanged)));

        private static void OnHorizontalScaleChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            TimelinePanel panel = d as TimelinePanel;
            panel.InvalidateMeasure();
        }

        /// <summary>
        ///
        /// </summary>
        public double ViewportX
        {
            get { return (double)GetValue(ViewportXProperty); }
            set { SetValue(ViewportXProperty, value); }
        }

        public static readonly DependencyProperty ViewportXProperty =
            DependencyProperty.Register("ViewportX", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(OnViewportXChanged)));

        private static void OnViewportXChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var panel = d as TimelinePanel;
            panel.InvalidateVisual();
        }

        public double ViewportWidth
        {
            get { return (double)GetValue(ViewportWidthProperty); }
            set { SetValue(ViewportWidthProperty, value); }
        }

        public static readonly DependencyProperty ViewportWidthProperty =
            DependencyProperty.Register("ViewportWidth", typeof(double), typeof(TimelinePanel),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(OnViewportWidthChanged)));

        private static void OnViewportWidthChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var panel = d as TimelinePanel;
            panel.InvalidateVisual();
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="availableSize"></param>
        /// <returns></returns>
        protected override Size MeasureOverride(Size availableSize)
        {
            var size = new Size();

            foreach (UIElement item in this.InternalChildren)
            {
                var x = GetLeft(item) * this.HorizontalScale;
                var y = GetTop(item);

                item.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
                if (item.DesiredSize.Width + x > size.Width)
                {
                    size.Width = item.DesiredSize.Width + x;
                }
                if (item.DesiredSize.Height + y > size.Height)
                {
                    size.Height = item.DesiredSize.Height + y;
                }
            }
            return size;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="finalSize"></param>
        /// <returns></returns>
        protected override Size ArrangeOverride(Size finalSize)
        {
            foreach (UIElement item in this.InternalChildren)
            {
                var x = GetLeft(item) * this.HorizontalScale;
                var y = GetTop(item);
                item.Arrange(new Rect(new Point(x, y), item.DesiredSize));
            }
            return finalSize;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="drawingContext"></param>
        protected override void OnRender(DrawingContext drawingContext)
        {
            var span = 10 * this.HorizontalScale;

            var x = (double)((int)(this.ViewportX / span) * span);
            var xcount = (int)(this.ViewportWidth / span) + 2;

            var pen = new Pen(Brushes.Gray, 1);
            pen.DashStyle = new DashStyle();
            pen.DashStyle.Dashes.Add(3.0);


            for (int index = 0; index < xcount; index++)
            {
                var guideline = new GuidelineSet();
                guideline.GuidelinesX.Add(x - 0.5);
                guideline.GuidelinesX.Add(x + 0.5);

                drawingContext.PushGuidelineSet(guideline);
                drawingContext.DrawGeometry(null, pen, new LineGeometry(new Point(x, 0), new Point(x, this.ActualHeight)));
                drawingContext.Pop();

                x += span;
            }

            base.OnRender(drawingContext);
        }
    }
}
