﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Media;

namespace Nintendo.AudioToolkit.Behaviors
{
    public class ScrollViewerBehavior
    {
        public static readonly DependencyProperty HorizontalOffsetProperty =
            DependencyProperty.RegisterAttached("HorizontalOffset",
            typeof(double),
            typeof(ScrollViewerBehavior),
            new PropertyMetadata(0.0, OnHorizontalOffsetPropertyChanged));

        public static readonly DependencyProperty VerticalOffsetProperty =
            DependencyProperty.RegisterAttached("VerticalOffset",
            typeof(double),
            typeof(ScrollViewerBehavior),
            new PropertyMetadata(0.0, OnVerticalOffsetPropertyChanged));

        private static readonly DependencyProperty HorizontalScrollBarProperty =
            DependencyProperty.RegisterAttached("HorizontalScrollBar",
            typeof(ScrollBar),
            typeof(ScrollViewerBehavior),
            new UIPropertyMetadata(null));

        private static readonly DependencyProperty VerticalScrollBarProperty =
            DependencyProperty.RegisterAttached("VerticalScrollBar",
            typeof(ScrollBar),
            typeof(ScrollViewerBehavior),
            new PropertyMetadata(null));

        public static double GetHorizontalOffset(DependencyObject d)
        {
            return (double)d.GetValue(HorizontalOffsetProperty);
        }

        public static void SetHorizontalOffset(DependencyObject d, double value)
        {
            d.SetValue(HorizontalOffsetProperty, value);
        }

        public static double GetVerticalOffset(DependencyObject d)
        {
            return (double)d.GetValue(VerticalOffsetProperty);
        }

        public static void SetVerticalOffset(DependencyObject d, double value)
        {
            d.SetValue(VerticalOffsetProperty, value);
        }

        private static void OnHorizontalOffsetPropertyChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            ScrollViewer scrollViewer = d as ScrollViewer;
            if (scrollViewer != null)
            {
                if (scrollViewer.GetValue(HorizontalScrollBarProperty) == null)
                {
                    scrollViewer.LayoutUpdated += (s, ev) =>
                        {
                            if (scrollViewer.GetValue(HorizontalScrollBarProperty) == null)
                            {
                                AttachHorizontalScrollBar(scrollViewer);
                                AttachVerticalScrollBar(scrollViewer);
                            }
                        };
                }
                else
                {
                    scrollViewer.ScrollToHorizontalOffset((double)e.NewValue);
                }
            }
        }

        private static void OnVerticalOffsetPropertyChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            ScrollViewer scrollViewer = d as ScrollViewer;
            if (scrollViewer != null)
            {
                if (scrollViewer.GetValue(VerticalScrollBarProperty) == null)
                {
                    scrollViewer.LayoutUpdated += (s, ev) =>
                        {
                            if (scrollViewer.GetValue(VerticalScrollBarProperty) == null)
                            {
                                AttachHorizontalScrollBar(scrollViewer);
                                AttachVerticalScrollBar(scrollViewer);
                            }
                        };
                }
                else
                {
                    scrollViewer.ScrollToVerticalOffset((double)e.NewValue);
                }
            }
        }

        private static void AttachHorizontalScrollBar(ScrollViewer scrollViewer)
        {
            var scrollBar = GetScrollBar(scrollViewer, Orientation.Horizontal);
            if (scrollBar != null)
            {
                scrollViewer.SetValue(HorizontalScrollBarProperty, scrollBar);
                scrollViewer.ScrollToHorizontalOffset(ScrollViewerBehavior.GetHorizontalOffset(scrollViewer));
                scrollBar.ValueChanged += (s, e) => scrollViewer.SetValue(HorizontalOffsetProperty, e.NewValue);
            }
        }

        private static void AttachVerticalScrollBar(ScrollViewer scrollViewer)
        {
            var scrollBar = GetScrollBar(scrollViewer, Orientation.Vertical);
            if (scrollBar != null)
            {
                scrollViewer.SetValue(VerticalScrollBarProperty, scrollBar);
                scrollViewer.ScrollToVerticalOffset(GetVerticalOffset(scrollViewer));
                scrollBar.ValueChanged += (s, e) => SetVerticalOffset(scrollViewer, e.NewValue);
            }
        }

        private static ScrollBar GetScrollBar(FrameworkElement element, Orientation orientation)
        {
            return Descendants(element)
                .OfType<ScrollBar>()
                .Where(s => s.Orientation == orientation)
                .SingleOrDefault();
        }

        private static IEnumerable<DependencyObject> Descendants(DependencyObject d)
        {
            foreach (var index in Enumerable.Range(0, VisualTreeHelper.GetChildrenCount(d)))
            {
                var child = VisualTreeHelper.GetChild(d, index);
                yield return child;

                foreach (var grandson in Descendants(child))
                {
                    yield return grandson;
                }
            }
        }
    }
}
