﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <panels/SoundParameterPanel.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------

static u32 GetStartOffset(PreviewSound* pPreviewSound)
{
    NW_ASSERT(pPreviewSound != NULL);
    return pPreviewSound->GetStartOffset();
}

static f32 GetPitch(PreviewSound* pPreviewSound)
{
    NW_ASSERT(pPreviewSound != NULL);
    return pPreviewSound->GetPitch();
}

static f32 GetVolume(PreviewSound* pPreviewSound)
{
    NW_ASSERT(pPreviewSound != NULL);
    return pPreviewSound->GetVolume();
}

static f32 GetPan(PreviewSound* pPreviewSound)
{
    NW_ASSERT(pPreviewSound != NULL);
    return pPreviewSound->GetPan();
}

static f32 GetSurroundPan(PreviewSound* pPreviewSound)
{
    NW_ASSERT(pPreviewSound != NULL);
    return pPreviewSound->GetSurroundPan();
}

static void SetStartOffset(PreviewSound* pPreviewSound, u32 value)
{
    NW_ASSERT(pPreviewSound != NULL);
    pPreviewSound->SetStartOffset(value);
}

static void SetPitch(PreviewSound* pPreviewSound, f32 value)
{
    NW_ASSERT(pPreviewSound != NULL);
    pPreviewSound->SetPitch(value);
}

static void SetVolume(PreviewSound* pPreviewSound, f32 value)
{
    NW_ASSERT(pPreviewSound != NULL);
    pPreviewSound->SetVolume(value);
}

static void SetPan(PreviewSound* pPreviewSound, f32 value)
{
    NW_ASSERT(pPreviewSound != NULL);
    pPreviewSound->SetPan(value);
}

static void SetSurroundPan(PreviewSound* pPreviewSound, f32 value)
{
    NW_ASSERT(pPreviewSound != NULL);
    pPreviewSound->SetSurroundPan(value);
}

static void PrintU32Value(char* pText, u32 bufferLength, u32 value)
{
    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%d", value);
}

static void PrintF32Value(char* pText, u32 bufferLength, f32 value)
{
    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%4.2f", value);
}

//---------------------------------------------------------------------------

SoundParameterPanel::SoundParameterPanel() :
m_pPreviewSoundSelector(NULL),
m_SoundHandleSelectionChangedObserver(this, &SoundParameterPanel::OnSoundHandleSelectionChanged)
{
    SetMargin(nw::internal::dw::Thickness(0.f));

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_VolumeControl.Initialize(GetVolume, SetVolume, PrintF32Value);
    m_VolumeControl.SetLabelText("Volume:");
    m_VolumeControl.SetMinimumValue(0.f);
    m_VolumeControl.SetMaximumValue(4.f);
    m_VolumeControl.SetDefaultValue(1.f);
    m_VolumeControl.SetValueChange(0.01f);
    m_VolumeControl.SetValueLargeChange(0.1f);

    m_PitchControl.Initialize(GetPitch, SetPitch, PrintF32Value);
    m_PitchControl.SetLabelText("Pitch:");
    m_PitchControl.SetMinimumValue(0.01f);
    m_PitchControl.SetMaximumValue(4.f);
    m_PitchControl.SetDefaultValue(1.f);
    m_PitchControl.SetValueChange(0.01f);
    m_PitchControl.SetValueLargeChange(0.1f);

    m_StartOffsetControl.Initialize(GetStartOffset, SetStartOffset, PrintU32Value);
    m_StartOffsetControl.SetLabelText("StartOffset:");
    m_StartOffsetControl.SetMinimumValue(0);
    m_StartOffsetControl.SetMaximumValue(10 * 60 * 1000);
    m_StartOffsetControl.SetDefaultValue(0);
    m_StartOffsetControl.SetValueChange(1);
    m_StartOffsetControl.SetValueLargeChange(1000);

    m_PanControl.Initialize(GetPan, SetPan, PrintF32Value);
    m_PanControl.SetLabelText("Pan:");
    m_PanControl.SetMinimumValue(-2.f);
    m_PanControl.SetMaximumValue(2.f);
    m_PanControl.SetDefaultValue(0.f);
    m_PanControl.SetValueChange(0.01f);
    m_PanControl.SetValueLargeChange(0.1f);

    m_SurroundPanControl.Initialize(GetSurroundPan, SetSurroundPan, PrintF32Value);
    m_SurroundPanControl.SetLabelText("SPan:");
    m_SurroundPanControl.SetMinimumValue(-2.f);
    m_SurroundPanControl.SetMaximumValue(2.f);
    m_SurroundPanControl.SetDefaultValue(0.f);
    m_SurroundPanControl.SetValueChange(0.01f);
    m_SurroundPanControl.SetValueLargeChange(0.1f);

    AddItem(m_VolumeControl);
    AddItem(m_PitchControl);
    AddItem(m_StartOffsetControl);
    AddItem(m_PanControl);
    AddItem(m_SurroundPanControl);
}

void SoundParameterPanel::Initialize(IPreviewSoundSelector* pPreviewSoundSelector)
{
    NW_ASSERTMSG(m_pPreviewSoundSelector == NULL, "SoundParameterPanel is already initialized.");
    NW_ASSERT(pPreviewSoundSelector != NULL);

    m_pPreviewSoundSelector = pPreviewSoundSelector;
    pPreviewSoundSelector->GetSelectionChangedEvent() += m_SoundHandleSelectionChangedObserver;

    // 最初の項目を選択します。
    if(m_Container.GetItemCount() > 0)
    {
        m_Container.GetItemPlaceHolder(0).SetIsSelected(true);
    }

    UpdatePreviewSounds();

    for(s32 i=0; i<m_Container.GetItemCount(); ++i)
    {
        m_pListItems[i]->ResetValue();
    }
}

void SoundParameterPanel::AddItem(ModelParameterListItem<PreviewSound>& item)
{
    u32 index = m_Container.GetItemCount();
    NW_ASSERT(index < MAX_ITEM_COUNT);

    m_pListItems[index] = &item;

    item.SetLabelWidth(120.f);
    item.SetCheckVisibility(nw::internal::dw::VISIBLE);
    m_Container.AddItem(&item);
}

void SoundParameterPanel::UpdatePreviewSounds()
{
    NW_ASSERTMSG(m_pPreviewSoundSelector != NULL, "SoundParameterPanel is not initialized.");

    nw::snd::PreviewSound* pPreviewSound = m_pPreviewSoundSelector->GetSelectedPreviewSoundFirst();

    for(s32 i=0; i<m_Container.GetItemCount(); ++i)
    {
        m_pListItems[i]->SetModel(pPreviewSound);
    }
}

void SoundParameterPanel::OnSoundHandleSelectionChanged()
{
    UpdatePreviewSounds();
}

} // snd
} // nw
