﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <panels/PreviewBankPanel.h>
#include <models/PreviewSoundArchive.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace {
    const s32 MIDI_THREAD_PRIORITY = 2;
}

namespace nw {
namespace snd {

static void PrintF32Value(char* pText, u32 bufferLength, f32 value)
{
    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%4.2f", value);
}

static f32 GetVolume(util::MidiManager* pMidiManager)
{
    NW_ASSERT(pMidiManager != NULL);
    return pMidiManager->GetVolume();
}

static void SetVolume(util::MidiManager* pMidiManager, f32 value)
{
    NW_ASSERT(pMidiManager != NULL);
    pMidiManager->SetVolume(value);
}

PreviewBankPanel::PreviewBankPanel()
: m_pPreviewSoundArchive(NULL)
, m_pLastSoundArchive(NULL)
, m_MidiThreadPriority(MIDI_THREAD_PRIORITY)
, m_pMemoryForMidiManager(NULL)
, m_OutputLines(nw::snd::OUTPUT_LINE_MAIN)
{
    SetMargin(nw::internal::dw::Thickness(0.f));

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_SeqLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_SeqLabel.SetIsFocusable(false);
    m_SeqLabel.SetText("[SEQ]");
    m_List.AddItem(&m_SeqLabel, false);

    m_List.AddItem(&m_PreviewSequence);

    m_DataVolumeLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_DataVolumeLabel.SetIsFocusable(false);
    m_List.AddItem(&m_DataVolumeLabel);

    m_BankLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_BankLabel.SetIsFocusable(false);
    m_BankLabel.SetText("[BANK]");
    m_List.AddItem(&m_BankLabel, false);

    for ( s32 i = 0; i < PreviewSoundArchive::PREVIEW_BANK_COUNT; ++i )
    {
        m_List.AddItem(&m_PreviewBank[i]);
    }

    m_EmptyLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));
    m_EmptyLabel.SetIsFocusable(false);
    m_EmptyLabel.SetText("");
    m_List.AddItem(&m_EmptyLabel, false);

    m_Volume.Initialize(GetVolume, SetVolume, PrintF32Value);
    m_Volume.SetLabelText("Volume:");
    m_Volume.SetMinimumValue(0.f);
    m_Volume.SetMaximumValue(2.f);
    m_Volume.SetDefaultValue(1.f);
    m_Volume.SetValueChange(0.01f);
    m_Volume.SetValueLargeChange(0.1f);
    m_Volume.SetCheckVisibility(nw::internal::dw::VISIBLE);
    m_List.AddItem(&m_Volume);

    m_LampBar.SetOrientation(nw::internal::dw::HORIZONTAL);
    m_LampBar.SetDock(nw::internal::dw::DOCK_BOTTOM);
    m_LampBar.SetMargin(nw::internal::dw::Thickness(0.f));
    m_LampBar.SetBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f)));

    m_LampBar.GetContents().AddItem(&m_MainLamp);
    m_LampBar.GetContents().AddItem(&m_DrcLamp);
    m_LampBar.GetContents().AddItem(&m_MainAndDrcLamp);

    m_MainLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_MainLamp.SetLabelText("Main");
    m_MainLamp.SetIsLamped(true);

    m_DrcLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_DrcLamp.SetLabelText("DRC");

    m_MainAndDrcLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_MainAndDrcLamp.SetLabelText("Main & DRC");

    m_Container.GetContents().AddItem(&m_LampBar);
    m_Container.GetContents().AddItem(&m_List);
}

void PreviewBankPanel::Initialize(nw::snd::PreviewSoundArchive& previewSoundArchive)
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif

    size_t setupSize =  m_MidiManager.GetRequiredMemSize();
    m_pMemoryForMidiManager = MEMAllocFromDefaultHeap(setupSize);
    bool result = m_MidiManager.Initialize( m_MidiThreadPriority, m_pMemoryForMidiManager, setupSize );
    NW_LOG( "MidiManager::Initialize(%d)\n", result );

    UnloadBanks();

    for ( s32 i = 0; i < previewSoundArchive.GetPreviewBankCount(); ++i )
    {
        m_PreviewBank[i].Initialize(previewSoundArchive.GetPreviewBank(i));
    }
    m_PreviewSequence.Initialize(previewSoundArchive);
    m_Volume.SetModel(&m_MidiManager);

    // 最初のサウンドハンドルを選択します。
    if(previewSoundArchive.GetPreviewBankCount() > 0)
    {
        m_List.GetItemPlaceHolder(1).SetIsSelected(true);
    }

    m_pPreviewSoundArchive = &previewSoundArchive;
}

void PreviewBankPanel::Finalize()
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif
    m_PreviewSequence.Finalize();

    m_MidiManager.Finalize();
    //nn::midi::Close();

    MEMFreeToDefaultHeap( m_pMemoryForMidiManager );
}

bool PreviewBankPanel::OnPreviewUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsHold(nw::dev::Pad::MASK_L))
    {
        if(pad.IsTrig(nw::dev::Pad::MASK_LEFT))
        {
            m_MainLamp.SetIsLamped(false);
            m_DrcLamp.SetIsLamped(false);
            m_MainAndDrcLamp.SetIsLamped(false);

            switch(m_OutputLines)
            {
            case nw::snd::OUTPUT_LINE_MAIN:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1;
                m_MainAndDrcLamp.SetIsLamped(true);
                break;

            case nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN;
                m_MainLamp.SetIsLamped(true);
                break;

            case nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1:
                m_OutputLines = nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1;
                m_DrcLamp.SetIsLamped(true);
                break;

            default:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN;
                m_MainLamp.SetIsLamped(true);
                break;
            }

            m_MidiManager.SetOutputLine(m_OutputLines);
            return true;
        }

        if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
        {
            m_MainLamp.SetIsLamped(false);
            m_DrcLamp.SetIsLamped(false);
            m_MainAndDrcLamp.SetIsLamped(false);

            switch(m_OutputLines)
            {
            case nw::snd::OUTPUT_LINE_MAIN:
                m_OutputLines = nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1;
                m_DrcLamp.SetIsLamped(true);
                break;

            case nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1;
                m_MainAndDrcLamp.SetIsLamped(true);
                break;

            case nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 | nw::snd::OUTPUT_LINE_DRC1:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN;
                m_MainLamp.SetIsLamped(true);
                break;

            default:
                m_OutputLines = nw::snd::OUTPUT_LINE_MAIN;
                m_MainLamp.SetIsLamped(true);
                break;
            }

            m_MidiManager.SetOutputLine(m_OutputLines);
            return true;
        }
    }

    return false;
}

bool PreviewBankPanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_A))
    {
        LoadBanks();
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_B))
    {
        UnloadBanks();
        return true;
    }

    return false;
}

void PreviewBankPanel::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    CheckSoundArchive();

    UpdateBankState();
    UpdateDataVolume();
}

void PreviewBankPanel::CheckSoundArchive()
{

    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    SoundArchive* pSoundArchive = &m_pPreviewSoundArchive->GetSoundArchive();

    if (!m_pPreviewSoundArchive->IsOpened())
    {
        pSoundArchive = NULL;
        m_pLastSoundArchive = NULL;

        return;
    }

    if (pSoundArchive != m_pLastSoundArchive)
    {
        m_PreviewSequence.UpdateInfo();

        UnloadBanks();
        LoadBanks();
    }

    m_pLastSoundArchive = pSoundArchive;
}

void PreviewBankPanel::UpdateBankState()
{
    if ( m_pPreviewSoundArchive == NULL ) { return; }

    for (s32 i = 0; i < m_pPreviewSoundArchive->GetPreviewBankCount(); ++i)
    {
        bool isLoaded = IsLoaded( m_pPreviewSoundArchive->GetPreviewBank(i) );
        m_pPreviewSoundArchive->GetPreviewBank(i).SetIsLoaded( isLoaded );
    }
}

void PreviewBankPanel::UpdateDataVolume()
{
    int volume = 0;
    f32 volumeF32 = 0.0f;
    {
        nw::snd::SoundArchive::SoundInfo info;
        if ( m_PreviewSequence.ReadSelectedSoundInfo( &info ) )
        {
            volume = info.volume;
            volumeF32 = static_cast<f32>( volume ) / 127.0f;
        }
    }

    static const u32 DATA_VOLUME_LABEL_LENGTH = 32;
    char label[DATA_VOLUME_LABEL_LENGTH];
    nw::ut::snprintf(label, DATA_VOLUME_LABEL_LENGTH, DATA_VOLUME_LABEL_LENGTH - 1, "     data vol ( %3d/%1.2lf )", volume, volumeF32);

    m_DataVolumeLabel.SetText(label);
}

bool PreviewBankPanel::IsLoaded(const PreviewBank& previewBank) const
{
    for (s32 i = 0; i < PreviewSoundArchive::PREVIEW_BANK_COUNT; ++i )
    {
        if (m_LoadedBankIds[i] == nw::snd::SoundArchive::INVALID_ID)
        {
            continue;
        }

        if (previewBank.GetBankID() == m_LoadedBankIds[i])
        {
            return true;
        }
    }

    return false;
}

void PreviewBankPanel::LoadBanks()
{
    if ( m_pPreviewSoundArchive == NULL ) { return; }

    m_pPreviewSoundArchive->GetSoundHeap().LoadState( 0 );

    for (s32 i = 0; i < m_pPreviewSoundArchive->GetPreviewBankCount(); ++i)
    {
        m_LoadedBankIds[i] = m_pPreviewSoundArchive->GetPreviewBank(i).GetBankID();

        if(m_LoadedBankIds[i] != nw::snd::SoundArchive::INVALID_ID)
        {
            m_pPreviewSoundArchive->LoadData(m_LoadedBankIds[i]);
        }
    }

    m_MidiManager.Prepare(
        m_LoadedBankIds,
        m_pPreviewSoundArchive->GetSoundArchive(),
        m_pPreviewSoundArchive->GetSoundDataManager()
    );

    m_pPreviewSoundArchive->GetSoundHeap().SaveState();
}

void PreviewBankPanel::UnloadBanks()
{
    if ( m_pPreviewSoundArchive == NULL ) { return; }

    m_pPreviewSoundArchive->GetSoundHeap().LoadState( 0 );

    for (s32 i = 0; i < m_pPreviewSoundArchive->GetPreviewBankCount(); ++i)
    {
        m_LoadedBankIds[i] = nw::snd::SoundArchive::INVALID_ID;
    }

    m_MidiManager.Prepare(
        m_LoadedBankIds,
        m_pPreviewSoundArchive->GetSoundArchive(),
        m_pPreviewSoundArchive->GetSoundDataManager()
    );
}

} // snd
} // nw
