﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewEffectManager.h>

namespace nw {
namespace snd {

PreviewEffectManager::PreviewEffectManager() :
m_AssinedSize(0)
{
}

void PreviewEffectManager::Initialize(void* pBuffer, u32 bufferSize, OutputDevice device)
{
    Finalize();

    u32 bufferSizePerAuxBus = bufferSize / AUX_BUS_COUNT;

    UpdateChannelMode(device);

    for(s32 i = 0; i < AUX_BUS_COUNT; ++i)
    {
        PreviewAuxBus::InitializeArgs args;
        args.auxBus = IndexToAuxBus(i);
        args.device = device;
        args.pEffects = m_AuxBusEffects[i];
        args.effectCount = EFFECT_COUNT;
        args.pBuffer = nw::ut::AddOffsetToPtr(pBuffer, bufferSizePerAuxBus * i);
        args.bufferSize = bufferSizePerAuxBus;

        m_AuxBuses[i].Initialize(args);
    }
}

void PreviewEffectManager::Finalize()
{
    for(s32 i = 0; i < AUX_BUS_COUNT; ++i)
    {
        m_AuxBuses[i].Finalize();
    }

    m_AssinedSize = 0;
}

PreviewAuxBus* PreviewEffectManager::GetAuxBus(s32 auxBusIndex)
{
    if(auxBusIndex >= AUX_BUS_COUNT)
    {
        NW_ASSERTMSG(false, "[PreviewEffectManager] invalid auxBusIndex.\n");
        return NULL;
    }

    return &m_AuxBuses[auxBusIndex];
}

PreviewAuxBus* PreviewEffectManager::GetAuxBus(AuxBus auxBus)
{
    return GetAuxBus(AuxBusToIndex(auxBus));
}

void PreviewEffectManager::Update()
{
    if(GetIsDirty())
    {
        Apply();
    }
}

void PreviewEffectManager::Apply()
{
    Clear();

    for(s32 i = 0; i < AUX_BUS_COUNT; ++i)
    {
        m_AuxBuses[i].Apply();

        m_AssinedSize += m_AuxBuses[i].GetAssinedSize();
    }

    ResetIsDirty();

    m_AppliedEvent.Invoke();
}

void PreviewEffectManager::Clear()
{
    for(s32 i = 0; i < AUX_BUS_COUNT; ++i)
    {
        m_AuxBuses[i].Clear();
    }

    m_AssinedSize = 0;
}

void PreviewEffectManager::OnBeginSoundFrame()
{
    // TODO : ★エフェクトのCPU使用率を取得する。
#if 0
    int auxProcess = 0;

    auxProcess += static_cast<int>(
        snd::detail::AxManager::GetInstance().GetEffectProcessTick(snd::AUX_A));
    auxProcess += static_cast<int>(
        snd::detail::AxManager::GetInstance().GetEffectProcessTick(snd::AUX_B));
    auxProcess += static_cast<int>(
        snd::detail::AxManager::GetInstance().GetEffectProcessTick(snd::AUX_C));

    if ( mAuxProcess < auxProcess ) mAuxProcess = auxProcess;
#endif
}

bool PreviewEffectManager::GetIsDirty() const
{
    for(s32 auxBusIndex = 0; auxBusIndex < AUX_BUS_COUNT; ++auxBusIndex)
    {
        for(s32 effectIndex = 0; effectIndex < EFFECT_COUNT; ++effectIndex)
        {
            if(m_AuxBusEffects[auxBusIndex][effectIndex].GetIsDirty())
            {
                return true;
            }
        }
    }

    return false;
}

void PreviewEffectManager::ResetIsDirty()
{
    for(s32 auxBusIndex = 0; auxBusIndex < AUX_BUS_COUNT; ++auxBusIndex)
    {
        for(s32 effectIndex = 0; effectIndex < EFFECT_COUNT; ++effectIndex)
        {
            m_AuxBusEffects[auxBusIndex][effectIndex].ResetIsDirty();
        }
    }
}

AuxBus PreviewEffectManager::IndexToAuxBus(s32 value)
{
    switch(value)
    {
    case 0:
        return AUX_BUS_A;

    case 1:
        return AUX_BUS_B;

    case 2:
        return AUX_BUS_C;
    }

    NW_FATAL_ERROR("[PreviewEffectManager] invalid index.\n");
    return AUX_BUS_A;
}

s32 PreviewEffectManager::AuxBusToIndex(AuxBus value)
{
    switch(value)
    {
    case AUX_BUS_A:
        return 0;

    case AUX_BUS_B:
        return 1;

    case AUX_BUS_C:
        return 2;
    }

    NW_FATAL_ERROR("[PreviewEffectManager] invalid AuxBus.\n");
    return 0;
}

void PreviewEffectManager::UpdateChannelMode(OutputDevice device)
{
    if (device == OUTPUT_DEVICE_DRC0 || device == OUTPUT_DEVICE_DRC1)
    {
        for(s32 auxBusIndex = 0; auxBusIndex < AUX_BUS_COUNT; ++auxBusIndex)
        {
            for(s32 effectIndex = 0; effectIndex < EFFECT_COUNT; ++effectIndex)
            {
                m_AuxBusEffects[auxBusIndex][effectIndex].SetChannelMode(FxBase::CHANNEL_MODE_4CH);
            }
        }
    }
}

} // nw::snd
} // nw
