﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <controls/PeakMeter.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace snd {

const f32 PeakMeter::PEAKHOLD_SIZE = 2.f;

PeakMeter::PeakMeter() :
m_Orientation(nw::internal::dw::VERTICAL),
m_Value(0.f),
m_PeakValue(0.f),
m_MinimumValue(0.f),
m_MaximumValue(1.f),
m_PeakHoldChange(0.01f),
m_PeakHoldFrames(60.f),
m_RemainPeakHoldFrames(0.f),
m_BarColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.40f, 0.6f, 0.2f, 1.f))),
m_OverflowBarColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.7f, 0.2f, 0.2f, 1.f)))
{
    SetPadding(0.f);
}

nw::internal::dw::Orientation PeakMeter::GetOrientation() const
{
    return m_Orientation;
}

void PeakMeter::SetOrientation(nw::internal::dw::Orientation value)
{
    m_Orientation = value;
}

float PeakMeter::GetValue() const
{
    return m_Value;
}

void PeakMeter::SetValue(float value)
{
    m_Value = value;

    if( m_PeakValue < value )
    {
        m_PeakValue = value;
        m_RemainPeakHoldFrames = m_PeakHoldFrames;
    }
}

float PeakMeter::GetPeakValue() const
{
    return m_PeakValue;
}

bool PeakMeter::IsPeakHoldEnabled() const
{
    return !(m_PeakHoldFrames < 0.f || nw::ut::FloatEquals(m_PeakHoldFrames, 0.f));
}

bool PeakMeter::IsValueInRange() const
{
    return m_MinimumValue <= m_Value && m_Value <= m_MaximumValue;
}

bool PeakMeter::IsPeakValueInRange() const
{
    return m_MinimumValue <= m_PeakValue && m_PeakValue <= m_MaximumValue;
}

float PeakMeter::GetMinimumValue() const
{
    return m_MinimumValue;
}

void PeakMeter::SetMinimumValue(float value)
{
    m_MinimumValue = value;
}

float PeakMeter::GetMaximumValue() const
{
    return m_MaximumValue;
}

void PeakMeter::SetMaximumValue(float value)
{
    m_MaximumValue = value;
}

float PeakMeter::GetPeakHoldChange() const
{
    return m_PeakHoldChange;
}

void PeakMeter::SetPeakHoldChange(float value)
{
    m_PeakHoldChange = value;
}

float PeakMeter::GetPeakHoldFrames() const
{
    return m_PeakHoldFrames;
}

void PeakMeter::SetPeakHoldFrames(float value)
{
    m_PeakHoldFrames = value;
}

const nw::ut::Color4f PeakMeter::GetBarColor() const
{
    return m_BarColor;
}

void PeakMeter::SetBarColor(const nw::ut::Color4f& value)
{
    m_BarColor = value;
}

const nw::ut::Color4f PeakMeter::GetOverflowBarColor() const
{
    return m_OverflowBarColor;
}

void PeakMeter::SetOverflowBarColor(const nw::ut::Color4f& value)
{
    m_OverflowBarColor = value;
}

const nw::math::Vector2 PeakMeter::OnMeasure(nw::internal::dw::UIRenderer& renderer) const
{
    nw::math::Vector2 size = Base::OnMeasure(renderer);

    if(size.IsZero())
    {
        switch(m_Orientation)
        {
        case nw::internal::dw::HORIZONTAL:
            return nw::math::Vector2(8.f, 2.f);

        case nw::internal::dw::VERTICAL:
            return nw::math::Vector2(2.f, 8.f);

        default:
            NW_ASSERTMSG(false, "Invalid Orientation.");
            return nw::math::Vector2(0.f, 0.f);
        }
    }

    return size;
}

void PeakMeter::OnUpdate(const nw::internal::dw::UIElementTreeContext& context)
{
    if(!IsPeakHoldEnabled())
    {
        return;
    }

    if(m_RemainPeakHoldFrames < 0.f || nw::ut::FloatEquals(m_RemainPeakHoldFrames, 0.f))
    {
        m_PeakValue = nw::ut::Max(m_MinimumValue, GetPeakValueInRange() - m_PeakHoldChange);
    }
    else
    {
        m_RemainPeakHoldFrames -= 1.f;
    }
}

void PeakMeter::OnRender(const nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer, nw::internal::dw::UIElementRenderArgs& args) const
{
    switch(m_Orientation)
    {
    case nw::internal::dw::HORIZONTAL:
        RenderByHorizontal(context, renderer, args);
        break;

    case nw::internal::dw::VERTICAL:
        RenderByVertical(context, renderer, args);
        break;

    default:
        NW_ASSERTMSG(false, "Invalid Orientation.");
        break;
    }
}

float PeakMeter::GetValueInRange() const
{
    return nw::ut::Max(nw::ut::Min(m_Value, m_MaximumValue), m_MinimumValue);
}

float PeakMeter::GetPeakValueInRange() const
{
    return nw::ut::Max(nw::ut::Min(m_PeakValue, m_MaximumValue), m_MinimumValue);
}

void PeakMeter::RenderByHorizontal(const nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer, nw::internal::dw::UIElementRenderArgs& args) const
{
    nw::internal::dw::Thickness padding = GetPadding();
    nw::math::Vector2 size = GetContentAreaSize();

    float valuePosition = 0.f;
    float peakPosition = 0.f;
    float range = m_MaximumValue - m_MinimumValue;
    float value = GetValueInRange();
    float peakValue = GetPeakValueInRange();

    if(range > 0.f && !nw::ut::FloatEquals(range, 0.f))
    {
        valuePosition = size.x * value / range;
        peakPosition = size.x * peakValue / range;
    }

    if(!args.backgroundColor0.IsTransparent() ||
        !args.backgroundColor1.IsTransparent())
    {
        renderer.FillRectangle(
            &context,
            nw::internal::dw::DrawRectangleArgs().
            SetTopLeft(padding.left + valuePosition, padding.top).
            SetSize(size.x - valuePosition, size.y).
            SetColor(args.backgroundColor0, args.backgroundColor1));
    }

    nw::ut::Color4f barColor = IsValueInRange() ? m_BarColor : m_OverflowBarColor;

    if(!barColor.IsTransparent())
    {
        renderer.FillRectangle(
            &context,
            nw::internal::dw::DrawRectangleArgs().
            SetTopLeft(padding.left, padding.right).
            SetSize(valuePosition, size.y).
            SetColor(barColor));
    }

    // PeakHold
    if(IsPeakHoldEnabled() &&
        value != peakValue &&
        peakValue > 0.f && !nw::ut::FloatEquals(peakValue, 0.f) &&
        peakPosition > PEAKHOLD_SIZE)
    {
        nw::ut::Color4f peakHoldColor = IsPeakValueInRange() ? m_BarColor : m_OverflowBarColor;

        if(!peakHoldColor.IsTransparent())
        {
           renderer.FillRectangle(
                &context,
                nw::internal::dw::DrawRectangleArgs().
                SetTopLeft(padding.left + peakPosition - PEAKHOLD_SIZE, padding.top).
                SetSize(PEAKHOLD_SIZE, size.y).
                SetColor(peakHoldColor));
        }
    }
}

void PeakMeter::RenderByVertical(const nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer, nw::internal::dw::UIElementRenderArgs& args) const
{
    nw::internal::dw::Thickness padding = GetPadding();
    nw::math::Vector2 size = GetContentAreaSize();

    float valuePosition = 0.f;
    float peakPosition = 0.f;
    float range = m_MaximumValue - m_MinimumValue;
    float value = GetValueInRange();
    float peakValue = GetPeakValueInRange();

    if(range > 0.f && !nw::ut::FloatEquals(range, 0.f))
    {
        valuePosition = size.y * value / range;
        peakPosition = size.y * peakValue / range;
    }

    if(!args.backgroundColor0.IsTransparent() ||
        !args.backgroundColor1.IsTransparent())
    {
        renderer.FillRectangle(
            &context,
            nw::internal::dw::DrawRectangleArgs().
            SetTopLeft(padding.left, padding.right).
            SetSize(size.x, size.y - valuePosition).
            SetColor(args.backgroundColor0, args.backgroundColor1));
    }

    nw::ut::Color4f barColor = IsValueInRange() ? m_BarColor : m_OverflowBarColor;

    if(!barColor.IsTransparent())
    {
        renderer.FillRectangle(
            &context,
            nw::internal::dw::DrawRectangleArgs().
            SetTopLeft(padding.left, padding.top + size.y - valuePosition).
            SetSize(size.x, valuePosition).
            SetColor(barColor));
    }

    // PeakHold
    if(IsPeakHoldEnabled() &&
        !nw::ut::FloatEquals(peakPosition, 0.f) &&
        value != peakValue &&
        peakValue > 0.f && !nw::ut::FloatEquals(peakValue, 0.f))
    {
        nw::ut::Color4f peakHoldColor = IsPeakValueInRange() ? m_BarColor : m_OverflowBarColor;

        if(!peakHoldColor.IsTransparent())
        {
            renderer.FillRectangle(
                &context,
                nw::internal::dw::DrawRectangleArgs().
                SetTopLeft(padding.left, padding.top + size.y - peakPosition).
                SetSize(size.x, PEAKHOLD_SIZE).
                SetColor(peakHoldColor));
        }
    }
}

} // snd
} // nw
