﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <anim/GridDrawer.h>
#include <nw/gfnd/gfnd_Graphics.h>

namespace nw    {
namespace snd   {

//---------------------------------------------------------------------------
/* ctor */
GridDrawer::GridDrawer()
 : m_GridInterval( 10.0f )
 , m_GridCount( 10 )
 , m_GridColor( 0xFF, 0xFF, 0xFF, 0xFF )
 , m_GridLineWidth( 1.f )
 , m_ProjMtx( NULL )
 , m_ViewMtx( NULL )
 , m_Allocator( NULL )
 , m_DrawStatus( DRAW )
{
    std::memset( &m_GridBuffer, 0, sizeof( nw::dev::DrawBuffer ) );
}

//---------------------------------------------------------------------------
void
GridDrawer::Draw() const
{
    if ( m_DrawStatus == DRAW )
    {
        NW_ASSERT_NOT_NULL( m_ProjMtx );
        NW_ASSERT_NOT_NULL( m_ViewMtx );

        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();

        renderer->SetProjectionMtx( m_ProjMtx );
        renderer->SetViewMtx( m_ViewMtx );
        renderer->SetModelMatrix( m_ModelMtx );

        m_GfxContext.Apply();

        renderer->Begin();
        {
            renderer->DrawLine( m_GridColor, m_GridBuffer, m_GridLineWidth );
        }
        renderer->End();
    }
}

//---------------------------------------------------------------------------
void
GridDrawer::CreateCache( nw::ut::IAllocator* allocator )
{
    NW_ASSERT_NOT_NULL( allocator );
    m_Allocator = allocator;

    // グリッド描画用バッファを作成します。
    {
        m_GridBuffer.vertex = reinterpret_cast<nw::dev::Vertex*>( allocator->Alloc( sizeof( nw::dev::Vertex ) * m_GridCount * 4, GX2_VERTEX_BUFFER_ALIGNMENT ) );
        m_GridBuffer.index = reinterpret_cast<u16*>( allocator->Alloc( sizeof( u16 ) * m_GridCount * 4, GX2_INDEX_BUFFER_ALIGNMENT ) );

        f32 offset = m_GridInterval * ( m_GridCount - 1 ) / 2.f;

        for ( s32 i = 0; i < s32(m_GridCount); ++i )
        {
            {
                m_GridBuffer.vertex[i * 4 + 0].pos.Set( m_GridInterval * i - offset, 0.f, -offset );
                m_GridBuffer.vertex[i * 4 + 0].uv.Set( 0.f, 0.5f );
                m_GridBuffer.vertex[i * 4 + 0].color.Set( 0.f, 0.f, 0.f, 0.f );
            }

            {
                m_GridBuffer.vertex[i * 4 + 1].pos.Set( m_GridInterval * i - offset, 0.f, m_GridInterval * ( m_GridCount - 1 ) - offset );
                m_GridBuffer.vertex[i * 4 + 1].uv.Set( 1.f, 0.5f );
                m_GridBuffer.vertex[i * 4 + 1].color.Set( 1.f, 0.f, 0.f, 0.f );
            }

            {
                m_GridBuffer.vertex[i * 4 + 2].pos.Set( -offset, 0.f, m_GridInterval * i - offset );
                m_GridBuffer.vertex[i * 4 + 2].uv.Set( 0.f, 0.5f );
                m_GridBuffer.vertex[i * 4 + 2].color.Set( 0.f, 0.f, 0.f, 0.f );
            }

            {
                m_GridBuffer.vertex[i * 4 + 3].pos.Set( m_GridInterval * ( m_GridCount - 1 ) - offset, 0.f, m_GridInterval * i - offset );
                m_GridBuffer.vertex[i * 4 + 3].uv.Set( 1.f, 0.5f );
                m_GridBuffer.vertex[i * 4 + 3].color.Set( 1.f, 0.f, 0.f, 0.f );
            }
        }

        for ( s32 i = 0; i < s32(m_GridCount) * 4; ++i )
        {
            m_GridBuffer.index[i] = i;
        }

        m_GridBuffer.vertexNum = m_GridCount * 4;
        m_GridBuffer.indexNum = m_GridCount * 4;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        nw::dev::PrimitiveRenderer::GetInstance()->CreateVertexBuffer( m_GridBuffer );
#else
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_GridBuffer.vertex, sizeof( nw::dev::Vertex ) * m_GridCount * 4 );
        GX2Invalidate( GX2_INVALIDATE_CPU_ATTRIB_BUFFER, m_GridBuffer.index, sizeof( u16 ) * m_GridCount * 4 );
#endif
    }
}

//---------------------------------------------------------------------------
void
GridDrawer::DestroyCache()
{
    NW_ASSERT_NOT_NULL( m_Allocator );

    nw::ut::SafeFree( m_GridBuffer.vertex, m_Allocator );
    nw::ut::SafeFree( m_GridBuffer.index, m_Allocator );
}

//---------------------------------------------------------------------------
void
GridDrawer::UpdateCache()
{
    NW_ASSERT_NOT_NULL( m_Allocator );

    DestroyCache();
    CreateCache( m_Allocator );
}

} // namespace snd
} // namespace nw
