﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <anim/AnimSoundFileBuffer.h>

namespace nw {
namespace snd {

using namespace nw::snd::internal;

//----------------------------------------------------------
AnimSoundFileBuffer::AnimSoundFileBuffer()
    : m_Allocator(NULL)
    , m_Buffer(NULL)
    , m_IsSetup(false)
{
}

//----------------------------------------------------------
AnimSoundFileBuffer::~AnimSoundFileBuffer()
{
    Finalize();
}

//----------------------------------------------------------
bool AnimSoundFileBuffer::Initialize(ut::IAllocator* allocator)
{
    if (m_Buffer != NULL)
    {
        return false;
    }

    NW_ASSERT_NOT_NULL(allocator);

    m_Allocator = allocator;

    // align は適当
    void* memory = m_Allocator->Alloc(sizeof(Impl), 64);
    if (memory == NULL)
    {
        return false;
    }

    m_Buffer = new(memory) Impl;
    m_Buffer->Resolve(m_Reference);

    return true;
}

//----------------------------------------------------------
void AnimSoundFileBuffer::Finalize()
{
    if (m_Buffer != NULL)
    {
        m_Allocator->Free(m_Buffer);
        m_Buffer = NULL;
    }
    m_Allocator = NULL;
}

//----------------------------------------------------------
void AnimSoundFileBuffer::Setup(void* bfasd)
{
    // 再読み込みを考慮して、いったんメモリを 0 で埋めてから参照を再度解決する
    std::memset(m_Buffer, 0, sizeof(Impl));
    m_Buffer->Resolve(m_Reference);

    const AnimSoundFile::FileHeader& original =
        *(reinterpret_cast<const AnimSoundFile::FileHeader*>(bfasd));

    m_Buffer->header.version = original.header.version;

    const AnimSoundFile::DataBlock& origData = *original.GetDataBlock();
    m_Buffer->data.body.frameSize = origData.body.frameSize;

    const AnimSoundFile::AnimEventTable& origTable = *origData.body.GetAnimEventTable();
    u32 eventNum = origTable.count;
    if (eventNum > Constants::ANIMEVENT_MAX)
    {
        eventNum = Constants::ANIMEVENT_MAX;
    }
    m_Buffer->animEventTableImpl.count = eventNum;

    for (u32 i = 0; i < eventNum; ++i)
    {
        m_Buffer->EnableEvent(i);

        AnimSoundFile::AnimEvent& tgtEvent = m_Buffer->animEventTableImpl.item[i];
        tgtEvent.frameInfo = origTable.item[i].frameInfo;
        AnimSoundFile::EventInfo& tgtInfo = m_Buffer->eventInfo[i];
        {
            Util::Reference reserve = tgtInfo.toSoundLabel;
            tgtInfo = *(origTable.item[i].GetEventInfo());
            tgtInfo.toSoundLabel = reserve;
        }

        std::memcpy(
            m_Buffer->soundLabel[i],
            origTable.item[i].GetEventInfo()->GetSoundLabel(),
            sizeof(char[Constants::SOUND_LABEL_MAX_LENGTH]));

        m_Reference[i].id = INVALID_ID;
        m_Reference[i].animEvent = &tgtEvent;
    }

    m_IsSetup = true;
}

//----------------------------------------------------------
bool AnimSoundFileBuffer::AddEvent(int id)
{
    NW_ASSERT_NOT_NULL(m_Buffer);

    const u32 newIdx = m_Buffer->animEventTableImpl.count;
    if (Constants::ANIMEVENT_MAX <= newIdx)
    {
        NW_WARNING(false, "[AnimSoundPlayer] Fail to add event.\n   Currently number of event must be less than 64.\n");
        return false;
    }

    m_Buffer->animEventTableImpl.count += 1; // 1つ加算
    m_Buffer->EnableEvent(newIdx);

    m_Reference[newIdx].id = id;
    m_Reference[newIdx].animEvent = &GetEvent(newIdx);

    return true;
}

//----------------------------------------------------------
void AnimSoundFileBuffer::RemoveEvent(int id)
{
    NW_ASSERT_NOT_NULL(m_Buffer);

    s32 idx = FindIndex(id);
    if (idx <= INVALID_INDEX)
    {
        return;
    }
    s32 eventNum = static_cast<s32>(m_Buffer->animEventTableImpl.count);
    NW_ASSERT(idx >= 0 && idx < eventNum);

    // idx より先にあるものを一つずつ詰めていって、
    // 最後のもの（これが2重になってる）をDisableすればよいかな

    const int tgt = eventNum - 1;
    for (int i = idx; i < tgt; ++i)
    {
        const int next = i + 1;
        GetEvent(i).frameInfo = GetEvent(next).frameInfo;

        Util::Reference reserve = GetEventInfo(i).toSoundLabel;
        GetEventInfo(i) = GetEventInfo(next);
        GetEventInfo(i).toSoundLabel = reserve;

        m_Reference[i].id = m_Reference[next].id;
        m_Reference[i].animEvent = m_Reference[next].animEvent;

        std::memcpy(
            GetEventSoundLabel(i),
            GetEventSoundLabel(next),
            sizeof(char[Constants::SOUND_LABEL_MAX_LENGTH]));
    }

    m_Buffer->animEventTableImpl.count = tgt;
    m_Buffer->DisableEvent( tgt );

    m_Reference[tgt].id = INVALID_ID;
    m_Reference[tgt].animEvent = NULL;
}

//----------------------------------------------------------
s32 AnimSoundFileBuffer::FindIndex(int id)
{
    for (int i = 0; i < Constants::ANIMEVENT_MAX; ++i)
    {
        if (m_Reference[i].id == id)
        {
            return i;
        }
    }

    return -1;
}

//----------------------------------------------------------
void AnimSoundFileBuffer::UpdateAllEventID(const u16 idList[], u32 listSize)
{
    for (u32 i = 0; i < listSize; ++i)
    {
        // 扱える最大数を超えたら何もせず抜ける
        if (i >= Constants::ANIMEVENT_MAX)
        {
            break;
        }

        m_Reference[i].id = idList[i];
    }

    // Debug
    //for (u32 i = 0; i < Constants::ANIMEVENT_MAX; ++i)
    //{
    //    NW_LOG("[%d] %d\n", i, m_Reference[i].id);
    //}
}

//----------------------------------------------------------
internal::AnimSoundFile::AnimEvent& AnimSoundFileBuffer::GetEvent(int idx)
{
    NW_ASSERT(idx < static_cast<s32>(m_Buffer->animEventTableImpl.count));
    return m_Buffer->animEventTableImpl.item[idx];
}

//----------------------------------------------------------
internal::AnimSoundFile::EventInfo& AnimSoundFileBuffer::GetEventInfo(int idx)
{
    NW_ASSERT(idx < static_cast<s32>(m_Buffer->animEventTableImpl.count));
    return m_Buffer->eventInfo[idx];
}

//----------------------------------------------------------
char* AnimSoundFileBuffer::GetEventSoundLabel(int idx)
{
    NW_ASSERT(idx < static_cast<s32>(m_Buffer->animEventTableImpl.count));
    return m_Buffer->soundLabel[idx];
}

//----------------------------------------------------------
void AnimSoundFileBuffer::Impl::Resolve(Reference reference[])
{
    // 参照を解決させる
    // ライブラリ側の対応バージョンなどに合わせなければいけないのだけれど、暫定
    header.signature = NW_UT_MAKE_SIGWORD('F', 'A', 'S', 'D');
    header.version = 0x01000000;
    header.dataBlocks = 1;
    {
        Util::ReferenceWithSize& item = blockReferenceTable.item[0];
        item.typeId = ElementType_AnimSoundFile_DataBlock;
        item.offset = (ut::GetOffsetFromPtr(&header, &data));
        item.size = sizeof(AnimSoundFile::DataBlock); // これで良いかは不明
    }
    // data.headerは使用されてない？
    data.header.kind = NW_UT_MAKE_SIGWORD('D', 'A', 'T', 'A');
    data.body.toAnimEventTable.typeId = ElementType_Table_EmbeddingTable;
    data.body.toAnimEventTable.offset =
        ut::GetOffsetFromPtr(&(data.body), &animEventTableImpl);

    animEventTableImpl.count = ut::ResU32(Constants::ANIMEVENT_MAX);
    for (int i = 0; i < Constants::ANIMEVENT_MAX; ++i)
    {
        AnimSoundFile::AnimEvent& item = animEventTableImpl.item[i];
        item.toEventInfo.typeId = ElementType_AnimSoundFile_EventInfo;
        // DisableEvent() で無効なオフセットを登録するため、ここでは設定しない
        //item.toEventInfo.offset =
        //    ut::GetOffsetFromPtr(&item, &(eventInfo[i]));

        eventInfo[i].toSoundLabel.typeId = ElementType_General_String;
        eventInfo[i].toSoundLabel.offset =
            ut::GetOffsetFromPtr(&(eventInfo[i]), &(soundLabel[i]));

        DisableEvent(i);

        reference[i].id = INVALID_ID;
        reference[i].animEvent = NULL;
    }
}

//----------------------------------------------------------
void AnimSoundFileBuffer::Impl::EnableEvent(u32 index)
{
    AnimSoundFile::AnimEvent& item = animEventTableImpl.item[index];
    item.toEventInfo.offset =
        ut::GetOffsetFromPtr(&item, &(eventInfo[index]));
}

//----------------------------------------------------------
void AnimSoundFileBuffer::Impl::DisableEvent(u32 index)
{

    AnimSoundFile::AnimEvent& item = animEventTableImpl.item[index];
    item.toEventInfo.offset = Util::Reference::INVALID_OFFSET;
}

} // namespace snd
} // namespace nw

//eof
