﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <anim/AnimSoundController.h>

#include <nw/dev.h>
#include <anim/GfxCoordinator.h>

#include "../Application.h"

// AnimSoundをコントロールするクラス。

namespace {

// 無効なサウンドID。イベントにサウンド名が指定されていないとコレが来る。
const u32 INVALID_SOUND_ID = 0xffffffff;

}

namespace nw {
namespace snd {

//----------------------------------------------------------
AnimSoundController::AnimSoundController()
    : m_Allocator(NULL)
    , m_pAnimSound(NULL)
    , m_pPlayer(NULL)
    , m_SeekDoEvent(false)
    , m_Looped(false)
    , m_IsEnabled(false)
{
}

//----------------------------------------------------------
AnimSoundController::~AnimSoundController()
{
}

//----------------------------------------------------------
bool AnimSoundController::Initialize(ut::IAllocator* allocator, SoundArchivePlayer& player)
{
    NW_ASSERT_NOT_NULL(allocator);
    NW_ASSERT(m_pAnimSound == NULL);

    m_Allocator = allocator;

    m_pPlayer = &player;

    m_CurrentBuffer = NULL;

    return true;
}

//----------------------------------------------------------
void AnimSoundController::Finalize()
{
    for (uint i=0; i<m_CacheBuffer.Size(); ++i)
    {
        DeleteCache(m_CacheBuffer[i]);
    }
}

//----------------------------------------------------------
bool AnimSoundController::IsAvailable() const
{
    if (m_CurrentBuffer == NULL || m_pAnimSound == NULL)
    {
        return false;
    }

    if (!m_pAnimSound->IsAvailable() || ! m_CurrentBuffer->IsSetuped())
    {
        return false;
    }

    return true;
}

//----------------------------------------------------------
void AnimSoundController::SetEnabled(bool enabled)
{
    m_IsEnabled = enabled;

    if (!m_IsEnabled)
    {
        m_SeekDoEvent = false;
        StopAll();
    }
}

//----------------------------------------------------------
void AnimSoundController::LoadSoundId()
{
    if (m_CurrentBuffer == NULL)
    {
        return;
    }

    const u32 eventNum = m_CurrentBuffer->GetEventCount();
    for (u32 i = 0; i < eventNum; ++i)
    {
        const u32 id = m_CurrentBuffer->GetEventInfo(i).placeForSoundId;

        if (id == INVALID_SOUND_ID)
        {
            continue;
        }

        // ストリームサウンドはロード不要
        if(Application::GetInstance().GetSoundArchive().GetSoundArchive().GetSoundType(id) == SoundArchive::SOUND_TYPE_STRM)
        {
            continue;
        }

        Application::GetInstance().GetSoundArchive().LoadData(id);
    }
}

//----------------------------------------------------------
void  AnimSoundController::StopAll()
{
    if (m_pAnimSound != NULL)
    {
        m_pAnimSound->StopAllSound();
    }
}

//----------------------------------------------------------
bool AnimSoundController::Load(const char* path, bool isReload)
{
    Cache* cache = SearchCache(path);

    if (cache == NULL || isReload)
    {
#if defined(NW_DEBUG)
        NW_LOG("cache not found, Load.\n");
#endif
        u8* bfasd = NULL;
        dev::FileDeviceManager& fs = *(dev::FileDeviceManager::GetInstance());
        dev::FileDevice::LoadArg asdArg;
        asdArg.path = path;
        asdArg.alignment = 64; // 不明
        asdArg.allocator = m_Allocator;
        bfasd = fs.Load(asdArg);
        if (bfasd == NULL)
        {
            return false;
        }

        if (cache == NULL)
        {

            void* memory = m_Allocator->Alloc(sizeof(AnimSound));
            if (memory == NULL)
            {
                return false;
            }

            cache = AddCache(path);
            cache->animSound = new(memory) AnimSound(*m_pPlayer);
        }
        else
        {
            cache->buffer.Finalize();
        }

        cache->buffer.Initialize(m_Allocator);
        cache->buffer.Setup(bfasd);

        m_Allocator->Free(bfasd);
    }
#if defined(NW_DEBUG)
    else
    {
        NW_LOG("use cache\n");
    }
#endif

    // 直前のアニメサウンドのUnsetup
    Unsetup();

    m_CurrentBuffer = &(cache->buffer);
    m_pAnimSound = cache->animSound;

    // 今から使うものを一度Unsetup
    Unsetup();
    if (Setup())
    {
        SetEnabled(true);
        return true;
    }

    return false;
}

//----------------------------------------------------------
void AnimSoundController::Unload()
{
    Unsetup();
}

//----------------------------------------------------------
bool AnimSoundController::Setup()
{
    NW_ASSERT_NOT_NULL(m_CurrentBuffer);
    if (! m_pAnimSound->Initialize(m_CurrentBuffer->ptr()))
    {
        return false;
    }

    bool result = m_pAnimSound->ConvertSoundId(m_pPlayer->GetSoundArchive());

    return result;
}

//----------------------------------------------------------
void AnimSoundController::Unsetup()
{
    if (m_pAnimSound != NULL)
    {
        m_pAnimSound->Finalize();
    }
}

//----------------------------------------------------------
void AnimSoundController::DeleteEvent(s32 id)
{
    if (!IsAvailable())
    {
        return;
    }

    NW_ASSERT_NOT_NULL(m_CurrentBuffer);
    m_CurrentBuffer->RemoveEvent(id);
}

//----------------------------------------------------------
void AnimSoundController::ModifyEvent(s32 id, const AnimSoundModifyInfo& info)
{
    if (!IsAvailable())
    {
        return;
    }

    s32 index = m_CurrentBuffer->FindIndex(id);
    if (index <= AnimSoundFileBuffer::INVALID_INDEX)
    {
        // 未登録IDの場合追加を試みる
        if (!m_CurrentBuffer->AddEvent(id))
        {
            // 上限を超えたら、ここでは何もせず抜ける
            return;
        }
    }

    Unsetup();

    // ここでは必ずインデックスは見つかるはず
    index = m_CurrentBuffer->FindIndex(id);
    NW_ASSERT(index > AnimSoundFileBuffer::INVALID_INDEX);
    internal::AnimSoundFile::AnimEvent& event = m_CurrentBuffer->GetEvent(index);
    // note: 将来的に変化する可能性あり
    {
        event.frameInfo.startFrame = info.startFrame;
        event.frameInfo.endFrame = info.endFrame;
        event.frameInfo.frameFlag = info.frameFlag;
        event.frameInfo.loopOffset = info.loopOffset;
        event.frameInfo.loopInterval = info.loopInterval;
    }

    internal::AnimSoundFile::EventInfo& eventInfo = m_CurrentBuffer->GetEventInfo(index);
    eventInfo.placeForSoundId = info.soundID;
    eventInfo.volume = info.volume;
    eventInfo.playDirection = info.playDirection;
    eventInfo.sequenceVariableNo = info.sequenceVariableNo;
    eventInfo.pitch = info.pitch;

    std::memcpy(
        m_CurrentBuffer->GetEventSoundLabel(index),
        info.soundName,
        sizeof(char[260]));
    // 更新
    Setup();
}

//----------------------------------------------------------
void AnimSoundController::UpdateAllEventID(const u16 idList[], u32 listSize)
{
    if (m_CurrentBuffer == NULL)
    {
        return;
    }

    m_CurrentBuffer->UpdateAllEventID(idList, listSize);
}

//----------------------------------------------------------
void AnimSoundController::Update()
{
    if (!IsAvailable())
    {
        return;
    }

    if (!m_IsEnabled)
    {
        return;
    }

    // 先に、GfxCoordinator側のフレーム更新が行われていることを想定したコード。
    // いっそ、Gfxの処理をこのクラスに抱えてしまった方が良いかもしれない。

    f32 target = GfxCoordinator::GetInstance().GetFrame();
    f32 delta = GfxCoordinator::GetInstance().GetRate();
    if (delta == 0.0F)
    {
        delta = target - m_pAnimSound->GetCurrentFrame();
        if (m_Looped != false)
        {
            m_Looped = false;
            delta = -delta;
        }
    }

    // そもそもこれらの条件の時は即returnでも良いかもしれない
    f32 frameSize = static_cast<f32>(m_pAnimSound->GetFrameSize() - 1);
    if (target < 0.0f)
    {
        // animのstartFrameは負の値もあり得そう
        //target += m_pAnimSound->GetFrameSize();
        target = 0.0f;
    }
    else if (frameSize < target)
    {
        //target -= m_pAnimSound->GetFrameSize();
        target = frameSize;
    }

    AnimSound::PlayDirection direction = (0.0f <= delta) ?
        AnimSound::PLAY_DIRECTION_FORWARD :
        AnimSound::PLAY_DIRECTION_BACKWARD;

    if (GfxCoordinator::GetInstance().IsPlayingAnim() != false || m_SeekDoEvent != false)
    {
        f32 current = m_pAnimSound->GetCurrentFrame();

        if (GfxCoordinator::GetInstance().IsPlayingAnim() != false)
        {
            if (current > target && direction == AnimSound::PLAY_DIRECTION_FORWARD)
            {
                m_pAnimSound->ResetFrame(GfxCoordinator::GetInstance().GetRangeStart());
            }

            if (current < target && direction == AnimSound::PLAY_DIRECTION_BACKWARD)
            {
                m_pAnimSound->ResetFrame(GfxCoordinator::GetInstance().GetRangeEnd());
            }
        }

        m_pAnimSound->UpdateFrame(target, direction);
    }
    else
    {
        m_pAnimSound->ResetFrame(target);
    }
}

//----------------------------------------------------------
AnimSoundController::Cache* AnimSoundController::AddCache(const char* path)
{
    NW_NULL_ASSERT(path);

    void* memory = m_Allocator->Alloc(sizeof(Cache));
    Cache* cache = new(memory) Cache;
    cache->buffer.Initialize(m_Allocator);
    nw::ut::strncpy(cache->path, PATH_MAX - 1, path, PATH_MAX - 1);

    // リングバッファに追加し、リングバッファからあぶれたものの後始末を。
    Cache* oldCache = m_CacheBuffer.Push(cache);
    DeleteCache(oldCache);

    return cache;
}

//----------------------------------------------------------
void AnimSoundController::DeleteCache(AnimSoundController::Cache* cache)
{
    if (cache != NULL)
    {
        cache->path[0] = '\0';
        cache->animSound->~AnimSound();
        m_Allocator->Free(cache->animSound);
        cache->buffer.Finalize();
        m_Allocator->Free(cache);
    }
}
//----------------------------------------------------------
AnimSoundController::Cache* AnimSoundController::SearchCache(const char* path)
{
    for (uint i=0; i<m_CacheBuffer.Size(); ++i)
    {
        Cache* c = m_CacheBuffer[i];
        if (c != NULL && std::strcmp(c->path, path) == 0)
        {
            return c;
        }
    }
    return NULL;
}

//----------------------------------------------------------
void AnimSoundController::SeekDoEvent( bool doEvent)
{
    m_SeekDoEvent = doEvent;
}

//----------------------------------------------------------
void AnimSoundController::Looped()
{
    m_Looped = true;
}

} // namespace snd
} // namespace nw

//eof
