﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/snd.h>
#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>
#include <Constants.h>
#ifdef NW_ENABLE_SNDCTRL
#include <nw/snd/ctrl/sndctrl_SequenceVariable.h>
#else
#include <models/SequenceVariable.h>
#endif

namespace nw {
namespace snd {

class PreviewSoundArchive;

namespace ctrl {

class SoundController;

namespace internal {

class SoundControllerImpl;

}
}

//---------------------------------------------------------------------------
//! @brief PreviewSound
//---------------------------------------------------------------------------
#if defined(NW_ENABLE_SNDCTRL)
class PreviewSound
{
public:
    enum State
    {
        STATE_STOPPED,
        STATE_PLAYING,
        STATE_PAUSED,
    };

public:
    PreviewSound();

    void Initialize(
        PreviewSoundArchive* previewSoundArchive,
        ctrl::SoundController* soundController);
    void Finalize();

    const nw::snd::SoundHandle& GetSoundHandle() const;
    nw::snd::SoundHandle& GetSoundHandle();

    nw::snd::SoundArchive::ItemId GetSoundID() const;
    void SetSoundID(nw::snd::SoundArchive::ItemId soundID);

    u32 GetSoundIndex() const;
    void SetSoundIndex(u32 index);
    void SetLabel(const char* label);

    const char* GetLabel() const;
    State GetState() const;

    void Update();

    bool Play();
    bool Stop();
    bool Pause();
    void Next(s32 offset);

    void ApplyParameters();

    u32 GetStartOffset() const;
    void SetStartOffset(u32 value);

    f32 GetPitch() const;
    void SetPitch(f32 value);

    f32 GetLpfFrequency() const;
    void SetLpfFrequency(f32 value);

    BiquadFilterType GetBiquadFilterType() const;
    void SetBiquadFilterType(BiquadFilterType value);

    f32 GetBiquadFilterValue() const;
    void SetBiquadFilterValue(f32 value);

    f32 GetVolume() const;
    void SetVolume(f32 value);

    f32 GetPan() const;
    void SetPan(f32 value);

    f32 GetSurroundPan() const;
    void SetSurroundPan(f32 value);

    f32 GetMainSend() const;
    void SetMainSend(f32 value);

    f32 GetFxSend(AuxBus auxBus) const;
    void SetFxSend(AuxBus auxBus, f32 value);

    f32 GetMainOutVolume() const;
    void SetMainOutVolume(f32 value);

    f32 GetMainPan() const;
    void SetMainPan(f32 value);

    f32 GetMainSurroundPan() const;
    void SetMainSurroundPan(f32 value);

    f32 GetMainMainSend() const;
    void SetMainMainSend(f32 value);

    f32 GetMainFxSend(AuxBus auxBus) const;
    void SetMainFxSend(AuxBus auxBus, f32 value);

    f32 GetDrcOutVolume() const;
    void SetDrcOutVolume(f32 value);

    f32 GetDrcPan() const;
    void SetDrcPan(f32 value);

    f32 GetDrcSurroundPan() const;
    void SetDrcSurroundPan(f32 value);

    f32 GetDrcMainSend() const;
    void SetDrcMainSend(f32 value);

    f32 GetDrcFxSend(AuxBus auxBus) const;
    void SetDrcFxSend(AuxBus auxBus, f32 value);


    f32 GetRemote0OutVolume() const;
    void SetRemote0OutVolume(f32 value);

    f32 GetRemote1OutVolume() const;
    void SetRemote1OutVolume(f32 value);

    f32 GetRemote2OutVolume() const;
    void SetRemote2OutVolume(f32 value);

    f32 GetRemote3OutVolume() const;
    void SetRemote3OutVolume(f32 value);

    bool GetIsMainOutEnabled() const;
    void SetIsMainOutEnabled(bool value);

    bool GetIsDrcOutEnabled() const;
    void SetIsDrcOutEnabled(bool value);

    bool GetIsRemote0OutEnabled() const;
    void SetIsRemote0OutEnabled(bool value);

    bool GetIsRemote1OutEnabled() const;
    void SetIsRemote1OutEnabled(bool value);

    bool GetIsRemote2OutEnabled() const;
    void SetIsRemote2OutEnabled(bool value);

    bool GetIsRemote3OutEnabled() const;
    void SetIsRemote3OutEnabled(bool value);

    void SetIsTrackMuteEnabled(u32 index, bool value);
    bool GetIsTrackMuteEnabled(u32 index) const;

    void SetTrackVolume(u32 index, f32 value);
    f32  GetTrackVolume(u32 index) const;

    // シーケンス変数の操作
    bool GetVariables(ctrl::internal::ToolSoundSequenceVariableContainer* pVariables);
    bool SetVariables(const ctrl::internal::ToolSoundSequenceVariableContainer* pVariables);

private:
    bool ValidateSoundID();

private:
    PreviewSoundArchive* m_pPreviewSoundArchive;
    nw::snd::ctrl::internal::SoundControllerImpl* m_SoundController;
};

#else

class PreviewSound
{
public:
    enum State
    {
        STATE_STOPPED,
        STATE_PLAYING,
        STATE_PAUSED,
    };

private:
    // シーケンス変数操作用関数
    typedef bool (*ReadVariableFunc)(
        SequenceSoundHandle* handle,
        nw::snd::internal::fnd::BinS32 trackNo,
        nw::snd::internal::fnd::BinS32 varNo,
        nw::snd::internal::fnd::BinS32* var
    );
    typedef bool (*WriteVariableFunc)(
        SequenceSoundHandle* handle,
        nw::snd::internal::fnd::BinS32 trackNo,
        nw::snd::internal::fnd::BinS32 varNo,
        nw::snd::internal::fnd::BinS32 var
    );

public:
    PreviewSound();

    void Initialize(PreviewSoundArchive* previewSoundArchive);
    void Finalize();

    const nw::snd::SoundHandle& GetSoundHandle() const;
    nw::snd::SoundHandle& GetSoundHandle();

    nw::snd::SoundArchive::ItemId GetSoundID() const;
    void SetSoundID(nw::snd::SoundArchive::ItemId soundID);

    u32 GetSoundIndex() const;
    void SetSoundIndex(u32 index);
    void SetLabel(const char* label);

    const char* GetLabel() const;
    State GetState() const;

    void Update();

    bool Play();
    bool Stop();
    bool Pause();
    void Next(s32 offset);

    void ApplyParameters();

    u32 GetStartOffset() const;
    void SetStartOffset(u32 value);

    f32 GetPitch() const;
    void SetPitch(f32 value);

    f32 GetLpfFrequency() const;
    void SetLpfFrequency(f32 value);

    BiquadFilterType GetBiquadFilterType() const;
    void SetBiquadFilterType(BiquadFilterType value);

    f32 GetBiquadFilterValue() const;
    void SetBiquadFilterValue(f32 value);

    f32 GetVolume() const;
    void SetVolume(f32 value);

    f32 GetPan() const;
    void SetPan(f32 value);

    f32 GetSurroundPan() const;
    void SetSurroundPan(f32 value);

    f32 GetMainSend() const;
    void SetMainSend(f32 value);

    f32 GetFxSend(AuxBus auxBus) const;
    void SetFxSend(AuxBus auxBus, f32 value);

    f32 GetMainOutVolume() const;
    void SetMainOutVolume(f32 value);

    f32 GetMainPan() const;
    void SetMainPan(f32 value);

    f32 GetMainSurroundPan() const;
    void SetMainSurroundPan(f32 value);

    f32 GetMainMainSend() const;
    void SetMainMainSend(f32 value);

    f32 GetMainFxSend(AuxBus auxBus) const;
    void SetMainFxSend(AuxBus auxBus, f32 value);

    f32 GetDrcOutVolume() const;
    void SetDrcOutVolume(f32 value);

    f32 GetDrcPan() const;
    void SetDrcPan(f32 value);

    f32 GetDrcSurroundPan() const;
    void SetDrcSurroundPan(f32 value);

    f32 GetDrcMainSend() const;
    void SetDrcMainSend(f32 value);

    f32 GetDrcFxSend(AuxBus auxBus) const;
    void SetDrcFxSend(AuxBus auxBus, f32 value);

    f32 GetRemote0OutVolume() const;
    void SetRemote0OutVolume(f32 value);

    f32 GetRemote1OutVolume() const;
    void SetRemote1OutVolume(f32 value);

    f32 GetRemote2OutVolume() const;
    void SetRemote2OutVolume(f32 value);

    f32 GetRemote3OutVolume() const;
    void SetRemote3OutVolume(f32 value);

    bool GetIsMainOutEnabled() const;
    void SetIsMainOutEnabled(bool value);

    bool GetIsDrcOutEnabled() const;
    void SetIsDrcOutEnabled(bool value);

    bool GetIsRemote0OutEnabled() const;
    void SetIsRemote0OutEnabled(bool value);

    bool GetIsRemote1OutEnabled() const;
    void SetIsRemote1OutEnabled(bool value);

    bool GetIsRemote2OutEnabled() const;
    void SetIsRemote2OutEnabled(bool value);

    bool GetIsRemote3OutEnabled() const;
    void SetIsRemote3OutEnabled(bool value);

    void SetIsTrackMuteEnabled(u32 index, bool value);
    bool GetIsTrackMuteEnabled(u32 index) const;

    void SetTrackVolume(u32 index, f32 value);
    f32  GetTrackVolume(u32 index) const;

    // シーケンス変数の操作
    bool GetVariables(ToolSoundSequenceVariableContainer* pVariables);
    bool SetVariables(const ToolSoundSequenceVariableContainer* pVariables);
    void InvalidateAllVariables();

private:
    void ApplySequenceVariables();

    // シーケンス変数の制御
    bool GetVariables(ToolSoundSequenceVariableContainer* pVariables, ReadVariableFunc pFunc);
    bool SetVariables(SequenceVariableContainer& Variables, WriteVariableFunc pFunc);
    void InvalidateVariables(ToolSoundSequenceVariableContainer* pVariables);

    SequenceVariableContainer& GetGlobalVariables();

    void UpdateState();
    void SetParameterDirty(bool isDirty);
    bool ValidateSoundID();

private:
    PreviewSoundArchive* m_pPreviewSoundArchive;

    nw::snd::SoundArchive::ItemId m_SoundID;
    nw::snd::SoundHandle m_SoundHandle;

    State m_State;
    bool m_IsParameterDirty;

    // 共通設定
    u32 m_StartOffset;
    f32 m_Pitch;
    f32 m_LpfFrequency;
    BiquadFilterType m_BiquadFilterType;
    f32 m_BiquadFilterValue;

    // 全体設定
    f32 m_Volume;
    f32 m_Pan;
    f32 m_SurroundPan;
    f32 m_MainSend;
    f32 m_FxSendA;
    f32 m_FxSendB;
    f32 m_FxSendC;

    // MAIN設定
    f32 m_MainOutVolume;
    f32 m_MainPan;
    f32 m_MainSurroundPan;
    f32 m_MainMainSend;
    f32 m_MainFxSendA;
    f32 m_MainFxSendB;
    f32 m_MainFxSendC;

    // DRC設定
    f32 m_DrcOutVolume;
    f32 m_DrcPan;
    f32 m_DrcSurroundPan;
    f32 m_DrcMainSend;
    f32 m_DrcFxSendA;
    f32 m_DrcFxSendB;
    f32 m_DrcFxSendC;

    // リモート設定
    f32 m_Remote0OutVolume;
    f32 m_Remote1OutVolume;
    f32 m_Remote2OutVolume;
    f32 m_Remote3OutVolume;

    bool m_IsMainOutEnabled;
    bool m_IsDrcOutEnabled;
    bool m_IsRemote0OutEnabled;
    bool m_IsRemote1OutEnabled;
    bool m_IsRemote2OutEnabled;
    bool m_IsRemote3OutEnabled;
    bool m_TracksMute[Constants::PREVIEW_SOUND_TRACK_MAX];
    f32 m_TracksVolume[Constants::PREVIEW_SOUND_TRACK_MAX];

    // 再生開始時にシーケンス変数を設定するためのコンテナ
    SequenceVariableContainer  m_LocalVariables;
    SequenceVariableContainer  m_TrackVariables[ SequenceVariableTraits::TrackCount ];

    nw::ut::CriticalSection m_Lock;
};

#endif

} // nw::snd
} // nw
