﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SOUNDPLAYER_HIO_MESSAGE_HANDLER_H_
#define NW_SOUNDPLAYER_HIO_MESSAGE_HANDLER_H_

#include <nw/ut/ut_Delegate.h>
#include <nw/ut/ut_IoStream.h>
#include <nw/ut/ut_LinkList.h>
#include <mcs/HioProtocol.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    HIOメッセージを処理するクラスです。
//!
//! @details  GetTargetMessageID() が返すHIOメッセージを対象に
//!           OnInvoke にてメッセージ処理します。
//---------------------------------------------------------------------------
class HioMessageHandler
{
public:
    //! @brief 実行イベントです。
    typedef nw::ut::DelegateEvent0 InvokedEvent;

    //! @brief 実行イベントのハンドラです。
    typedef InvokedEvent::Slot InvokedEventHandler;

    //! @brief LinkList 用の参照クラスです。
    struct Reference
    {
        Reference(HioMessageHandler& value) : value(value) { }

        HioMessageHandler& value;   //!< HioMessageHandler への参照です。
        nw::ut::LinkListNode node;  //!< LinkListNode です。
    };

public:
    /* ctor */ HioMessageHandler() : m_Reference(*this) { }
    /* dtor */ virtual ~HioMessageHandler() { }

public:
    //---------------------------------------------------------------------------
    //! @brief         実行イベントを取得します。
    //!
    //! @return        実行イベントを返します。
    //---------------------------------------------------------------------------
    InvokedEvent& GetInvokedEvent() { return m_InvokedEvent; }

    //---------------------------------------------------------------------------
    //! @brief         HIOメッセージを処理します。
    //!
    //! @details       HIOメッセージを処理します。
    //!
    //! @param[in]     header       HIOメッセージヘッダを指定します。
    //! @param[in]     data         HIOメッセージデータを指定します。
    //! @param[in]     writeStream  返信用の出力ストリームを指定します。
    //---------------------------------------------------------------------------
    void Invoke(const HioMessageHeader& header, const void* data, nw::ut::IoStream& writeStream)
    {
        if(!CanRespond(header))
        {
#if defined(NW_DEBUG)
            NW_WARNING(false, "%s(0x%08x) is not invoked.", header.message.ToString(), u32(header.message));
#endif
            return;
        }

        OnInvoke(header, data, writeStream);
        m_InvokedEvent.Invoke();
    }

    //---------------------------------------------------------------------------
    //! @brief         HIOメッセージを処理可能かどうかを調べます。
    //!
    //! @param[in]     header  HIOメッセージヘッダを指定します。
    //! @param[in]     data    HIOメッセージデータを指定します。
    //!
    //! @return        HIOメッセージを処理可能な場合は true、不可能な場合は false を返します。
    //---------------------------------------------------------------------------
    virtual bool CanRespond(const HioMessageHeader& header) const
    {
        return header.message == GetTargetMessage();
    }

    //---------------------------------------------------------------------------
    //! @brief         LinkList 用の参照を取得します。
    //!
    //! @details       この関数の戻り値である HioMessageHandler::Reference を利用して、
    //!                HioMessageHandler を格納する LinkList を構築できます。
    //!
    //! @return        HioMessageHandler::Reference を返します。
    //---------------------------------------------------------------------------
    Reference& GetReference() { return m_Reference; }

protected:
    //---------------------------------------------------------------------------
    //! @brief         処理対象のHIOメッセージを取得します。
    //!
    //! @return        処理対象のHIOメッセージを返します。
    //---------------------------------------------------------------------------
    virtual HioMessageType GetTargetMessage() const = 0;

    //---------------------------------------------------------------------------
    //! @brief         HIOメッセージを処理します。
    //!
    //! @param[in]     header  HIOメッセージヘッダを指定します。
    //! @param[in]     data    HIOメッセージデータを指定します。
    //! @param[in]     writeStream  返信用の出力ストリームを指定します。
    //---------------------------------------------------------------------------
    virtual void OnInvoke(
        const HioMessageHeader& header,
        const void* data,
        nw::ut::IoStream& writeStream) = 0;

private:
    InvokedEvent m_InvokedEvent;    //!< 実行イベントです。
    Reference    m_Reference;       //!< LinkList 用の参照です。
};

} // namespace nw::snd
} // namespace nw

#endif // NW_SOUNDPLAYER_HIO_MESSAGE_HANDLER_H_
