﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <float.h>
#include <math.h>
#include <binary/IffFormat.h>

namespace nw {
namespace snd {

#pragma pack(1)

//---------------------------------------------------------------------------
//! @brief  IEEE Standard 754 floating point number
//---------------------------------------------------------------------------
typedef struct IEEE754
{
    IEEE754() :
        head(0),
        hiMant(0),
        loMant(0)
    { }

    //---------------------------------------------------------------------------

    operator f64() const
    {
        f64 result;
        int expon;

        expon = head & 0x7fff;

        if (expon == 0 && hiMant == 0 && loMant == 0)
        {
            result = 0;
        }
        else
        {
            if (expon == 0x7FFF)
            {    /* Infinity or NaN */
                result = DBL_MAX;
            }
            else
            {
                expon -= 0x3fff;

                result = hiMant * pow(2.0, (expon -= 31));
                result += loMant * pow(2.0, (expon -= 32));
            }
        }

        return (head & 0x8000) != 0 ? -result : result;
    }

    //---------------------------------------------------------------------------

    internal::fnd::BinU16 head;
    internal::fnd::BinU32 hiMant;
    internal::fnd::BinU32 loMant;
} IEEE754;

//---------------------------------------------------------------------------
//! @brief  Aiff COMM チャンク
//---------------------------------------------------------------------------
typedef struct AiffCommonChunk
{
    static const u32 ValidID = 'COMM';

    /* ctor */ AiffCommonChunk() :
        header(ValidID),
        numChannels(0),
        numSampleFrames(0),
        sampleSize(0)
    { }

    //---------------------------------------------------------------------------

    static u32 GetValidBodySize()
    {
        return sizeof(AiffCommonChunk) - sizeof(ChunkHeader);
    }

    bool IsValid()
    {
        return header.id == ValidID;
    }

    //---------------------------------------------------------------------------

    BigEndianChunkHeader header;
    internal::fnd::BinU16 numChannels;
    internal::fnd::BinU32 numSampleFrames;
    internal::fnd::BinU16 sampleSize;
    IEEE754 sampleRate;
} AiffCommChunk;

//---------------------------------------------------------------------------
//! @brief  Aiff SSND チャンク
//---------------------------------------------------------------------------
typedef struct AiffSoundDataChunk
{
    static const u32 ValidID = 'SSND';

    /* ctor */ AiffSoundDataChunk() :
        header(ValidID),
        offset(0),
        blockSize(0)
    { }

    //---------------------------------------------------------------------------

    static u32 GetValidBodySize()
    {
        return sizeof(AiffSoundDataChunk) - sizeof(ChunkHeader);
    }

    bool IsValid()
    {
        return header.id == ValidID;
    }

    //---------------------------------------------------------------------------

    BigEndianChunkHeader header;
    internal::fnd::BinU32 offset;
    internal::fnd::BinU32 blockSize;
} AiffSoundDataChunk;

//---------------------------------------------------------------------------
//! @brief  Aiff FORM チャンクヘッダ
//---------------------------------------------------------------------------
typedef struct AiffFormChunkHeader
{
    static const u32 ValidID = 'FORM';
    static const u32 ValidFormatType = 'AIFF';

    /* ctor */ AiffFormChunkHeader() :
        header(ValidID)
    { }

    //---------------------------------------------------------------------------

    bool IsValid()
    {
        return header.id == ValidID && formType == ValidFormatType;
    }

    //---------------------------------------------------------------------------

    BigEndianChunkHeader header;
    internal::fnd::BinU32 formType;
} AiffFormChunkHeader;

#pragma pack()

} // snd
} // nw
