﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "MidiManager.h" // for NoteOnCallback
#include "Channel.h"

class UnmanagedSequenceSoundPlayer :
    public nn::atk::detail::driver::NoteOnCallback
{
public:
    typedef nn::atk::detail::driver::Channel* (*NoteOnCallback)(
        nn::atk::detail::driver::SequenceSoundPlayer* seqPlayer,
        int bankNo,
        const nn::atk::detail::driver::NoteOnInfo& noteOnInfo,
        void* userData
    );

    UnmanagedSequenceSoundPlayer();
    ~UnmanagedSequenceSoundPlayer();

    void Initialize() {
        auto* pReceiver = NintendoWare::SoundRuntime::ChannelCafe::GetOutputReceiver();
        mPlayer.Initialize( pReceiver );
        pReceiver->AddReferenceCount( 1 );
    }
    void Finalize() { mPlayer.Finalize(); }

    bool Setup( uint32_t allocTrackFlag, NoteOnCallback noteOnCallback, void* userData );
    void ForceTrackMute(uint32_t trackMask) { mPlayer.ForceTrackMute(trackMask); }
    void SetSeqData( const void* seq, const void* banks[], const void* warcs[], bool warcIsIndividuals[], int seqOffset ) {
        nn::atk::detail::driver::SequenceSoundPlayer::PrepareArg arg;
        arg.seqFile = seq;
        for (int i = 0; i < nn::atk::SeqBankMax; i++)
        {
            arg.bankFiles[i] = banks[i];
            arg.warcFiles[i] = warcs[i];
            arg.warcIsIndividuals[i] = warcIsIndividuals[i];
        }
        arg.seqOffset = seqOffset;
        mPlayer.Prepare( arg );
    }
    void Start() { mPlayer.Start(); }
    void Pause(bool flag) { mPlayer.Pause(flag); }
    void Stop() { mPlayer.Stop(); }

    bool IsActive() { return mPlayer.IsActive(); }
    bool IsStarted() { return mPlayer.IsStarted(); }
    bool IsPause() { return mPlayer.IsPause(); }
    bool IsPlayFinished() { return mPlayer.IsPlayFinished(); }

    void SetVolume( float volume ) { mPlayer.SetVolume(volume);}
    void SetChannelPriority( int prio ) { mPlayer.SetChannelPriority(prio);}

    nn::atk::detail::driver::SequenceTrack* GetPlayerTrack( int trackNo ) { return mPlayer.GetPlayerTrack( trackNo );}
    const nn::atk::detail::driver::SequenceTrack* GetPlayerTrack( int trackNo ) const { return mPlayer.GetPlayerTrack( trackNo );}

    short GetLocalVariable( int varNo ) const { return mPlayer.GetLocalVariable(varNo); }
    void SetLocalVariable( int varNo, short var ) { mPlayer.SetLocalVariable(varNo,var);}

    static short GetGlobalVariable( int varNo ) { return nn::atk::detail::driver::SequenceSoundPlayer::GetGlobalVariable(varNo); }
    static void SetGlobalVariable( int varNo, short var ) { nn::atk::detail::driver::SequenceSoundPlayer::SetGlobalVariable(varNo,var);}

    short GetTrackVariable( int trackNo, int varNo ) const;
    void SetTrackVariable( int trackNo, int varNo, short var );


private:
    static const int TRACK_BUFFER_SIZE = sizeof(nn::atk::detail::driver::MmlSequenceTrack) * 16;

    virtual nn::atk::detail::driver::Channel* NoteOn(
        nn::atk::detail::driver::SequenceSoundPlayer* seqPlayer,
        uint8_t bankIndex,
        const nn::atk::detail::driver::NoteOnInfo& noteOnInfo
    );

    NoteOnCallback mNoteOnCallback;
    void* mUserData;
    nn::atk::detail::driver::SequenceSoundPlayer mPlayer;
    nn::atk::detail::driver::MmlParser mMmlParser;
    nn::atk::detail::driver::MmlSequenceTrackAllocator mTrackAllocator;
    void* mMmlSequenceTrackBuffer;
};

namespace NintendoWare { namespace SoundRuntime {

using namespace System;
using namespace System::Text;
using namespace System::Runtime::InteropServices;

public ref class SequenceSoundPlayerCafe
{
public:
    SequenceSoundPlayerCafe() {
        mUnmanaged = new UnmanagedSequenceSoundPlayer();
        mUnmanagedNoteOnCallback = gcnew UnmanagedNoteOnCallback( NoteOn );
        mNoteOnCallbackHandle = GCHandle::Alloc( mUnmanagedNoteOnCallback );
        mGCHandle = GCHandle::Alloc( this );
    }
    ~SequenceSoundPlayerCafe() {
        Stop();
        mNoteOnCallbackHandle.Free();
        mGCHandle.Free();
        delete mUnmanaged;
    }

    void Setup( uint32_t allocTrackFlag, NoteOnCallback^ callback, Object^ userData )
    {
        IntPtr ip = Marshal::GetFunctionPointerForDelegate(mUnmanagedNoteOnCallback);
        UnmanagedMidiManager::NoteOnCallback cb =
            static_cast<UnmanagedMidiManager::NoteOnCallback>(ip.ToPointer());

        mUserCallback = callback;
        mUserData = userData;

        mUnmanaged->Setup( allocTrackFlag, cb, GCHandle::ToIntPtr(mGCHandle).ToPointer() );
    }
    void ForceTrackMute( uint32_t trackMask )
    {
        mUnmanaged->ForceTrackMute( trackMask );
    }
    void SetSeqData( IntPtr seq, array<IntPtr>^ banks, array<IntPtr>^ warcs, array<bool>^ warcsIsIndividuals, int seqOffset )
    {
        pin_ptr<IntPtr> banksPtr = &banks[0];
        pin_ptr<IntPtr> warcsPtr = &warcs[0];
        pin_ptr<bool>   warcsIsIndividualsPtr = &warcsIsIndividuals[0];

        try
        {
            mUnmanaged->SetSeqData(
                seq.ToPointer(),
                reinterpret_cast<const void**>(banksPtr),
                reinterpret_cast<const void**>(warcsPtr),
                warcsIsIndividualsPtr,
                seqOffset );
        }
        finally
        {
            banksPtr = nullptr;
            warcsPtr = nullptr;
        }
    }
    void Initialize()
    {
        mUnmanaged->Initialize();
    }
    void Start()
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->Start();
        AtkSoundSystem::UnlockSoundThread();
    }
    void Pause( bool flag )
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->Pause( flag );
        AtkSoundSystem::UnlockSoundThread();
    }
    void Stop()
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->Stop();
        mUnmanaged->Finalize();
        AtkSoundSystem::UnlockSoundThread();
    }
    bool IsActive()
    {
        return mUnmanaged->IsActive();
    }
    bool IsStarted()
    {
        return mUnmanaged->IsStarted();
    }
    bool IsPause()
    {
        return mUnmanaged->IsPause();
    }
    bool IsPlayFinished()
    {
        return mUnmanaged->IsPlayFinished();
    }
    void SetVolume( float volume )
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->SetVolume( volume );
        AtkSoundSystem::UnlockSoundThread();
    }
    void SetChannelPriority( int prio )
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->SetChannelPriority( prio );
        AtkSoundSystem::UnlockSoundThread();
    }

    short GetLocalVariable( int varNo )
    {
        return mUnmanaged->GetLocalVariable(varNo);
    }
    void SetLocalVariable( int varNo, short var )
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->SetLocalVariable(varNo,var);
        AtkSoundSystem::UnlockSoundThread();
    }

    static short GetGlobalVariable( int varNo )
    {
        return UnmanagedSequenceSoundPlayer::GetGlobalVariable(varNo);
    }
    static void SetGlobalVariable( int varNo, short var )
    {
        AtkSoundSystem::LockSoundThread();
        UnmanagedSequenceSoundPlayer::SetGlobalVariable(varNo,var);
        AtkSoundSystem::UnlockSoundThread();
    }

    short GetTrackVariable( int trackNo, int varNo )
    {
        return mUnmanaged->GetTrackVariable(trackNo,varNo);
    }
    void SetTrackVariable( int trackNo, int varNo, short var )
    {
        AtkSoundSystem::LockSoundThread();
        mUnmanaged->SetTrackVariable(trackNo,varNo,var);
        AtkSoundSystem::UnlockSoundThread();
    }

private:
    [UnmanagedFunctionPointer(CallingConvention::Cdecl)]
    delegate IntPtr UnmanagedNoteOnCallback(
        IntPtr seqPlayer,
        int bankNo,
        IntPtr noteOnInfo,
        IntPtr userData
    );

    static IntPtr NoteOn(
        IntPtr seqPlayerPtr,
        int bankNo,
        IntPtr noteOnInfoPtr,
        IntPtr userData
    )
    {
        NoteOnInfo % noteOnInfo = safe_cast<NoteOnInfo%>(Marshal::PtrToStructure(noteOnInfoPtr,NoteOnInfo::typeid));
        GCHandle handle = GCHandle::FromIntPtr(userData);
        SequenceSoundPlayerCafe^ seqPlayer = safe_cast<SequenceSoundPlayerCafe^>(handle.Target);

        if ( seqPlayer->mUserCallback == nullptr ) return IntPtr::Zero;

        ChannelCafe^ channel = seqPlayer->mUserCallback(
            seqPlayerPtr,
            bankNo,
            noteOnInfo,
            seqPlayer->mUserData
        );
        if ( channel == nullptr ) return IntPtr::Zero;
        return channel->GetUnmanagedPtr();
    }

    UnmanagedSequenceSoundPlayer* mUnmanaged;
    UnmanagedNoteOnCallback^ mUnmanagedNoteOnCallback;
    GCHandle mNoteOnCallbackHandle;
    GCHandle mGCHandle;

    NoteOnCallback^ mUserCallback;
    Object^ mUserData;
};

public ref class SeqFileReaderCafe
{
public:
    SeqFileReaderCafe( IntPtr seqData ) {
        m_pSeqFile = seqData.ToPointer();
        mUnmanaged = new nn::atk::detail::SequenceSoundFileReader( m_pSeqFile );
    }
    ~SeqFileReaderCafe() {
        delete mUnmanaged;
    }
    IntPtr GetBaseAddress() {
        // return IntPtr( const_cast<void*>( mUnmanaged->GetBaseAddress() ) );
        return IntPtr( const_cast<void*>( mUnmanaged->GetSequenceData() ) );
    }
    IntPtr GetFileAddress() {
        return IntPtr( m_pSeqFile );
    }
    bool ReadOffsetByLabel( String^ labelName, [Out] UInt32 % offsetPtr ) {
        IntPtr hString = Marshal::StringToHGlobalAnsi(labelName);
        uint32_t offset;
        // bool result = mUnmanaged->ReadOffsetByLabel((const char*)hString.ToPointer(),&offset);
        bool result = mUnmanaged->GetOffsetByLabel((const char*)hString.ToPointer(),&offset);
        if ( result ) {
            offsetPtr = offset;
        }
        Marshal::FreeHGlobal(hString);
        return result;
    }

private:
    nn::atk::detail::SequenceSoundFileReader* mUnmanaged;
    void* m_pSeqFile;
};


}}
