﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sndlib/exprparser.h"
#include "sndlib/lexer.h"

namespace sndlib
{

using namespace std;

// TODO: 単項マイナスや、プラスの厳密にする（式のはじめのみ認める）

// 演算子優先順位（低い順）
#define OP_PRIO_OR       0
#define OP_PRIO_AND      1
#define OP_PRIO_EQUAL    2
#define	OP_PRIO_COMPARE  3
#define	OP_PRIO_SHIFT    4
#define	OP_PRIO_PLUS     5
#define	OP_PRIO_MUL      6
#define	OP_PRIO_PRIM     7

ExprParser::ExprParser( bool undefinedSymbolIsZero )
    : mUndefinedSymbolIsZero( undefinedSymbolIsZero )
{
}

ExprParser::~ExprParser(void)
{
}

__int64 ExprParser::ParseSymbol( const string& symbol, Lexer& lexer )
{
    for ( int i = 0; i < SYMBOL_TABLE_NUM ; i++ ) {
        if ( const SymbolTable* symTable = mSymbolTable[ i ] ) {
            SymbolTable::const_iterator p =
                symTable->find( symbol );
            if ( p != symTable->end() ) {
                return p->second;
            }
        }
    }

    if ( mUndefinedSymbolIsZero ) return 0;

    stringstream s;
    s << "Undefined symbol \"" << symbol << "\"";
    throw Lexer::Exception( s.str(), lexer );
}

template< int T >
__int64 ExprParser::ParseTerm( Lexer& lexer )
{
    Lexer::Token token = lexer.ReadToken();

    switch( token ) {
    case Lexer::NUMBER:
    {
        const __int64 x = lexer.GetNumberValue();
        lexer.ReadToken();
        return x;
    }

    case Lexer::LBRACE:
    {
        __int64 x = 0;
        while( 1 ) {
            token = lexer.ReadToken();
            if ( token != Lexer::NUMBER ) {
                throw Lexer::Exception("Bit specified constants are illegal.", lexer );
            }
            const __int64 n = lexer.GetNumberValue();
            if ( n < 0 ) {
                throw Lexer::Exception("Bit number is negative.", lexer );
            }
            if ( n > 63 ) {
                throw Lexer::Exception("Constant too big", lexer );
            }
            token = lexer.ReadToken();
            if ( token == Lexer::MINUS ) {
                token = lexer.ReadToken();
                if ( token != Lexer::NUMBER ) {
                    throw Lexer::Exception("Bit specified constants are illegal.", lexer );
                }
                const __int64 end = lexer.GetNumberValue();
                if ( end < 0 ) {
                    throw Lexer::Exception("Bit number is negative.", lexer );
                }
                if ( end > 63 ) {
                    throw Lexer::Exception("Constant too big", lexer );
                }
                if ( n < end ) {
                    for( __int64 b = n; b <= end ; b++ ) {
                        x |= ( 1LL << b );
                    }
                }
                else {
                    for( __int64 b = end; b <= n ; b++ ) {
                        x |= ( 1LL << b );
                    }
                }
                token = lexer.ReadToken();
            }
            else {
                x  |= ( 1LL << n );
            }

            if ( token == Lexer::RBRACE ) break;
            if ( token == Lexer::COMMA ) continue;

            throw Lexer::Exception("Bit specified constants are illegal.", lexer );
        }
        lexer.ReadToken();
        return x;
    }

    case Lexer::SYMBOL:
    {
        const string symbol = lexer.GetStringValue();
        __int64 x = ParseSymbol( symbol, lexer );
        lexer.ReadToken();
        return x;
    }

    case Lexer::MINUS:
        return - ParseTerm< T >( lexer );

    case Lexer::PLUS:
        return ParseTerm< T >( lexer );

    case Lexer::LP:
    {
        __int64 x = ParseTerm< 0 >( lexer );
        token = lexer.GetLastToken();
        if ( token != Lexer::RP ) {
            throw Lexer::Exception( "Unmatch ( )", lexer );
        }
        lexer.ReadToken();
        return x;
    }

    default:
    {
        stringstream s;
        s << "Syntax error \"" << lexer.GetLastTokenString() << "\"";
        throw Lexer::Exception( s.str(), lexer );
    }
    }

}


template<>
__int64 ExprParser::ParseTerm< OP_PRIO_MUL >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_MUL+1 >( lexer );

    while( 1 )
    {
        Lexer::Token token = lexer.GetLastToken();

        switch( token ) {
        case Lexer::MUL:
            left *= ParseTerm< OP_PRIO_MUL+1 >( lexer );
            break;

        case Lexer::DIV:
            if ( __int64 d = ParseTerm< OP_PRIO_MUL+1 >( lexer ) ) {
                left /= d;
                break;
            }
            throw Lexer::Exception( "Devide by zero", lexer );

        default:
            return left;
        }
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_PLUS >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_PLUS+1 >( lexer );

    while( 1 )
    {
        Lexer::Token token = lexer.GetLastToken();

        switch( token ) {
        case Lexer::PLUS:
            left += ParseTerm< OP_PRIO_PLUS+1 >( lexer );
            break;

        case Lexer::MINUS:
            left -= ParseTerm< OP_PRIO_PLUS+1 >( lexer );
            break;

        default:
            return left;
        }
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_SHIFT >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_SHIFT+1 >( lexer );

    while( 1 )
    {
        Lexer::Token token = lexer.GetLastToken();

        switch( token ) {
        case Lexer::LSHIFT:
            left <<= ParseTerm< OP_PRIO_SHIFT+1 >( lexer );
            break;

        case Lexer::RSHIFT:
            left >>= ParseTerm< OP_PRIO_SHIFT+1 >( lexer );
            break;

        default:
            return left;
        }
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_COMPARE >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_COMPARE+1 >( lexer );

    Lexer::Token token = lexer.GetLastToken();

    switch( token ) {
    case Lexer::LT:
        return left < ParseTerm< OP_PRIO_COMPARE+1 >( lexer );

    case Lexer::LE:
        return left <= ParseTerm< OP_PRIO_COMPARE+1 >( lexer );

    case Lexer::GT:
        return left > ParseTerm< OP_PRIO_COMPARE+1 >( lexer );

    case Lexer::GE:
        return left >= ParseTerm< OP_PRIO_COMPARE+1 >( lexer );

    default:
        return left;
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_EQUAL >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_EQUAL+1 >( lexer );

    Lexer::Token token = lexer.GetLastToken();

    switch( token ) {
    case Lexer::EQ:
        return left == ParseTerm< OP_PRIO_EQUAL+1 >( lexer );

    default:
        return left;
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_AND >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_AND+1 >( lexer );

    while( 1 )
    {
        Lexer::Token token = lexer.GetLastToken();

        switch( token ) {
        case Lexer::AND:
            left &= ParseTerm< OP_PRIO_AND+1 >( lexer );
            break;

        default:
            return left;
        }
    }
}

template<>
__int64 ExprParser::ParseTerm< OP_PRIO_OR >( Lexer& lexer )
{
    __int64 left = ParseTerm< OP_PRIO_OR+1 >( lexer );

    while( 1 )
    {
        Lexer::Token token = lexer.GetLastToken();

        switch( token ) {
        case Lexer::OR:
            left |= ParseTerm< OP_PRIO_OR+1 >( lexer );
            break;

        default:
            return left;
        }
    }
}

__int64 ExprParser::Parse(
    Lexer& lexer,
    const SymbolTable* symTable1,
    const SymbolTable* symTable2,
    const SymbolTable* symTable3
    )
{
    mSymbolTable[0] = symTable1;
    mSymbolTable[1] = symTable2;
    mSymbolTable[2] = symTable3;

    return ParseTerm< 0 >( lexer );
}

} // namespace sndlib

