﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef SMFCONV_H_
#define SMFCONV_H_

#include <exception>
#include <string>
#include <vector>
#include <sndlib/cmdline.h>

void print_usage();
const std::string MakeTrackLabelPrefix( const std::string& prefix, int number );
const std::string MakeLocalTrackLabelPrefix( int number );

class ParseException : public std::exception {
public:
    explicit ParseException(const std::string& _S)
        : std::exception(""), _Str(_S) {}
    virtual const char *what() const
        {return (_Str.c_str()); }

public:
    std::string filename;
private:
    std::string _Str;
};

class ChannelEventException : public ParseException {
public:
    explicit ChannelEventException(const std::string& _S, int _channel, int _tick);

    int channel;
    unsigned int measure;
    unsigned int beat;
    unsigned int tick;
};

class MetaEventException : public ParseException {
public:
    explicit MetaEventException(const std::string& _S, int _tick);

    unsigned int measure;
    unsigned int beat;
    unsigned int tick;
};

class ControlChangeEventException : public ParseException {
public:
    explicit ControlChangeEventException(const std::string& _S, int _cc, int _value, int _tick);

    int controlChange;
    int value;
    unsigned int measure;
    unsigned int beat;
    unsigned int tick;
};

class my_exception : public std::exception {
public:
    explicit my_exception(const std::string& _S)
        : exception(""), _Str(_S) {}
    virtual const char *what() const
        {return (_Str.c_str()); }
private:
    std::string _Str;
};

struct CommandLineArg
{
    CommandLineArg()
        : update_flag(false),
          verbose_flag(false),
          help_flag( false),
          timebase(48)
    {
    }

    typedef std::vector<std::string> FileList_t;

    FileList_t files;
    bool verbose_flag;
    bool update_flag;
    bool help_flag;
    std::string out_file;
    int timebase;
};

class MyCommandLineArgParser : public sndlib::CommandLineArgParser
{
public:
    MyCommandLineArgParser( CommandLineArg& arg ) :
        mArg( arg )
    {
        AddOption("h");
        AddOption("help");
        AddOption("u");
        AddOption("update");
        AddOption("v");
        AddOption("verbose");
        AddOptionWithArg("o");
        AddOptionWithArg("timebase");
    }

protected:
    virtual void OptionHandler( std::string opt, std::string arg )
    {
        if ( opt == "help" || opt == "h" ) { mArg.help_flag = true; }
        else if ( opt == "verbose" || opt == "v" ) { mArg.verbose_flag = true; }
        else if ( opt == "update" || opt == "u" ) { mArg.update_flag = true; }
        else if ( opt == "o" ) { mArg.out_file = arg; }
        else if ( opt == "timebase" ) { mArg.timebase = StringToInt(arg); }
    }
    virtual void ArgHandler( std::string arg )
    {
        mArg.files.push_back( arg );
    }

private:
    static int StringToInt( const std::string& str )
    {
        char* e;
        unsigned long i = strtoul(str.c_str(),&e,0);
        if ( *e != '\0' ) {
            std::stringstream s;
            s << "invalid timebase \"" << str <<"\"";
            throw sndlib::CommandLineArgParserException( s.str() );
        }
        return i;
    }

    CommandLineArg& mArg;
};

#endif // SMFCONV_H_

