﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(AFX_SMFPARSER_H__C0E5CD18_72A1_4E19_9BBD_00A9C1EFEB2C__INCLUDED_)
#define AFX_SMFPARSER_H__C0E5CD18_72A1_4E19_9BBD_00A9C1EFEB2C__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "SmfEvent.h"
#include "smfconv.h"
#include <deque>

typedef sndlib::BigEndian Endian;
typedef sndlib::strm::binary<sndlib::uint8_t,  Endian> u8;
typedef sndlib::strm::binary<sndlib::int8_t,   Endian> s8;
typedef sndlib::strm::binary<sndlib::uint16_t, Endian> u16;
typedef sndlib::strm::binary<sndlib::int16_t,  Endian> s16;
typedef sndlib::strm::binary<sndlib::uint32_t, Endian> u32;
typedef sndlib::strm::binary<sndlib::int32_t,  Endian> s32;


struct BeatInfo
{
    unsigned int measure;
    unsigned int beat;
    unsigned int tick;
};

class BeatStream :
    public std::list<BeatEvent*>
{
public:
    ~BeatStream() { std::for_each(begin(), end(), sndlib::util::DeleteObject()); }

     void tickToBeat(tick_t tick, BeatInfo* beat) const;
     tick_t BeatStream::measureToTick(unsigned int measure) const;
};

class EventStream :
    public std::list<Event*>
{
public:
    EventStream() : lastTick(0) {}
    ~EventStream() { std::for_each(begin(), end(), sndlib::util::DeleteObject()); }

    tick_t lastTick; // 一番最後のイベントのチック値
};

// １小節
struct Measure :
    public std::list<Event*>
{
    Measure() : isPattern(false) {}

    bool isEqual( const Measure& rhs ) const;

    bool isPattern;
};


// パターン
struct Pattern : public Measure
{
    Pattern(const char* label_) : label(label_) {}
    ~Pattern() { std::for_each(begin(), end(), sndlib::util::DeleteObject()); }

    std::string label;
};

// トラック単位の小節集合
struct SmfTrack :
    public std::vector<Measure>
{
public:
    SmfTrack( int measures ) : std::vector<Measure>( measures ) {}
    ~SmfTrack() {
        for( SmfTrack::iterator p = begin(); p != end(); ++p ) {
            std::for_each(p->begin(), p->end(), sndlib::util::DeleteObject());
        }
    }
};

// 小節パターンの集合
struct PatternArray :
    public std::vector<Pattern*>
{
    ~PatternArray() {  std::for_each(begin(), end(), sndlib::util::DeleteObject()); }
};

class SmfParser
{
public:
    SmfParser();
    virtual ~SmfParser();

    void parse( const char* filename, const CommandLineArg& cmdInfo,const std::string& labelPrefix );

private:
    typedef std::deque<NoteEvent*> NoteEventList;
    typedef std::map<int , NoteEventList > NoteEventMap;

    struct TrackStatus
    {
        tick_t tick;
        u8 running_status;
        bool end_of_track;
        NoteEventMap note_event_map[CHANNEL_MAX];

        u8 rpnLSB;
        u8 rpnMSB;
        u8 nrpnLSB;
        u8 nrpnMSB;
        bool rpn_flag; // true: rpn enabled, false: nrpn enabled

        TrackStatus();
        bool checkRestNoteOnEvent() const;
    };

    void parseHeader(sndlib::strm::fstream& in);
    void parseMidiEvent(sndlib::strm::fstream&in, u8 status, TrackStatus* track, const std::string& labelPrefix);
    void parseSysExEvent(sndlib::strm::fstream& in, u8 status, TrackStatus* track);
    void parseMetaEvent(sndlib::strm::fstream& in, TrackStatus* track, const CommandLineArg& cmdInfo, const std::string& labelPrefix );

    void eventNoteOn(int channel, int key, int velocity, TrackStatus* track);
    void eventNoteOff(int channel, int key, int velocity, TrackStatus* track);
    void eventControlChange(int channel, int control, int value, TrackStatus* track, const std::string& labelPrefix);
    void eventProgramChange(int channel, int program, TrackStatus* track);
    void eventPitchBend(int channel, int value, TrackStatus* track);

public:
    u16 mTracks;
    u16 mFormat;
    u16 mDivison;

    BeatStream mBeatStream;
    EventStream mMidiEventStream[ CHANNEL_MAX ];
};

#endif // !defined(AFX_SMFPARSER_H__C0E5CD18_72A1_4E19_9BBD_00A9C1EFEB2C__INCLUDED_)

