﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(AFX_SMFEVENT_H__5EF9E929_08AE_437C_879C_985FB122984E__INCLUDED_)
#define AFX_SMFEVENT_H__5EF9E929_08AE_437C_879C_985FB122984E__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

// #include <sndlib.h>
#include <sndlib/types.h>
#include <sndlib/stream.h>
#include <sndlib/utillity.h>

struct Pattern;


static const int CHANNEL_MAX = 16;

typedef sndlib::uint32_t tick_t;
typedef sndlib::uint8_t byte_t;

std::string convKey2String(int key);

class var_t
{
public:
    typedef unsigned long value_type;

    var_t() {}
    var_t(const value_type& rhs) : v(rhs) {}
    var_t& operator= (const value_type& rhs) { v = rhs; return *this; }
    operator value_type() const { return v; }
private:
    value_type v;
};

struct Event
{
    tick_t tick;

    Event( tick_t t ) : tick(t) {}
    virtual ~Event() {}

    virtual Event* clone() const = 0;
    virtual std::ostream& put(std::ostream& out) const = 0;
    virtual bool isEqual(const Event* rhs) const = 0;
    virtual bool isTextEvent() const { return false; } // true だと、put() で先頭に空白を入れない
    virtual bool isImportantEvent() const { return true; } // false のイベントのみなら、そのトラックは出力されない
    virtual bool isFixedEvent() const { return true; } // falseなら、ティックが後ろへずらされることがある
    virtual bool isMetaEvent() const { return false; }
    virtual void updateDuration( unsigned int in, unsigned int out) {}
};
struct MidiEvent : public Event
{
    MidiEvent(tick_t t) : Event(t) {}
};
struct MetaEvent : public Event
{
    MetaEvent( tick_t t ) : Event(t) {}

    virtual bool isMetaEvent() const { return true; }
};




struct BeatEvent : public MetaEvent
{
    BeatEvent(tick_t t, int len, const byte_t* data, int tpqn);
    void setNext(tick_t next);
    virtual std::ostream& put(std::ostream& out) const { return out; }
    virtual bool isEqual(const Event* rhs) const;
    virtual BeatEvent* clone() const { return new BeatEvent(*this); }

    unsigned char numerator; // 分子
    unsigned char denomirator; // 分母
    unsigned char click_interval;
    unsigned char midi_clock_duration;

    unsigned int measure; // 次の拍子イベントまでの小節数
    unsigned int division; // 単位音符分解能
};

struct TempoEvent : public MetaEvent
{
    TempoEvent(tick_t t, int len, const byte_t* data);
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual bool isFixedEvent() const { return false; }
    virtual TempoEvent* clone() const { return new TempoEvent(*this); }

    unsigned long tempo;
};

struct WaitEvent : public Event
{
    WaitEvent(tick_t t, unsigned long duration);

    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual WaitEvent* clone() const { return new WaitEvent(*this); }

    unsigned long duration;
};

struct TextEvent : public MetaEvent
{
    TextEvent(tick_t t, std::string text_);

    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual TextEvent* clone() const { return new TextEvent(*this); }
    virtual bool isTextEvent() const { return true; }

    std::string text;
};

struct TextAllEvent : public TextEvent
{
    TextAllEvent( tick_t t, std::string text_ ) : TextEvent( t, text_ ) {}

    virtual bool isImportantEvent() const { return false; }
};

struct PatternEvent : public Event
{
    PatternEvent(tick_t t, const struct Pattern* patt);

    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual PatternEvent* clone() const { return new PatternEvent(*this); }

    const struct Pattern* pattern;
};

struct NoteEvent : public MidiEvent
{
    NoteEvent(tick_t t, byte_t key, byte_t velocity);
    void noteOff(tick_t t);
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual NoteEvent* clone() const { return new NoteEvent(*this); }
    virtual void updateDuration( unsigned int in, unsigned int out);

    int key;
    int velocity;
    tick_t duration;

};

struct ProgramChangeEvent : public MidiEvent
{
    ProgramChangeEvent(tick_t t, byte_t program);
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual bool isFixedEvent() const { return false; }
    virtual ProgramChangeEvent* clone() const { return new ProgramChangeEvent(*this); }

    int program;
};

struct PitchBendEvent : public MidiEvent
{
    PitchBendEvent(tick_t t, int bend);
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual bool isFixedEvent() const { return false; }
    virtual PitchBendEvent* clone() const { return new PitchBendEvent(*this); }

    int bend;
};

struct ControlChangeEvent : public MidiEvent
{
    ControlChangeEvent(tick_t t, byte_t control, byte_t value, const std::string& labelPrefix );
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual bool isFixedEvent() const { return false; }
    virtual ControlChangeEvent* clone() const { return new ControlChangeEvent(*this); }

    int control;
    int value;
    std::string labelPrefix;
};

struct RpnNrpnEvent : public MidiEvent
{
    enum Type
    {
        RPN,
        NRPN
    };
    RpnNrpnEvent(tick_t t, Type type, byte_t msb, byte_t lsb, byte_t value );
    virtual std::ostream& put(std::ostream& out) const;
    virtual bool isEqual(const Event* rhs) const;
    virtual bool isFixedEvent() const { return false; }
    virtual RpnNrpnEvent* clone() const { return new RpnNrpnEvent(*this); }

    Type type;
    int msb;
    int lsb;
    int value;
};



inline std::ostream& operator<<(std::ostream& s, const Event& event)
{
    if (! event.isTextEvent() ) s << "    ";
    return event.put(s);
}

sndlib::strm::fstream& operator>>(sndlib::strm::fstream& s, var_t& a);



#endif // !defined(AFX_SMFEVENT_H__5EF9E929_08AE_437C_879C_985FB122984E__INCLUDED_)

