﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker;

    public static class SoundPlayer
    {
        public static string SlashToBackSlash(string path)
        {
            string newPath = path.Replace("/", "\\");

            return newPath;
        }

        private static string CommonRoot
        {
            get
            {
                string result = Environment.GetEnvironmentVariable("CAFE_ROOT");
                if (null == result)
                {
                    throw new DirectoryNotFoundException();
                }

                return result;
            }
        }


        ///--------------------------------
        /// <summary>
        /// サウンドプレイヤーの起動
        /// </summary>
        public static void Start(string dvdRootPath, SoundProjectDocument projectDocument, string soundPlayerDirectory)
        {
            if (null == projectDocument) { throw new ArgumentNullException("projectDocument"); }

            // SoundPlayerの起動
            Cursor oldCursor = Cursor.Current;

            try
            {
                Cursor.Current = Cursors.WaitCursor;

                RunSoundPlayer(dvdRootPath, projectDocument, soundPlayerDirectory);
            }
            finally
            {
                Cursor.Current = oldCursor;
            }
        }

        ///--------------------------------
        /// <summary>
        /// サウンドプレイヤーの停止
        /// </summary>
        public static void Stop(string soundPlayerDirectory)
        {
            Cursor oldCursor = Cursor.Current;

            try {
                Cursor.Current = Cursors.WaitCursor;

                ShutdownSoundPlayer(soundPlayerDirectory);
            }
            finally {
                Cursor.Current = oldCursor;
            }
        }

        ///--------------------------------
        /// <summary>
        /// SoundPlayerの起動
        /// </summary>
        private static void RunSoundPlayer(string dvdRootPath, SoundProjectDocument projectDocument, string soundPlayerDirectoryPath)
        {
            if ( null == dvdRootPath ) { throw new ArgumentNullException( "dvdRootPath" ); }
            if ( null == projectDocument ) { throw new ArgumentNullException( "projectDocument" ); }

            if( Directory.Exists(soundPlayerDirectoryPath) == false )
            {
                return;
            }
            if( CommonRoot == null )
            {
                return;
            }

            string cafeRun = Path.Combine(soundPlayerDirectoryPath, "CafeRun.bat");

            /******************************************************************************
               コマンドライン引数の作成
            ******************************************************************************/
            string strElfFilePath = Path.Combine(soundPlayerDirectoryPath, "SoundPlayer.elf");

#if false   // TODO : 一時的な DvdRoot が設定できるようになるまで無効化します。
            string strDdfFilePath = Path.Combine( soundPlayerDirectoryPath, "SoundPlayer.ddf" );

            /******************************************************************************
                DDFファイルの書き出し
            ******************************************************************************/
            using ( StreamWriter sw = File.CreateText( strDdfFilePath ) ) {
                sw.WriteLine( "[Input]" );
                sw.WriteLine( "DvdRoot=\"" + dvdRootPath + "\"" );
            }
#endif

            /******************************************************************************
                ProcessStartInfoの作成
            ******************************************************************************/
            ProcessStartInfo startInfo = new ProcessStartInfo(cafeRun);

#if false   // TODO : 一時的な DvdRoot が設定できるようになるまで無効化します。
            startInfo.Arguments =
                "-enf -d=\"" + strDdfFilePath + "\" " +
                "\"" + strElfFilePath + "\" " +
                "-a /" + Path.GetFileNameWithoutExtension( projectDocument.Resource.Key ) + ".brsar";
            startInfo.WorkingDirectory = dvdRootPath;
#else
            startInfo.WorkingDirectory = Path.GetDirectoryName(cafeRun);
#endif

            /******************************************************************************
               プロセスの起動
            ******************************************************************************/
            Process soundPlayer = Process.Start( startInfo );

#if false   // HACK : cmd.exe / caferun からは即時に制御が返らないため、Wait しないようにします。
            soundPlayer.WaitForExit();
#endif
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static void ShutdownSoundPlayer(string soundPlayerDirectoryPath)
        {
            if (Directory.Exists(soundPlayerDirectoryPath) == false)
            {
                return;
            }
            if (CommonRoot == null)
            {
                return;
            }

            string cafestop = Path.Combine(soundPlayerDirectoryPath, "CafeStop.bat");
            ProcessStartInfo startInfo = new ProcessStartInfo(cafestop);
            startInfo.WorkingDirectory = Path.GetDirectoryName(cafestop);

            /******************************************************************************
               stop
            ******************************************************************************/
            Process OdStop = Process.Start( startInfo );
            OdStop.WaitForExit();
        }
    }
}
