﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.WaveFileFormat
{
    class WaveFileReaderException : WaveFileFormatException
    {
        public WaveFileReaderException(string s) : base(s) { }
    }

    public abstract class WaveFileReader
    {
        public static WaveFileReader CreateInstance(string filePath)
        {
            Format format = GuessFileFormat(filePath);

            switch (format)
            {
                case Format.WAV:
                    return new WaveFileWavReader();
                case Format.AIFF:
                    return new WaveFileAiffReader();
                default:
                    throw new WaveFileReaderException("Not supported file format \"" + filePath + "\"");
            }
        }

        enum Format
        {
            UnSupported,
            AIFF,
            WAV,
        };

        static Format GuessFileFormat(string filePath)
        {
            using (FileStream strm = File.OpenRead(filePath))
            {
                BinaryReader reader = new BinaryReader(strm);

                string fileChunk = new string(reader.ReadChars(4));
                reader.ReadInt32(); // skip
                string format = new string(reader.ReadChars(4));

                if (fileChunk == "RIFF")
                {
                    if (format == "WAVE") return Format.WAV;
                }
                else if (fileChunk == "FORM")
                {
                    if (format == "AIFF") return Format.AIFF;
                }
            }

            return Format.UnSupported;
        }

        public abstract WaveFile Open(string filePath);
        public abstract void Close();

        public abstract Int16[] ReadInt16(long samples);
    }

    class WaveFileImpl
    {
        public delegate Int16[] ToInt16Handler(byte[] buffer);

        public static Int16[] S8ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 1, i++)
            {
                array[i] = (Int16)((sbyte)buffer[pos] << 8);
            }

            return array;
        }
        public static Int16[] U8ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 1, i++)
            {
                array[i] = (Int16)(((short)buffer[pos] - 0x80) << 8);
            }

            return array;
        }
        public static Int16[] BigS16ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 2];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 2, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS16ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 2];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 2, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 1] << 8) | buffer[pos]);
            }

            return array;
        }
        public static Int16[] BigS24ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 3];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 3, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS24ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 3];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 3, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 2] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] BigS32ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 4];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 4, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos] << 8) | buffer[pos + 1]);
            }

            return array;
        }
        public static Int16[] LittleS32ToInt16(byte[] buffer)
        {
            Int16[] array = new Int16[buffer.Length / 4];
            for (int pos = 0, i = 0; pos < buffer.Length; pos += 4, i++)
            {
                array[i] = (Int16)(((sbyte)buffer[pos + 3] << 8) | buffer[pos + 2]);
            }

            return array;
        }
    }

}
