﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.WaveFileFormat
{

    public class WaveFileAiff : WaveFile
    {
        AiffCommonChunk commonChunk = new AiffCommonChunk();
        AiffInstChunk instChunk = new AiffInstChunk();
        AiffSoundDataChunk soundDataChunk = new AiffSoundDataChunk();
        AiffMarkerChunk markerChunk = new AiffMarkerChunk();

        public AiffCommonChunk CommonChunk
        {
            set { commonChunk = value; }
            get { return commonChunk; }
        }
        public AiffInstChunk InstChunk
        {
            set { instChunk = value; }
            get { return instChunk; }
        }
        public AiffSoundDataChunk SoundDataChunk
        {
            set { soundDataChunk = value; }
            get { return soundDataChunk; }
        }
        public AiffMarkerChunk MarkerChunk
        {
            set { markerChunk = value; }
            get { return markerChunk; }
        }

        public class AiffCommonChunk
        {
            public Int16 numChannels = 1;
            public UInt32 numSampleFrames = 0;
            public Int16 sampleSize = 16;
            public double sampleRate = 32000.0;

            public void Read(BinaryReader reader)
            {
                numChannels = reader.ReadInt16();
                numSampleFrames = reader.ReadUInt32();
                sampleSize = reader.ReadInt16();
                UInt16 head = reader.ReadUInt16();
                UInt32 hiMant = reader.ReadUInt32();
                UInt32 loMant = reader.ReadUInt32();

                IeeeExtended ie = new IeeeExtended(head, hiMant, loMant);
                sampleRate = ie.ToDouble();
            }
        }

        public class AiffInstChunk
        {
            public byte baseNote = 60;
            public byte detune = 0;
            public byte lowNote = 0;
            public byte highNote = 127;
            public byte lowVelocity = 0;
            public byte highVelocity = 127;
            public Int16 gain = 0;
            public AiffLoop sustainLoop = new AiffLoop();
            public AiffLoop releaseLoop = new AiffLoop();

            public void Read(BinaryReader reader)
            {
                baseNote = reader.ReadByte();
                detune = reader.ReadByte();
                lowNote = reader.ReadByte();
                highNote = reader.ReadByte();
                lowVelocity = reader.ReadByte();
                highVelocity = reader.ReadByte();
                gain = reader.ReadInt16();
                sustainLoop.Read(reader);
                releaseLoop.Read(reader);
            }
        }

        public class AiffSoundDataChunk
        {
            public UInt32 dataOffset = 0;
            public UInt32 dataBlockSize = 0;

            public void Read(BinaryReader reader)
            {
                dataOffset = reader.ReadUInt32();
                dataBlockSize = reader.ReadUInt32();
            }
        }

        public class AiffLoop
        {
            public enum Mode : short
            {
                NoLoop = 0,
                Forward = 1,
                ForwardBackward = 2,
            };

            public Mode playMode = Mode.NoLoop;
            public Int16 beginLoop = 0;
            public Int16 endLoop = 0;

            public void Read(BinaryReader reader)
            {
                playMode = (AiffLoop.Mode)reader.ReadInt16();
                beginLoop = reader.ReadInt16();
                endLoop = reader.ReadInt16();
            }
        }

        public class AiffMarker
        {
            public UInt32 Position;
            public string Name;
        }

        public class AiffMarkerChunk
        {
            SortedList list = new SortedList();

            public AiffMarker GetMarker(Int16 id)
            {
                return (AiffMarker)list[id];
            }

            public void Read(BinaryReader reader)
            {
                list.Clear();

                UInt16 numMakers = reader.ReadUInt16();
                for (int i = 0; i < numMakers; i++)
                {
                    Int16 id = reader.ReadInt16();
                    UInt32 position = reader.ReadUInt32();
                    Byte nameLen = reader.ReadByte();

                    AiffMarker marker = new AiffMarker();
                    marker.Position = position;
                    marker.Name = new string(reader.ReadChars(nameLen));

                    list.Add(id, marker);

                    reader.BaseStream.Position = ((reader.BaseStream.Position + 0x01) & ~0x01);
                }
            }
        }

        public override bool IsLoop
        {
            get
            {
                if (instChunk.sustainLoop.playMode == AiffLoop.Mode.NoLoop) return false;
                AiffMarker marker = markerChunk.GetMarker(instChunk.sustainLoop.beginLoop);
                if (marker == null) return false;
                marker = markerChunk.GetMarker(instChunk.sustainLoop.endLoop);
                if (marker == null) return false;
                return true;
            }
        }
        public override long LoopStartFrame
        {
            set
            {
                if (instChunk.sustainLoop.playMode == AiffLoop.Mode.NoLoop) return;
                Int16 markID = instChunk.sustainLoop.beginLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                marker.Position = (uint)value;
            }
            get
            {
                if (instChunk.sustainLoop.playMode == AiffLoop.Mode.NoLoop) return 0;
                Int16 markID = instChunk.sustainLoop.beginLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                return marker.Position;
            }
        }
        public override long LoopEndFrame
        {
            set
            {
                if (instChunk.sustainLoop.playMode == AiffLoop.Mode.NoLoop) return;
                Int16 markID = instChunk.sustainLoop.endLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                marker.Position = (uint)value;
            }
            get
            {
                if (instChunk.sustainLoop.playMode == AiffLoop.Mode.NoLoop) return 0;
                Int16 markID = instChunk.sustainLoop.endLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                return marker.Position;
            }
        }

        public override int ChannelCount
        {
            set
            {
                commonChunk.numChannels = (short)value;
            }
            get
            {
                return commonChunk.numChannels;
            }
        }
        public override long FrameCount
        {
            set
            {
                commonChunk.numSampleFrames = (uint)value;
            }
            get
            {
                return commonChunk.numSampleFrames;
            }
        }
        public override int SampleBit
        {
            set
            {
                commonChunk.sampleSize = (short)value;
            }
            get
            {
                return commonChunk.sampleSize;
            }
        }
        public override int SampleRate
        {
            set
            {
                commonChunk.sampleRate = value;
            }
            get
            {
                return (int)commonChunk.sampleRate;
            }
        }
        public override int OriginalKey
        {
            set
            {
                instChunk.baseNote = (byte)value;
            }
            get
            {
                return (int)instChunk.baseNote;
            }
        }
    }

    class IeeeExtended
    {
        UInt16 head;
        UInt32 hiMant;
        UInt32 loMant;

        public IeeeExtended(UInt16 h, UInt32 hi, UInt32 lo)
        {
            head = h;
            hiMant = hi;
            loMant = lo;
        }

        public double ToDouble()
        {
            double f;
            int expon;

            expon = head & 0x7fff;

            if (expon == 0 && hiMant == 0 && loMant == 0)
            {
                f = 0;
            }
            else
            {
                if (expon == 0x7FFF)
                {    /* Infinity or NaN */
                    f = Double.MaxValue;
                }
                else
                {
                    expon -= 0x3fff;

                    f = hiMant * Math.Pow(2.0, (expon -= 31));
                    f += loMant * Math.Pow(2.0, (expon -= 32));
                }
            }

            if ((head & 0x8000) != 0)
                return -f;
            else
                return f;
        }
    }

}

