﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rXmlStrmSoundStrm
    {
        public class Track
        {
            int index = -1;
            string filePath;
            string fullPath;
            Nw4rSoundSetStrmFileFormat format;
            int volume = 127;
            int pan = 64;

            public Nw4rSoundSetStrmFileFormat FileFormat
            {
                set { format = value; }
                get { return format; }
            }
            public string FilePath
            {
                set { filePath = value; }
                get { return filePath; }
            }
            public string FullPath
            {
                set { fullPath = value; }
                get { return fullPath; }
            }
            public void SetFilePath(string filePath, string soundSetFilePath)
            {
                this.filePath = filePath;
                this.fullPath = Path.Combine(Path.GetDirectoryName(soundSetFilePath), filePath);
            }
            public int Index
            {
                set { index = value; }
                get { return index; }
            }
            public int Volume
            {
                set { volume = value; }
                get { return volume; }
            }
            public int Pan
            {
                set { pan = value; }
                get { return pan; }
            }
            public bool Equals(Track track)
            {
                if (!index.Equals(track.index)) return false;
                if (!NPath.Normalize(fullPath).Equals(NPath.Normalize(track.fullPath))) return false;
                if (!format.Equals(track.format)) return false;
                if (!volume.Equals(track.volume)) return false;
                if (!pan.Equals(track.pan)) return false;
                return true;
            }
        }

        string label;
        List<Track> trackList = new List<Track>();

        public string Label
        {
            set { label = value; }
            get { return label; }
        }
        public List<Track> TrackList
        {
            get { return trackList; }
        }

        public static int CompareTrackByIndex(Track a, Track b)
        {
            return a.Index - b.Index;
        }

        public Nw4rXmlStrmSoundStrm(string label)
        {
            this.label = label;
        }

        public bool Equals(Nw4rXmlStrmSoundStrm strm)
        {
            if (TrackList.Count != strm.TrackList.Count) return false;
            for (int i = 0; i < TrackList.Count; i++)
            {
                if (!TrackList[i].Equals(strm.TrackList[i])) return false;
            }
            return true;
        }

        internal static Nw4rXmlStrmSoundStrm Parse(XmlDocument doc, XmlElement strmSoundElem, string strmFilePath)
        {
            XmlElement elem; // テンポラリ用

            if (!strmSoundElem.HasAttribute("name"))
            {
                // TODO: throw exception
                return null;
            }

            string label = strmSoundElem.GetAttribute("name");
            Nw4rXmlStrmSoundStrm strm = new Nw4rXmlStrmSoundStrm(label);

            XmlElement trackArrayElem = (XmlElement)strmSoundElem.SelectSingleNode("strm_track_array");
            if (trackArrayElem != null)
            {
                foreach (XmlElement strmTrackElem in trackArrayElem.SelectNodes("strm_track"))
                {
                    Nw4rXmlStrmSoundStrm.Track track = new Nw4rXmlStrmSoundStrm.Track();
                    if (strmTrackElem.HasAttribute("index"))
                    {
                        track.Index = Int32.Parse(strmTrackElem.GetAttribute("index"));
                    }

                    elem = (XmlElement)strmTrackElem.SelectSingleNode("file");
                    if (elem != null)
                    {
                        if (elem.HasAttribute("path"))
                        {
                            track.SetFilePath(elem.GetAttribute("path"), strmFilePath);
                            track.FileFormat = Nw4rSoundSetStrmFileFormat.Parse(elem.GetAttribute("encode"));
                        }
                    }
                    elem = (XmlElement)strmTrackElem.SelectSingleNode("volume");
                    if (elem != null)
                    {
                        track.Volume = Int32.Parse(elem.InnerText);
                    }
                    elem = (XmlElement)strmTrackElem.SelectSingleNode("pan");
                    if (elem != null)
                    {
                        track.Pan = Int32.Parse(elem.InnerText);
                    }
                    strm.TrackList.Add(track);
                }
            }
            else
            {
                elem = (XmlElement)strmSoundElem.SelectSingleNode("file");
                if (elem != null)
                {
                    if (elem.HasAttribute("path"))
                    {
                        Nw4rXmlStrmSoundStrm.Track track = new Nw4rXmlStrmSoundStrm.Track();
                        track.SetFilePath(elem.GetAttribute("path"), strmFilePath);
                        track.FileFormat = Nw4rSoundSetStrmFileFormat.Parse(elem.GetAttribute("encode"));
                        strm.TrackList.Add(track);
                    }
                }
            }
            return strm;
        }

        internal XmlElement ToXmlElement(XmlDocument doc, string strmFilePath)
        {
            XmlElement elem; // テンポラリ用

            XmlElement strmElem = doc.CreateElement("strm_sound");
            strmElem.SetAttribute("name", this.Label);

            XmlElement trackArrayElem = doc.CreateElement("strm_track_array");
            foreach (Track track in trackList)
            {
                track.FilePath = NPath.MakeRelative(track.FullPath, Path.GetDirectoryName(strmFilePath));
                XmlElement trackElem = doc.CreateElement("strm_track");
                if (track.Index >= 0)
                {
                    trackElem.SetAttribute("index", track.Index.ToString());
                }

                if (track.FilePath != null)
                {
                    elem = doc.CreateElement("file");
                    elem.SetAttribute("path", track.FilePath);
                    elem.SetAttribute("encode", track.FileFormat.ToString());
                    trackElem.AppendChild(elem);

                    elem = doc.CreateElement("volume");
                    elem.InnerText = track.Volume.ToString();
                    trackElem.AppendChild(elem);

                    elem = doc.CreateElement("pan");
                    elem.InnerText = track.Pan.ToString();
                    trackElem.AppendChild(elem);
                }

                trackArrayElem.AppendChild(trackElem);
            }
            strmElem.AppendChild(trackArrayElem);

            return strmElem;
        }
    }

    public class Nw4rXmlStrmSound : Nw4rXmlFile
    {
        Nw4rXmlStrmSoundStrm strmSoundData;

        public Nw4rXmlStrmSoundStrm StrmSoundData
        {
            set { strmSoundData = value; }
            get { return strmSoundData; }
        }

        protected override string FileTitle { get { return "NintendoWare Stream Sound"; } }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement strmSoundElem = (XmlElement)bodyElem.SelectSingleNode("strm_sound");
            if (strmSoundElem != null)
            {
                strmSoundData = Nw4rXmlStrmSoundStrm.Parse(doc, strmSoundElem, this.FilePath);
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement strmSoundElem = strmSoundData.ToXmlElement(doc, this.FilePath);

            bodyElem.AppendChild(strmSoundElem);
        }
    }

}
