﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml;
using System.Text;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public struct Nw4rDecayCurve
    {
        public const int Log = 0;
        public const int Linear = 1;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Log, "log" ),
            new KeyValuePair<int, string>( Linear, "linear" ),
        };

        public static implicit operator int(Nw4rDecayCurve s) { return s.val_; }
        public static implicit operator Nw4rDecayCurve(int s) { return new Nw4rDecayCurve(s); }
        public static Nw4rDecayCurve Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rDecayCurve(int e) { val_ = e; }
        private int val_;
    }

    public class Nw4rSound3DParam
    {
        public bool VolumeCtrl = true;
        public bool PanCtrl = true;
        public bool SurroundPanCtrl = true;
        public bool PriorityCtrl = true;
        public bool FilterCtrl = false;
        public Nw4rDecayCurve DecayCurve = Nw4rDecayCurve.Log;
        public double DecayRatio = 0.5;
        public int DopplerFactor = 0;

        internal static Nw4rSound3DParam Parse(XmlDocument doc, XmlElement rootElem)
        {
            Nw4rSound3DParam param = new Nw4rSound3DParam();

            XmlElement flagElem = (XmlElement)rootElem.SelectSingleNode("flags");
            if (flagElem != null)
            {
                {
                    string attr = flagElem.GetAttribute("volume_ctrl");
                    if (attr == "on") param.VolumeCtrl = true;
                    else if (attr == "off") param.VolumeCtrl = false;
                }
                {
                    string attr = flagElem.GetAttribute("pan_ctrl");
                    if (attr == "on") param.PanCtrl = true;
                    else if (attr == "off") param.PanCtrl = false;
                }
                {
                    string attr = flagElem.GetAttribute("surround_pan_ctrl");
                    if (attr == "on") param.SurroundPanCtrl = true;
                    else if (attr == "off") param.SurroundPanCtrl = false;
                }
                {
                    string attr = flagElem.GetAttribute("priority_ctrl");
                    if (attr == "on") param.PriorityCtrl = true;
                    else if (attr == "off") param.PriorityCtrl = false;
                }
                {
                    string attr = flagElem.GetAttribute("filter_ctrl");
                    if (attr == "on") param.FilterCtrl = true;
                    else if (attr == "off") param.FilterCtrl = false;
                }
            }

            XmlElement decayElem = (XmlElement)rootElem.SelectSingleNode("decay");
            if (decayElem != null)
            {
                {
                    string attr = decayElem.GetAttribute("curve");
                    param.DecayCurve = Nw4rDecayCurve.Parse(attr);
                }
                if (decayElem.HasAttribute("ratio"))
                {
                    string attr = decayElem.GetAttribute("ratio");
                    param.DecayRatio = Double.Parse(attr);
                }
            }

            XmlElement dopplerFactorElem = (XmlElement)rootElem.SelectSingleNode("doppler_factor");
            if (dopplerFactorElem != null)
            {
                param.DopplerFactor = Int32.Parse(dopplerFactorElem.InnerText);
            }

            return param;
        }
        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement rootElem = doc.CreateElement("sound3d_param");

            {
                XmlElement flagElem = doc.CreateElement("flags");
                flagElem.SetAttribute("volume_ctrl", VolumeCtrl ? "on" : "off");
                flagElem.SetAttribute("pan_ctrl", PanCtrl ? "on" : "off");
                flagElem.SetAttribute("surround_pan_ctrl", SurroundPanCtrl ? "on" : "off");
                flagElem.SetAttribute("priority_ctrl", PriorityCtrl ? "on" : "off");
                flagElem.SetAttribute("filter_ctrl", FilterCtrl ? "on" : "off");
                rootElem.AppendChild(flagElem);
            }
            {
                XmlElement decayElem = doc.CreateElement("decay");
                decayElem.SetAttribute("curve", DecayCurve.ToString());
                decayElem.SetAttribute("ratio", DecayRatio.ToString());
                rootElem.AppendChild(decayElem);
            }
            {
                XmlElement dopplerFactorElem = doc.CreateElement("doppler_factor");
                dopplerFactorElem.InnerText = DopplerFactor.ToString();
                rootElem.AppendChild(dopplerFactorElem);
            }

            return rootElem;
        }
    }

}
