﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    /// <summary>
    /// サウンドアーカイブコンバータ
    /// </summary>
    public class Nw4rSoundArchiveConverter
    {
        #region ** フィールド

        // サウンドアーカイブモデル
        private Nw4rSoundArchive _soundArchive = null;

        // コンバートコンポーネント
        private Nw4rSoundFileManager _fileManager = null;
        private Nw4rSoundArchiveConverterSettings _settings = new Nw4rSoundArchiveConverterSettings();

        // コンバートコマンド
        private Dictionary<string, Nw4rConvertCommand> _commands = new Dictionary<string, Nw4rConvertCommand>();
        private Nw4rConvertCommandProcessor _convertProcessor = new Nw4rConvertCommandProcessor();
        private Util.ProcessHandler _processHandler = null;

        // 状態
        private Stopwatch _convertTime = new Stopwatch();	// コンバート時間計測用ストップウォッチ
        private bool _canceled = false;				// キャンセル状態
        private object _cancelLock = new object();		// キャンセルロック

        #endregion

        public Nw4rSoundArchiveConverter(Util.ProcessHandler processHandler)
        {
            _processHandler = processHandler;

            if (null != _processHandler)
            {
                _processHandler.Killed += OnProcessKilled;
            }
        }

        #region ** プロパティ

        public Nw4rSoundArchiveConverterSettings Settings
        {
            get { return _settings; }
            set { _settings = value; }
        }

        private string DependenciesFilePath
        {
            get
            {
                if (null == _soundArchive) { return string.Empty; }
                return Path.ChangeExtension(_soundArchive.ProjectFilePath, ".depend");
            }
        }

        private Nw4rSoundArchiveConvertWarningFlags WarningFlags
        {
            get
            {
                if (null == _soundArchive) { return Nw4rSoundArchiveConvertWarningFlags.NoWarning; }
                return _soundArchive.ConvertSettings.WarningFlags;
            }
        }

        #endregion

        #region ** イベント

        public event Nw4rConverterOutputTextEventHandler OutputLine;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnOutputLine(Nw4rConverterOutputLineEventArgs e)
        {
            if (null == OutputLine) { return; }

            // サイレント設定の場合、「情報」以下は出力しない
            if (_settings.Silent)
            {
                if (e.Line.Level < LineLevel.Information) { return; }
            }

            OutputLine(this, e);
        }

        private void OnConverterOutputLine(object sender, Nw4rCommandOutputLineEventArgs e)
        {
            OutputLineInternal(e.OutputInformation);
        }

        private void OnProcessKilled(object sender, EventArgs e)
        {
            try
            {
                Monitor.Enter(_cancelLock);

                _canceled = true;
                _convertProcessor.End();
            }
            finally
            {
                Monitor.Exit(_cancelLock);
            }
        }

        #endregion

        #region ** メソッド

        #region ** コンバート開始終了ログ出力

        public void OutputBeginConvertLog()
        {
            _convertTime.Start();

            OutputLineInternal(new InformationLine(LineLevel.Level1,
                                                     new BeginConvertMessage(_soundArchive.ProjectName).Text));
        }

        public void OutputEndConvertLog(bool convertSucceeded)
        {
            _convertTime.Stop();

            if (_canceled)
            {
                OutputCanceledConvertLog();
                return;
            }

            if (convertSucceeded)
            {
                OutputLineInternal(new InformationLine(LineLevel.Level1,
                    new EndConvertSucceededMessage(_soundArchive.ProjectName, _convertTime.ElapsedMilliseconds).Text));
            }
            else
            {
                OutputLineInternal(new InformationLine(LineLevel.Level1,
                                                    new EndConvertFailedMessage(_soundArchive.ProjectName).Text));
            }
        }

        public void OutputCanceledConvertLog()
        {
            _convertTime.Stop();

            OutputLineInternal(new InformationLine(LineLevel.Level1,
                                            new EndConvertCanceledMessage(_soundArchive.ProjectName).Text));
        }

        #endregion

        #region ** プロジェクトのロード

        public void LoadProject(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            try
            {

                _commands.Clear();

                _soundArchive = new Nw4rSoundArchive();
                _soundArchive.LoadProject(filePath);

                _fileManager = new Nw4rSoundFileManager(DependenciesFilePath, _soundArchive.IntermediateDirectoryPath);
                _fileManager.AggregateOutput = _soundArchive.AggregateIntermediateOutput;

            }
            catch (Nw4rFileFormatException exception)
            {

                Debug.WriteLine(exception.ToString());
                OutputLineInternal(new ErrorLine(exception.Message, GetTargetItems(exception.OwnerLabel)));

                throw exception;

            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception.ToString());
                OnOutputLine(new Nw4rConverterOutputLineEventArgs(new ErrorLine(exception.Message)));
                throw exception;
            }
        }

        #endregion

        #region ** コンバート

        public void Convert()
        {
            Convert(true);
        }

        public void Convert(bool update)
        {
            if (null == _soundArchive) { throw new Nw4rProjectNotLoadedException(); }

            try
            {

                if (!update)
                {
                    _fileManager.Clean();
                }

                // キャッシュマップファイルをロードする
                _fileManager.Load();

                // 名前解決
                Nw4rFileFormatExceptionList results = new Nw4rFileFormatExceptionList();
                _soundArchive.ResolveName(results);

                if (0 < results.Count)
                {
                    throw new Nw4rFileFormatException(results);
                }

                // 未参照アイテムの警告
                if (0 != (WarningFlags & Nw4rSoundArchiveConvertWarningFlags.WarnUnreferencedItem))
                {
                    WarnUnreferencedItems();
                }

                // グループ未登録アイテムの警告
                if (0 != (WarningFlags & Nw4rSoundArchiveConvertWarningFlags.WarnIndependentItem))
                {
                    WarningIndependentItems();
                }

                // コンバート
                PerformConvert(update);

                // キャッシュマップファイルを出力する
                _fileManager.Save();

            }
            catch (Nw4rFileFormatException exception)
            {

                if (null == exception.Exceptions)
                {
                    Debug.WriteLine(exception.ToString());
                    OutputLineInternal(new ErrorLine(exception.Message, GetTargetItems(exception.OwnerLabel)));
                }
                else
                {
                    foreach (Nw4rFileFormatException child in exception.Exceptions)
                    {
                        Debug.WriteLine(child.ToString());
                        OutputLineInternal(new ErrorLine(child.Message, GetTargetItems(child.OwnerLabel)));
                    }
                }

                throw exception;

            }
            catch (Exception exception)
            {

                if (!_canceled)
                {
                    Debug.WriteLine(exception.ToString());
                    OnOutputLine(new Nw4rConverterOutputLineEventArgs(new ErrorLine(exception.Message)));
                    throw exception;
                }

            }
        }

        public void ReConvert()
        {
            Convert(false);
        }

        private void WarnUnreferencedItems()
        {
            // 未参照のバンク
            foreach (Nw4rBank bank in _soundArchive.AllBanks)
            {
                if (_soundArchive.ReferencedBanks.Contains(bank.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningUnreferencedItem(bank.Label).Text, bank));
            }

            // 未参照のプレイヤー
            foreach (Nw4rPlayer player in _soundArchive.AllPlayers)
            {
                if (_soundArchive.ReferencedPlayers.Contains(player.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningUnreferencedItem(player.Label).Text, player));
            }
        }

        private void WarningIndependentItems()
        {
            // グループに登録済みのアイテムを抽出する
            Nw4rSoundSetItemCollection groupingItems = new Nw4rSoundSetItemCollection();

            foreach (Nw4rGroup group in _soundArchive.AllGroups)
            {
                foreach (Model.Nw4rSoundSetItem dependItem in group.AllDependItems)
                {

                    if (groupingItems.Contains(dependItem.Key)) { continue; }
                    groupingItems.Add(dependItem);

                }
            }

            foreach (Nw4rGroup group in _soundArchive.AllInnerGroups)
            {
                foreach (Model.Nw4rSoundSetItem dependItem in group.AllDependItems)
                {

                    if (groupingItems.Contains(dependItem.Key)) { continue; }
                    groupingItems.Add(dependItem);

                }
            }


            // 未登録のアイテムを警告
            // ウェーブサウンドセット
            foreach (Nw4rWaveSoundSet wsdSet in _soundArchive.AllWaveSoundSets)
            {
                if (groupingItems.Contains(wsdSet.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningIndependentItem(wsdSet.Label).Text, wsdSet));
            }

            // シーケンスサウンドセット
            foreach (Nw4rSequenceSoundSet seqSoundSet in _soundArchive.AllSequenceSoundSets)
            {
                if (groupingItems.Contains(seqSoundSet.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningIndependentItem(seqSoundSet.Label).Text, seqSoundSet));
            }

            // シーケンスサウンド
            foreach (Nw4rSequenceSound seqSound in _soundArchive.AllSequenceSounds)
            {
                if (seqSound.Parent is Nw4rSequenceSoundSet) { continue; }
                if (groupingItems.Contains(seqSound.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningIndependentItem(seqSound.Label).Text, seqSound));
            }

            // バンク
            foreach (Nw4rBank bank in _soundArchive.AllBanks)
            {
                if (groupingItems.Contains(bank.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningIndependentItem(bank.Label).Text, bank));
            }

            // インナーグループ
            foreach (Nw4rGroup group in _soundArchive.AllInnerGroups)
            {
                if (_soundArchive.ReferencedInnerGroups.Contains(group.Key)) { continue; }
                OutputLineInternal(new WarningLine(new WarningIndependentItem(group.Label).Text, group));
            }
        }

        private void PerformConvert(bool update)
        {
            try
            {

                // ストリームサウンド
                foreach (Nw4rStreamSound sound in _soundArchive.AllStreamSounds)
                {

                    if (_canceled) { return; }

                    CreateStreamSoundIntermediateFile(sound, update);	    // XML中間ファイルの作成
                    CreateStreamSoundFile(sound);						    // バイナリファイルの作成
                    sound.PrepareConvert();								    // サウンドアーカイブのコンバート準備
                    CreateStreamSoundConvertCommand(sound, update);	    // コンバートコマンドの作成

                }

                // ウェーブサウンド（ウェーブサウンドセット内も含む）
                foreach (Nw4rWaveSound sound in _soundArchive.AllWaveSounds)
                {

                    if (_canceled) { return; }

                    CreateWaveSoundFile(sound);						    // バイナリファイルの作成
                    sound.PrepareConvert();								    // サウンドアーカイブのコンバート準備
                    CreateWaveSoundConvertCommand(sound, update);		    // コンバートコマンドの作成

                }

                // ウェーブサウンドセット
                foreach (Nw4rWaveSoundSet wsdSet in _soundArchive.AllWaveSoundSets)
                {

                    if (_canceled) { return; }

                    CreateWaveSoundSetFile(wsdSet);					    // バイナリファイルの作成
                    wsdSet.PrepareConvert();							    // サウンドアーカイブのコンバート準備
                    CreateWaveSoundSetConvertCommand(wsdSet, update);	    // コンバートコマンドの作成

                }

                // バンク
                foreach (Nw4rBank bank in _soundArchive.AllBanks)
                {

                    if (_canceled) { return; }

                    CreateBankFile(bank);								    // バイナリファイルの作成
                    bank.PrepareConvert();								    // サウンドアーカイブのコンバート準備
                    CreateBankConvertCommand(bank, update);			    // コンバートコマンドの作成

                }

                // シーケンスサウンドはバンクを参照する可能性があるため、
                // これまでに登録されたコンバートコマンドを処理する
                PerformConvert(_convertProcessor);

                // シーケンスサウンド（シーケンスサウンドセット内も含む）
                foreach (Nw4rSequenceSound sound in _soundArchive.AllSequenceSounds)
                {

                    if (_canceled) { return; }

                    CreateSequenceSoundIntermediateFile(sound, update);	// XML中間ファイルの作成
                    CreateSequenceSoundFile(sound);					    // バイナリファイルの作成
                    sound.PrepareConvert();								    // サウンドアーカイブのコンバート準備
                    CreateSequenceSoundConvertCommand(sound, update);	    // コンバートコマンドの作成

                }

                // グループ
                foreach (Nw4rGroup group in _soundArchive.AllGroups)
                {

                    if (_canceled) { return; }

                    CreateGroupFile(group);							    // バイナリファイルの作成
                    group.PrepareConvert();								    // サウンドアーカイブのコンバート準備
                    CreateGroupConvertCommand(group, update);			    // コンバートコマンドの作成

                }

                // 残りのコンバートコマンドを処理する
                PerformConvert(_convertProcessor);

            }
            finally
            {
                // キャッシュマップファイルを出力する
                _fileManager.Save(false);
            }
        }

        private void PerformConvert(Nw4rConvertCommandProcessor processor)
        {
            Debug.Assert(null != processor);

            try
            {

                processor.OutputLine += OnConverterOutputLine;

                try
                {

                    Monitor.Enter(_cancelLock);
                    if (_canceled) { return; }

                    processor.Begin();

                }
                finally
                {
                    Monitor.Exit(_cancelLock);
                }

                processor.Wait();

            }
            finally
            {
                processor.End();
                processor.OutputLine -= OnConverterOutputLine;
            }

            if (null != processor.LastException) { throw processor.LastException; }
        }

        #region ** XML中間ファイルの作成

        private void CreateStreamSoundIntermediateFile(Nw4rStreamSound streamSound, bool update)
        {
            Debug.Assert(null != streamSound);

            Nw4rSoundOutput cache = _fileManager.GetIntermediateOutput(streamSound);
            if (null == cache) { return; }

            Nw4rSoundIntermediateOutput intermediateCache = new Nw4rSoundIntermediateOutput(cache);
            streamSound.XmlIntermediateFile = intermediateCache.File;


            // バイナリキャッシュが無効な場合は、中間ファイルを作成する
            if (_fileManager.IsValidBinaryOutputCache(streamSound)) { return; }

            Nw4rConvertCommand command = new Nw4rConvert2StreamXmlCommand(streamSound, intermediateCache);
            command.Silent = _settings.Silent;
            command.ForcedConvert = !update;
            command.ProcessHandler = _processHandler;

            Nw4rConvertCommandProcessor processor = new Nw4rConvertCommandProcessor();
            processor.Push(command);
            PerformConvert(processor);
        }

        private void CreateSequenceSoundIntermediateFile(Nw4rSequenceSound sequenceSound, bool update)
        {
            Debug.Assert(null != sequenceSound);

            Nw4rSoundOutput cache = _fileManager.GetIntermediateOutput(sequenceSound);
            if (null == cache) { return; }

            Nw4rSoundIntermediateOutput intermediateCache = new Nw4rSoundIntermediateOutput(cache);
            sequenceSound.XmlIntermediateFile = intermediateCache.File;


            // バイナリキャッシュが無効な場合は、中間ファイルを作成する
            if (_fileManager.IsValidBinaryOutputCache(sequenceSound)) { return; }

            Nw4rConvert2SequenceXmlCommand command = new Nw4rConvert2SequenceXmlCommand(sequenceSound, intermediateCache);
            command.Silent = _settings.Silent;
            command.ForcedConvert = !update;
            command.ProcessHandler = _processHandler;
            command.TimeBase = _soundArchive.SmfConvertTimeBase;

            Nw4rConvertCommandProcessor processor = new Nw4rConvertCommandProcessor();
            processor.Push(command);
            PerformConvert(processor);
        }

        #endregion

        #region ** バイナリファイルの作成

        private void CreateStreamSoundFile(Nw4rStreamSound streamSound)
        {
            Debug.Assert(null != streamSound);

            Nw4rStreamSoundOutput soundCache = new Nw4rStreamSoundOutput(_fileManager.GetBinaryOutput(streamSound));

            // サウンドに関連付けて、サウンドアーカイブに追加する
            streamSound.BinaryFile = soundCache.File;
            _soundArchive.AddBinaryFile(soundCache.File);
        }

        private void CreateWaveSoundFile(Nw4rWaveSound waveSound)
        {
            Debug.Assert(null != waveSound);

            Nw4rSoundBinaryOutput soundCache = new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutput(waveSound));

            // サウンドに関連付ける
            // WSDSETに統合されるので、サウンドアーカイブには追加しない
            waveSound.BinaryFile = soundCache.File;
        }

        private void CreateWaveSoundSetFile(Nw4rWaveSoundSet waveSoundSet)
        {
            Debug.Assert(null != waveSoundSet);

            Nw4rWaveSoundSetOutput soundCache = new Nw4rWaveSoundSetOutput(_fileManager.GetBinaryOutput(waveSoundSet));

            // サウンドに関連付けて、サウンドアーカイブに追加する
            waveSoundSet.BinaryFile = soundCache.File;
            _soundArchive.AddBinaryFile(soundCache.File);
        }

        private void CreateSequenceSoundFile(Nw4rSequenceSound sequenceSound)
        {
            Debug.Assert(null != sequenceSound);

            Nw4rSequenceSoundOutput soundCache = new Nw4rSequenceSoundOutput(_fileManager.GetBinaryOutput(sequenceSound));

            // サウンドに関連付けて、サウンドアーカイブに追加する
            sequenceSound.BinaryFile = soundCache.File;
            _soundArchive.AddBinaryFile(soundCache.File);
        }

        private void CreateBankFile(Nw4rBank bank)
        {
            Debug.Assert(null != bank);

            bank.Load();

            // バンクが参照する波形ファイル
            foreach (Nw4rInstrument instrument in bank.Components)
            {
                foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
                {
                    foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
                    {
                        velocityRegion.BinaryFile = new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutput(velocityRegion)).File;
                    }
                }
            }


            // バンク バイナリファイル
            Nw4rBankOutput soundCache = new Nw4rBankOutput(_fileManager.GetBinaryOutput(bank));

            // バンクに関連付けて、サウンドアーカイブに追加する
            bank.BinaryFile = soundCache.File;
            _soundArchive.AddBinaryFile(soundCache.File);
        }

        private void CreateGroupFile(Nw4rGroup group)
        {
            if (!group.XmlData.WaveArchiveShare) { return; }

            Debug.Assert(null != group);

            // グループ バイナリファイル
            Nw4rGroupOutput soundCache = new Nw4rGroupOutput(_fileManager.GetBinaryOutput(group));

            // グループに関連付けて、サウンドアーカイブに追加する
            group.BinaryFile = soundCache.File;
            _soundArchive.AddBinaryFile(soundCache.File);


            // グループ専用のウェーブサウンドセット、バンク バイナリファイル
            foreach (Model.Nw4rSoundSetItem item in group.AllDependItems)
            {

                Nw4rSoundOutput itemSoundCache = _fileManager.GetBinaryOutputForGroup(group, item);
                if (null == itemSoundCache) { continue; }

                // グループに関連付ける
                itemSoundCache.File.Label = _soundArchive.BinaryFiles[item.BinaryFile.Key].Label;
                item.SetBinaryFileForGroup(group, itemSoundCache.File as Nw4rSoundBinaryFile);

            }
        }

        #endregion

        #region ** コンバートコマンドの作成

        private void CreateStreamSoundConvertCommand(Nw4rStreamSound streamSound, bool update)
        {
            Debug.Assert(null != streamSound);

            // バイナリファイル作成コマンド
            Nw4rConvertCommand command = null;
            Nw4rStreamSoundOutput output = new Nw4rStreamSoundOutput(_fileManager.GetBinaryOutput(streamSound));

            if (null != streamSound.XmlIntermediateFile)
            {

                command = new Nw4rConvertXml2StreamCommand(streamSound, output);

                // コンバートが完了したら、中間ファイルを削除する
                new FileCleaner(_convertProcessor, streamSound.XmlIntermediateFile);

            }
            else
            {
                command = new Nw4rConvert2StreamCommand(streamSound, output);
            }

            // コンバートコマンドを追加する
            AddConvertCommand(streamSound.BinaryFile.Key, command, update);
        }

        private void CreateWaveSoundConvertCommand(Nw4rWaveSound waveSound, bool update)
        {
            Debug.Assert(null != waveSound);

            // コンバートコマンドを追加する
            Nw4rSoundOutput cache = _fileManager.GetBinaryOutput(waveSound);
            AddConvertCommand(waveSound.BinaryFile.Key, new Nw4rConvert2WaveCommand(waveSound, cache), update);
        }

        private void CreateWaveSoundSetConvertCommand(Nw4rWaveSoundSet waveSoundSet, bool update)
        {
            Debug.Assert(null != waveSoundSet);

            // コンバートコマンドを追加する
            AddConvertCommand(waveSoundSet.BinaryFile.Key,
                               new Nw4rConvert2WsdArchiveCommand(waveSoundSet,
                                                new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutput(waveSoundSet))),
                               update);
        }

        private void CreateSequenceSoundConvertCommand(Nw4rSequenceSound sequenceSound, bool update)
        {
            Debug.Assert(null != sequenceSound);

            // バイナリファイル作成コマンド
            Nw4rConvertCommand command = new Nw4rConvert2SequenceCommand(sequenceSound,
                                                    new Nw4rSequenceSoundOutput(_fileManager.GetBinaryOutput(sequenceSound)));

            // コンバートが完了したら、中間ファイルを削除する
            if (null != sequenceSound.XmlIntermediateFile && !_soundArchive.KeepIntermediateRseq)
            {
                new FileCleaner(_convertProcessor, sequenceSound.XmlIntermediateFile);
            }

            // コンバートコマンドを追加する
            AddConvertCommand(sequenceSound.BinaryFile.Key, command, update);
        }

        private void CreateBankConvertCommand(Nw4rBank bank, bool update)
        {
            Debug.Assert(null != bank);

            // 各ベロシティリージョンのバイナリファイル作成コマンド
            foreach (Nw4rInstrument instrument in bank.Components)
            {
                foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
                {
                    foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
                    {

                        Nw4rSoundOutput waveCache = _fileManager.GetBinaryOutput(velocityRegion);
                        AddConvertCommand(velocityRegion.BinaryFile.Key,
                                            new Nw4rConvertBankWave2WaveCommand(velocityRegion, waveCache), update);

                    }
                }
            }

            // バイナリファイル作成コマンド
            Nw4rSoundBinaryOutput cache = new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutput(bank));
            AddConvertCommand(bank.BinaryFile.Key, new Nw4rConvert2BankCommand(bank, cache), update);
        }

        private void CreateGroupConvertCommand(Nw4rGroup group, bool update)
        {
            Debug.Assert(null != group);

            if (!group.XmlData.WaveArchiveShare) { return; }

            // ウェーブサウンドセット、バンクをグループ専用にコンバート
            foreach (Model.Nw4rSoundSetItem item in group.AllDependItems)
            {

                if (item is Nw4rWaveSoundSet)
                {
                    CreateWaveSoundSetConvertCommandForGroup(group, item as Nw4rWaveSoundSet, update);
                }
                else if (item is Nw4rBank)
                {
                    CreateBankConvertCommandForGroup(group, item as Nw4rBank, update);
                }

            }

            // バイナリファイル コンバート
            Nw4rGroupOutput cache = new Nw4rGroupOutput(_fileManager.GetBinaryOutput(group));
            AddConvertCommand(group.BinaryFile.Key, new Nw4rConvert2GroupCommand(group, cache), update);
        }

        private void CreateWaveSoundSetConvertCommandForGroup(Nw4rGroup group, Nw4rWaveSoundSet waveSoundSet, bool update)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != waveSoundSet);

            // バイナリファイル コンバート
            Nw4rSoundBinaryOutput cache = new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutputForGroup(group, waveSoundSet));
            AddConvertCommand(cache.File.FilePath, new Nw4rConvert2WsdArchiveCommand(waveSoundSet, cache, group), update);
        }

        private void CreateBankConvertCommandForGroup(Nw4rGroup group, Nw4rBank bank, bool update)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != bank);

            // バイナリファイル コンバート
            Nw4rSoundBinaryOutput cache = new Nw4rSoundBinaryOutput(_fileManager.GetBinaryOutputForGroup(group, bank));
            AddConvertCommand(cache.File.FilePath, new Nw4rConvert2BankCommand(bank, cache, group), update);
        }

        private void AddConvertCommand(string key, Nw4rConvertCommand command, bool update)
        {
            Debug.Assert(null != command);

            if (_commands.ContainsKey(key)) { return; }

            command.Silent = _settings.Silent;
            command.ForcedConvert = !update;
            command.ProcessHandler = _processHandler;

            _convertProcessor.Push(command);
            _commands.Add(key, command);
        }

        #endregion

        #endregion

        #region ** リンク

        public void Link()
        {
            Link(true);
        }

        public void Link(bool update)
        {
            Link(string.Empty, update);
        }

        public void Link(string installDirectory)
        {
            Link(installDirectory, true);
        }

        public void Link(string installDirectory, bool update)
        {
            if (null == _soundArchive) { throw new Nw4rProjectNotLoadedException(); }

            if (_canceled) { return; }

            try
            {

                string soundArchiveFilePath = _soundArchive.OutputBinarySoundArchiveFilePath;

                // リンクの準備
                _soundArchive.PrepareLink();

                // リンク
                if (!update || ShouldLinkBinaries(soundArchiveFilePath))
                {
                    LinkBinaries(soundArchiveFilePath);
                }

                // インストール
                Install(soundArchiveFilePath, _soundArchive.DvdDataPath, update);

                if (null != installDirectory && 0 < installDirectory.Length)
                {
                    Install(soundArchiveFilePath, installDirectory, update);
                }

            }
            catch (Nw4rFileFormatException exception)
            {

                Debug.WriteLine(exception.ToString());
                OutputLineInternal(new ErrorLine(exception.Message, GetTargetItems(exception.OwnerLabel)));

                throw exception;

            }
            catch (Exception exception)
            {

                if (!_canceled)
                {
                    Debug.WriteLine(exception.ToString());
                    OnOutputLine(new Nw4rConverterOutputLineEventArgs(new ErrorLine(exception.Message)));
                    throw exception;
                }
            }
        }

        private void LinkBinaries(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            OutputLineInternal(new InformationLine(Resources.Messages.Message_Linking));

            try
            {

                string directoryName = Path.GetDirectoryName(filePath);
                if (0 < directoryName.Length)
                {
                    Directory.CreateDirectory(directoryName);
                }

                Nw4rSoundArchiveFile soundArchiveFile = null;

                using (FileStream stream = File.Open(filePath, FileMode.Create, FileAccess.Write, FileShare.Read))
                {

                    BinaryWriter writer = BinaryWriterBigEndian.CreateInstance(stream);

                    soundArchiveFile = new Nw4rSoundArchiveFile(_soundArchive);
                    soundArchiveFile.Write(writer);
                    writer.Flush();

                }

                CreateMapFile(Path.ChangeExtension(filePath, "rsam"), soundArchiveFile);
                CreateIDFile(Path.ChangeExtension(filePath, "rsid"));

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, string.Empty);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(filePath, string.Empty);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(filePath, string.Empty);
            }
            catch (Exception exception)
            {

                Debug.WriteLine(exception.ToString());

                try
                {
                    File.Delete(filePath);
                }
                catch (Exception)
                {
                }

                throw exception;

            }
        }

        private bool ShouldLinkBinaries(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            // サウンドアーカイブファイルが存在する場合のみ、入力ファイル群と更新日時を比較する
            if (!File.Exists(filePath)) { return true; }

            DateTime soundArchiveFileDate = File.GetLastWriteTime(filePath);

            // プロジェクトファイルと比較
            if (File.GetLastWriteTime(_soundArchive.ProjectFilePath) > soundArchiveFileDate) { return true; }

            // サウンドセットファイルと比較
            foreach (Model.Nw4rSoundSet soundSet in _soundArchive.AllSoundSets)
            {
                if (File.GetLastWriteTime(soundSet.XmlData.FilePath) > soundArchiveFileDate) { return true; }
            }

            // コンバートされたバイナリファイルと比較
            foreach (Nw4rSoundBinaryFile binaryFile in _soundArchive.BinaryFiles)
            {
                if (0 == binaryFile.FilePath.Length) { continue; }
                if (File.GetLastWriteTime(binaryFile.FilePath) > soundArchiveFileDate) { return true; }
            }

            return false;
        }

        /// <summary>
        /// バイナリファイルを DVDData ディレクトリにインストールします。
        /// </summary>
        /// <param name="archiveFilePath">サウンドアーカイブファイルパス</param>
        private void Install(string archiveFilePath, string outputDirectory, bool update)
        {
            if (null == archiveFilePath) { throw new ArgumentNullException("archiveFilePath"); }
            if (0 == archiveFilePath.Length) { throw new ArgumentException("archiveFilePath"); }
            if (null == outputDirectory) { throw new ArgumentNullException("outputDirectory"); }
            if (0 == outputDirectory.Length) { throw new ArgumentException("outputDirectory"); }

            // サウンドアーカイブ
            InstallFile(archiveFilePath, Path.Combine(outputDirectory, Path.GetFileName(archiveFilePath)), update);

            // 外部ファイル（ストリームサウンドファイル）
            foreach (Nw4rSoundBinaryFile binaryFile in _soundArchive.BinaryFiles)
            {

                if (!(binaryFile is Nw4rStreamSoundBinaryFile)) { continue; }

                Nw4rStreamSoundBinaryFile streamSoundFile = binaryFile as Nw4rStreamSoundBinaryFile;
                InstallFile(streamSoundFile.FilePath,
                             TargetRelativeToPCAbsolutePath(outputDirectory, streamSoundFile.ExtensionFileRelativePath),
                             update);

            }
        }

        /// <summary>
        /// 実機上のコンテキスト相対パスをPC上の絶対パスに変換します。
        /// </summary>
        /// <param name="basePath">PC上の基準パス。</param>
        /// <param name="relativePath">実機上のコンテキスト相対パス。</param>
        /// <returns>PC上の絶対パス</returns>
        private string TargetRelativeToPCAbsolutePath(string basePath, string relativePath)
        {
            if (null == basePath) { throw new ArgumentNullException("basePath"); }
            if (null == relativePath) { throw new ArgumentNullException("relativePath"); }
            if (0 == relativePath.Length) { return string.Empty; }
            if ('/' != relativePath[0] && Path.IsPathRooted(relativePath)) { return string.Empty; }

            return Path.Combine(basePath, Path.GetFullPath(@"c:\" + relativePath).Substring(3));
        }

        /// <summary>
        /// ファイルをインストールします。
        /// </summary>
        /// <param name="srcFilePath">コピー元ファイルパス</param>
        /// <param name="destFilePath">コピー先ファイルパス</param>
        /// <param name="update">更新日時が新しい場合のみインストールする場合は true、必ずインストールする場合は false。</param>
        private void InstallFile(string srcFilePath, string destFilePath, bool update)
        {
            if (null == srcFilePath) { throw new ArgumentNullException("srcFilePath"); }
            if (!File.Exists(srcFilePath)) { throw new FileNotFoundException(srcFilePath); }
            if (null == destFilePath) { throw new ArgumentNullException("destFilePath"); }
            if (0 == destFilePath.Length) { throw new ArgumentException("destFilePath"); }

            if (File.Exists(destFilePath))
            {
                // update が true の場合、コピー元ファイルがコピー先ファイルより新しい場合のみ処理を行う
                if (update && File.GetLastWriteTime(srcFilePath) <= File.GetLastWriteTime(destFilePath)) { return; }
            }

            try
            {

                Directory.CreateDirectory(Path.GetDirectoryName(destFilePath));
                File.Copy(srcFilePath, destFilePath, true);

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, string.Empty);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(destFilePath, string.Empty);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(destFilePath, string.Empty);
            }
        }

        #endregion

        #region ** 関連ファイル出力

        private void CreateMapFile(string filePath, Nw4rSoundArchiveFile soundArchiveFile)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }
            if (null == soundArchiveFile) { throw new ArgumentNullException("soundArchiveFile"); }

            try
            {

                using (FileStream stream = File.Open(filePath, FileMode.Create, FileAccess.Write, FileShare.Read))
                {

                    StreamWriter writer = new StreamWriter(stream);
                    Nw4rSoundArchiveMapFile mapFile = new Nw4rSoundArchiveMapFile(soundArchiveFile);

                    mapFile.Write(writer, Path.GetDirectoryName(filePath));
                    writer.Flush();

                }

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, string.Empty);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(filePath, string.Empty);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(filePath, string.Empty);
            }
        }

        private void CreateIDFile(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            if (0 == filePath.Length) { throw new ArgumentException("filePath"); }

            try
            {

                using (FileStream stream = File.Open(filePath, FileMode.Create, FileAccess.Write, FileShare.Read))
                {

                    StreamWriter writer = new StreamWriter(stream);
                    Nw4rSoundArchiveIDFile idFile = new Nw4rSoundArchiveIDFile(_soundArchive);

                    idFile.Write(writer);
                    writer.Flush();

                }

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, string.Empty);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(filePath, string.Empty);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(filePath, string.Empty);
            }
        }

        #endregion

        #region ** ライン出力

        private void OutputLineInternal(OutputLine line)
        {
            if (null == line) { return; }
            OnOutputLine(new Nw4rConverterOutputLineEventArgs(line));
        }

        #endregion

        #region ** アイテム操作

        private string[] SplitLabel(string label)
        {
            if (null == label || 0 == label.Length) { return new string[0]; }
            return label.Split('.');
        }

        private Nw4rItem[] GetTargetItems(string label)
        {
            try
            {

                string[] labels = SplitLabel(label);
                List<Nw4rItem> targetItems = new List<Nw4rItem>();

                if (0 < labels.Length)
                {

                    if (null != _soundArchive && _soundArchive.AllItems.Contains(labels[0]))
                    {

                        Nw4rItem item = _soundArchive.AllItems[labels[0]];
                        if (null == item) { return new Nw4rItem[0]; }

                        targetItems.Add(item);

                    }

                    if (1 < labels.Length && targetItems[0] is Nw4rBank)
                    {

                        string targetLabel2 = labels[0] + "." + labels[1];

                        if (targetItems[0].Components.Contains(targetLabel2))
                        {

                            Nw4rItem item = targetItems[0].Components[targetLabel2] as Nw4rItem;

                            if (null != item)
                            {
                                targetItems.Add(item);
                            }

                        }

                    }

                }

                return targetItems.ToArray();

            }
            catch { return new Nw4rItem[0]; }
        }

        #endregion

        #endregion

        #region ** コンバート ユーティリティクラス

        private class FileCleaner
        {
            private Nw4rSoundIntermediateFile _file = null;

            public FileCleaner(Nw4rConvertCommand command, Nw4rSoundIntermediateFile file)
            {
                Debug.Assert(null != command);
                Debug.Assert(null != file);

                command.Completed += OnCommandCompleted;

                _file = file;
            }

            public FileCleaner(Nw4rConvertCommandProcessor processor, Nw4rSoundIntermediateFile file)
            {
                Debug.Assert(null != processor);
                Debug.Assert(null != file);

                processor.ProcessCompleted += OnCommandCompleted;

                _file = file;
            }

            private void OnCommandCompleted(object sender, Nw4rCommandEventArgs e)
            {
                try
                {
                    File.Delete(_file.FilePath);
                }
                catch (Exception)
                {
                }
            }
        }

        #endregion
    }

    /// <summary>
    /// サウンドアーカイブコンバータ設定
    /// </summary>
    public class Nw4rSoundArchiveConverterSettings
    {
        private bool _silent = false;	// サイレント設定

        public bool Silent
        {
            get { return _silent; }
            set { _silent = value; }
        }
    }

    #region ** イベントデリゲートとパラメータ

    public delegate void Nw4rConverterOutputTextEventHandler(object sender, Nw4rConverterOutputLineEventArgs e);

    public class Nw4rConverterOutputLineEventArgs : EventArgs
    {
        private OutputLine _line = null;

        public Nw4rConverterOutputLineEventArgs(OutputLine line)
        {
            Debug.Assert(null != line);
            _line = line;
        }

        #region ** プロパティ

        public OutputLine Line
        {
            get { return _line; }
        }

        #endregion
    }

    #endregion
}
