﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;

using NintendoWare.SoundFoundation.Legacies.Resources;
using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    internal class Nw4rConvert2BankCommand : Nw4rConvertCommandT<Nw4rBank, Nw4rSoundBinaryOutput>
    {
        private Nw4rGroup _group = null;

        public Nw4rConvert2BankCommand(Nw4rBank input, Nw4rSoundBinaryOutput output) : base(input, output) { }

        public Nw4rConvert2BankCommand(Nw4rBank input, Nw4rSoundBinaryOutput output, Nw4rGroup group) : base(input, output)
        {
            _group = group;
        }

        #region ** プロパティ

        protected override bool ShouldConvert
        {
            get
            {
                try
                {
                    return Output.Dirty;
                }
                catch (DependencyFileNotFoundException exception)
                {
                    throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
                }
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnInvoke()
        {
            if (null == _group)
            {
                OutputLineInternal(new InformationLine(string.Format(Messages.MessageFormat_Converting, Input.Label)));
            }
            else
            {
                string label = Input.Label + " for " + _group.Label;
                OutputLineInternal(new InformationLine(string.Format(Messages.MessageFormat_Converting, label)));
            }

            OutputWaveArchiveFile();
            OutputBankFile();
            OutputIncludeFile();
        }

        protected override void OnClean()
        {
            Output.OutputItem.Clean();
        }

        #endregion

        #region ** メソッド

        private void OutputBankFile()
        {
            try
            {

                // 出力ディレクトリの生成
                CreateDirectory(Path.GetDirectoryName(Output.File.FilePath), Input.Label);

                // バンクファイル
                Nw4rBankFile bankFile = new Nw4rBankFile(Input, (null == _group) ? null : _group.BinaryFile.Components);

                using (FileStream stream = File.Open(Output.File.FilePath, FileMode.Create, FileAccess.Write))
                {

                    BinaryWriter writer = BinaryWriterBigEndian.CreateInstance(stream);

                    bankFile.Write(writer);
                    writer.Flush();

                }

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
            }
            catch (DirectoryNotFoundException exception)
            {
                throw new Nw4rFileFormatException(exception.Message, Input.Label);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(Output.File.FilePath, Input.Label);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(Output.File.FilePath, Input.Label);
            }
        }

        private void OutputWaveArchiveFile()
        {
            // グループ用に出力する場合は、出力しない
            if (null != _group) { return; }

            try
            {

                // 出力ディレクトリの生成
                CreateDirectory(Path.GetDirectoryName(Output.File.ARamFilePath), Input.Label);

                // 波形アーカイブファイル
                Nw4rWsdWaveArchiveFile warFile = new Nw4rWsdWaveArchiveFile(Input.BinaryFile.Components);

                using (FileStream stream = File.Open(Output.File.ARamFilePath, FileMode.Create, FileAccess.Write))
                {

                    BinaryWriter writer = BinaryWriterBigEndian.CreateInstance(stream);

                    warFile.Write(writer);
                    writer.Flush();

                }

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
            }
            catch (DirectoryNotFoundException exception)
            {
                throw new Nw4rFileFormatException(exception.Message, Input.Label);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(Output.File.ARamFilePath, Input.Label);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(Output.File.ARamFilePath, Input.Label);
            }
        }

        private void OutputIncludeFile()
        {
            // グループ用に出力する場合は、出力しない
            if (null != _group) { return; }

            Nw4rBankOutput bankCache = new Nw4rBankOutput(Output.SoundCache);

            try
            {

                // バンクインクルードファイル
                Nw4rBankIncludeFile includeFile = new Nw4rBankIncludeFile(Input);

                using (FileStream stream = File.Open(bankCache.File.IncludeFilePath, FileMode.Create, FileAccess.Write))
                {

                    TextWriter writer = new StreamWriter(stream);

                    includeFile.Write(writer);
                    writer.Flush();

                }

            }
            catch (FileNotFoundException exception)
            {
                throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
            }
            catch (DirectoryNotFoundException exception)
            {
                throw new Nw4rFileFormatException(exception.Message, Input.Label);
            }
            catch (UnauthorizedAccessException)
            {
                throw new Nw4rFileAccessDeniedException(bankCache.File.IncludeFilePath, Input.Label);
            }
            catch (PathTooLongException)
            {
                throw new Nw4rFilePathTooLongException(bankCache.File.IncludeFilePath, Input.Label);
            }
        }

        #endregion
    }
}
