﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// Cafe バンクドキュメントをバンクドキュメントに変換します。
    /// </summary>
    public class BankCafeToBankDocumentConverter : IDocumentConverter
    {
        private readonly string outputDocumentTypeName = string.Empty;
        private readonly string outputFileExtension = string.Empty;
        private IObjectFactory<Type, Component> componentFactory;

        public BankCafeToBankDocumentConverter(
            string outputDocumentTypeName, string outputFileExtension)
            : this(outputDocumentTypeName, outputFileExtension, new BankComponentFactory())
        {
        }

        public BankCafeToBankDocumentConverter(
            string outputDocumentTypeName,
            string outputFileExtension,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.StringNotEmpty(outputDocumentTypeName);
            Ensure.Argument.NotNull(componentFactory);

            this.outputDocumentTypeName = outputDocumentTypeName;
            this.outputFileExtension = outputFileExtension;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Cafe.BankDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.outputDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as BankDocument,
                outputDocument as BankDocument);
        }

        private void ConvertInternal( BankDocument inputDocument, BankDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            string bankDictionaryPath = Path.GetDirectoryName(inputDocument.Resource.Key);

            outputDocument.Bank = this.ConvertBank(inputDocument.Bank, bankDictionaryPath);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource(
                    Path.ChangeExtension(inputDocument.Resource.Key, this.outputFileExtension));
            }
        }

        private Bank ConvertBank( Bank source, string basePath)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            Bank dest = this.componentFactory.Create(typeof(Bank)) as Bank;
            Ensure.Operation.ObjectNotNull(dest);

            //foreach (LegacyFormat.Nw4rXmlBankInst sourceInstrument in source.InstList)
            foreach( Instrument sourceInstrument in source.Children )
            {
                dest.Children.Add(this.ConvertInstrument(sourceInstrument, basePath));
            }

            return dest;
        }

        //-----------------------------------------------------------------
        // バンクアイテムのコンバート
        //-----------------------------------------------------------------

        private Instrument ConvertInstrument( Instrument source, string basePath)
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            ImportLog.WriteLine("  Instrument: ({0})", source.Name);
            Instrument dest = new Instrument();

            ComponentConverter.Converter(dest, source);

            foreach( KeyRegion sourceKeyRegion in source.Children )
            {
                dest.Children.Add( this.ConvertKeyRegion( sourceKeyRegion, basePath,
                                                          dest.EnvelopeMode));
            }

            return dest;
        }

        private KeyRegion ConvertKeyRegion(
                                           KeyRegion source,
                                           string basePath,
                                           //out bool envModeVel
                                           InstrumentEnvelopeMode envelopeMode
                                           )
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);

            ImportLog.WriteLine("    KeyRegion: ({0}, {1})", source.KeyMin, source.KeyMax);
            KeyRegion dest = new KeyRegion();

            ComponentConverter.Converter(dest, source);

            foreach( VelocityRegion sourceVelocityRegion in source.Children )
            {
                dest.Children.Add(this.ConvertVelocityRegion(
                                                             sourceVelocityRegion,
                                                             basePath,
                                                             //out mode,
                                                             envelopeMode,
                                                             source
                                                             ));
            }

            return dest;
        }

        private VelocityRegion ConvertVelocityRegion(
                                                     VelocityRegion source,
                                                     string basePath,
                                                     //out bool envModeVel,
                                                     InstrumentEnvelopeMode envelopeMode,
                                                     KeyRegion keySrc
                                                     )
        {
            Assertion.Argument.NotNull(source);
            Assertion.Argument.NotNull(basePath);
            Assertion.Argument.NotNull(keySrc);

            ImportLog.WriteLine(
                                "      VelocityRegion: ({0}, {1}) ({2})",
                                source.VelocityMin,
                                source.VelocityMax,
                                Path.Combine(basePath, source.FilePath).GetFullPath()
                                );
            VelocityRegion dest = new VelocityRegion();
            List<string> filters =
                new List<string> { ProjectParameterNames.FilePath,
                                   ProjectParameterNames.VelocityRegion.InterpolationType };
            ComponentConverter.Converter(dest, source, filters);

            dest.FilePath = Path.Combine(basePath, source.FilePath).GetFullPath();
            dest.InterpolationType = (source.InterpolationType == InterpolationType.None ?
                                      InterpolationType.Linear : source.InterpolationType);

            return dest;
        }

        //-----------------------------------------------------------------
        // パラメータのコンバート
        //-----------------------------------------------------------------

        private Envelope ConvertEnvelope( Envelope source)
        {
            Assertion.Argument.NotNull(source);

            return new Envelope()
            {
                Attack = source.Attack,
                Decay = source.Decay,
                Hold = source.Hold,
                Release = source.Release,
                Sustain = source.Sustain,
            };
        }
    }
}
