﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#define ENABLE_COMMENT
using System;
using System.Collections;
using System.IO;
using System.Text;
using NintendoWare.SoundFoundation.Core.IO;
using NintendoWare.SoundMaker.Framework.Resources;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    public struct StxtData
    {
        public string Label;
        public string FilePath;
        public int ChannelPriority;
        public int Volume;
        public int Pan;
        public double Pitch;
        public int Span;
        public int SendA;
        public int SendB;

#if ENABLE_COMMENT
        public string Comment;
#else
        public string Comment {
            get {
                return string.Empty;
            }
        }
#endif

        public StxtData(string label, string filePath, int channelPriority, int volume, int pan, double pitch, int span, int sendA, int sendB
#if ENABLE_COMMENT
                        , string comment
#endif
                        )
        {
            Label = label;
            FilePath = filePath;
            ChannelPriority = channelPriority;
            Volume = volume;
            Pan = pan;
            Pitch = pitch;
            Span = span;
            SendA = sendA;
            SendB = sendB;
#if ENABLE_COMMENT
            Comment = comment;
#endif
        }
    }
    public class StxtReader
    {
        private enum TokenType
        {
            Symbol,
            String,
            Double,
            PercentDouble,
            DBDouble,
            LeftBrace,
            RightBrace,
            NewLine,
#if ENABLE_COMMENT
            Comment,
#endif
            EndOfStream,
        }
        private struct Token
        {
            public TokenType Type;
            public object Value;
            public Token(TokenType type, object value)
            {
                Type = type;
                Value = value;
            }
            /// <summary>
            /// シンボルをパンの引数に使う LCRのプレフィックス付き数値に変換する。変換後の値の範囲は0～255。
            /// トークンがシンボルでない場合や、変換できない場合は null を返す。
            /// </summary>
            public int? LCRPrefixValue
            {
                get
                {
                    if (Type != TokenType.Symbol)
                    {
                        return null;
                    }
                    string sym = Value as string;
                    char prefix = char.ToLower(sym[0]);
                    try
                    {
                        if (prefix == 'l')
                        {
                            int val = int.Parse(sym.Substring(1));
                            if (val >= 1 && val <= 127)
                                return 127 - val;
                        }
                        if (prefix == 'c')
                        {
                            if (sym.Length == 1)
                            {
                                return 127;
                            }
                        }
                        if (prefix == 'r')
                        {
                            int val = int.Parse(sym.Substring(1));
                            if (val >= 1 && val <= 128)
                                return 127 + val;
                        }
                    }
                    catch
                    {
                    }
                    return null;
                }
            }
        }
        private static Token EndToken = new Token(TokenType.EndOfStream, null);
        private class Lexer
        {
            /// <summary>
            /// リソースからメッセージを取得
            /// </summary>
            /// <param name="id"></param>
            /// <returns></returns>
            private StreamReader Reader;
            private int lineNumber = 0;
            public Lexer(StreamReader reader)
            {
                Reader = reader;
            }
            /// <summary>
            /// 直前にReadまたはPeekされたトークンを含む行の行番号を取得します。
            /// </summary>
            public int LineNumber
            {
                get
                {
                    return lineNumber;
                }
            }
            public Token Read(bool ignoreNewLine)
            {
                return Read(ignoreNewLine, false);
            }
            public Token Peek()
            {
                return Read(false, true);
            }
            private ArrayList tokenBuf = new ArrayList();
            private Token Read(bool ignoreNewLine, bool peek)
            {
                if (tokenBuf.Count > 0)
                {
                    Token t = (Token)tokenBuf[0];
                    if (peek == false)
                        tokenBuf.RemoveAt(0);
                    if (ignoreNewLine && t.Type == TokenType.NewLine)
                        return Read(ignoreNewLine, peek);
                    else
                        return t;
                }
                else
                {
                    ReadLine();
                    return Read(ignoreNewLine, peek);
                }
            }
            private bool IsLetter(char c)
            {
                if (c >= 'A' && c <= 'Z' ||
                    c >= 'a' && c <= 'z' ||
                    c == '_')
                    return true;
                else
                    return false;
            }
            private bool IsDigit(char c)
            {
                if (c >= '0' && c <= '9')
                    return true;
                else
                    return false;
            }
            private bool IsLetterOrDigit(char c)
            {
                if (c >= '0' && c <= '9' ||
                    c >= 'A' && c <= 'Z' ||
                    c >= 'a' && c <= 'z' ||
                    c == '_')
                    return true;
                else
                    return false;
            }
            private void ReadLine()
            {
                string line = Reader.ReadLine();
                int i, j;
                if (line == null)
                {
                    tokenBuf.Add(new Token(TokenType.EndOfStream, null));
                    return;
                }
                lineNumber++;
                for (i = 0; i < line.Length;)
                {
                    if (line[i] == ' ' || line[i] == '\t' || line[i] == '\r' || line[i] == '\n')
                    {
                        i++;
                        continue;
                    }
                    if (line[i] == '/' && i + 1 < line.Length && line[i + 1] == '/')
                    {
#if ENABLE_COMMENT
                        if (i + 2 < line.Length)
                        {
                            string comment = line.Substring(i + 2).Trim();
                            if (comment.Length > 0)
                            {
                                tokenBuf.Add(new Token(TokenType.Comment, comment));
                            }
                        }
#endif
                        break;
                    }
                    if (IsLetter(line[i]) || line[i] == '_')
                    {
                        // シンボル
                        for (j = i; j < line.Length; j++)
                        {
                            if (IsLetterOrDigit(line[j]) || line[j] == '_')
                                continue;
                            break;
                        }
                        tokenBuf.Add(new Token(TokenType.Symbol, line.Substring(i, j - i)));
                        i = j;
                        continue;
                    }
                    else if (IsDigit(line[i]) || line[i] == '+' || line[i] == '-')
                    {
                        bool negative = false;
                        if (line[i] == '+')
                        {
                            i++;
                        }
                        if (line[i] == '-')
                        {
                            negative = true;
                            i++;
                        }
                        // 数値
                        bool point = false;
                        for (j = i; j < line.Length; j++)
                        {
                            if (IsDigit(line[j]))
                                continue;
                            if (point == false && j != i && line[j] == '.')
                            {
                                point = true;
                                continue;
                            }
                            if (j + 1 < line.Length && char.ToLower(line[j]) == 'd' && char.ToLower(line[j + 1]) == 'b')
                            {
                                if (j + 2 >= line.Length)
                                    break;
                                if (IsLetter(line[j + 2]) || line[j] == '_')
                                {
                                }
                                else
                                {
                                    break;
                                }
                            }
                            if (IsLetter(line[j]) || line[j] == '_' || i == j) // i==j の条件は、符号だけがきたとき用
                                throw new Exception(MessageResource.Stxt_IllegalNumberFormat);
                            break;
                        }
                        double val;
                        try
                        {
                            val = double.Parse(line.Substring(i, j - i));
                        }
                        catch
                        {
                            throw new Exception(MessageResource.Stxt_NumReadError);
                        }
                        if (negative)
                            val = -val;
                        if (j < line.Length && line[j] == '%')
                        {
                            // %付きの数値
                            tokenBuf.Add(new Token(TokenType.PercentDouble, val));
                            i = j + 1;
                            continue;
                        }
                        else if (j + 1 < line.Length && char.ToLower(line[j]) == 'd' && char.ToLower(line[j + 1]) == 'b')
                        {
                            // db付きの数値
                            tokenBuf.Add(new Token(TokenType.DBDouble, val));
                            i = j + 2;
                            continue;
                        }
                        else
                        {
                            // db, %なしの数値
                            tokenBuf.Add(new Token(TokenType.Double, val));
                            i = j;
                            continue;
                        }
                    }
                    else if (line[i] == '"')
                    {
                        // 文字列
                        for (j = i + 1; j < line.Length; j++)
                        {
                            if (line[j] != '"')
                                continue;
                            break;
                        }
                        if (j == line.Length)
                            throw new Exception(MessageResource.Stxt_DoubleQuotationExpected);
                        tokenBuf.Add(new Token(TokenType.String, line.Substring(i + 1, j - 1 - i)));
                        i = j + 1;
                        continue;
                    }
                    else if (line[i] == '{')
                    {
                        tokenBuf.Add(new Token(TokenType.LeftBrace, null));
                        i++;
                        continue;
                    }
                    else if (line[i] == '}')
                    {
                        tokenBuf.Add(new Token(TokenType.RightBrace, null));
                        i++;
                        continue;
                    }
                    else
                    {
                        throw new Exception(string.Format(MessageResource.Stxt_CantBeUsed, line[i]));
                    }
                }
                tokenBuf.Add(new Token(TokenType.NewLine, null));
            }
        }
        private class Parser
        {
            /// <summary>
            /// リソースからメッセージを取得
            /// </summary>
            /// <param name="id"></param>
            /// <returns></returns>
            private Lexer Lexer;
            public Parser(StreamReader reader)
            {
                Lexer = new Lexer(reader);
            }
            public StxtData[] Read(string StxtFile)
            {
                ArrayList data = new ArrayList(16);
                try
                {
                    for (;;)
                    {
                        Token t = Lexer.Read(true);
                        if (t.Type == TokenType.Symbol && t.Value as string == "TypeSoundMat")
                        {
                            // サウンドマテリアル開始
                            Token namet;
                            namet = Lexer.Read(true);
                            if (namet.Type != TokenType.Symbol)
                            {
                                throw new Exception(MessageResource.Stxt_SoundMaterialNameRequired);
                            }
                            string name = namet.Value as string;
                            for (int i = 0; i < data.Count; i++)
                            {
                                if (((StxtData)data[i]).Label == name)
                                {
                                    throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialRedefined, name));
                                }
                            }
                            Token brt = Lexer.Read(true);
                            if (brt.Type != TokenType.LeftBrace)
                            {
                                throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialLeftBraceRequired, name));
                            }
                            int wait = 0;
                            string command = null;
                            string filename = null;
                            int avol = -1;
                            int apri = -1;
                            int? apan = null;
                            double? atune = null;
                            int? aspan = null;
                            int? amixa = null;
                            int? amixb = null;
#if ENABLE_COMMENT
                            string comment = null;
#endif
                            bool hasWait;
                            for (;;)
                            {
                                // サウンドマテリアルの中身
                                Token l1 = Lexer.Read(true);
                                if (l1.Type == TokenType.RightBrace)
                                    break;
#if ENABLE_COMMENT
                                if (l1.Type == TokenType.Comment)
                                {
                                    if (comment == null)
                                    {
                                        comment = l1.Value as string;
                                    }
                                    continue;
                                }
#endif
                                hasWait = false;
                                if (l1.Type == TokenType.Double)
                                {
                                    wait = (int)(double)l1.Value;
                                    hasWait = true;
                                    l1 = Lexer.Read(false);
                                }
                                if (l1.Type != TokenType.Symbol)
                                {
                                    if (hasWait == false)
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialRightBraceRequired, name));
                                    else
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialCommandRequired, name));
                                }
                                command = l1.Value as string;
                                command = command.ToLower();
                                l1 = Lexer.Read(false);
                                if (command == "sample")
                                {
                                    if (l1.Type == TokenType.String)
                                    {
                                        if (filename == null)
                                        {
                                            FileInfo stxt = new FileInfo(StxtFile);
                                            filename = Path.Combine(stxt.DirectoryName, (l1.Value as string).Replace('/', '\\')).GetFullPath();
                                        }
                                    }
                                }
                                if (command == "avol")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (avol < 0)
                                        {
                                            avol = (int)((double)l1.Value / 2);
                                        }
                                    }
                                    else if (l1.Type == TokenType.PercentDouble)
                                    {
                                        if (avol < 0)
                                        {
                                            avol = (int)((double)l1.Value * 127 / 100);
                                        }
                                    }
                                    else if (l1.Type == TokenType.DBDouble)
                                    {
                                        if (avol < 0)
                                        {
                                            avol = (int)(Math.Pow(10, (double)l1.Value / 200) * 127);
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAvolArgumentRequired, name));
                                    }
                                    if (avol < 0 || avol > 127)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAvolArgumentOutOfRange, name));
                                    }
                                }
                                if (command == "apri")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (apri < 0)
                                        {
                                            apri = (int)(double)l1.Value;
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialApriArgumentRequired, name));
                                    }
                                    if (apri < 0 || apri > 27)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialApriArgumentOutOfRange, name));
                                    }
                                }
                                if (command == "apan")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (apan == null)
                                        {
                                            apan = (int)(double)l1.Value;
                                        }
                                    }
                                    else if (l1.LCRPrefixValue != null)
                                    {
                                        apan = l1.LCRPrefixValue.Value;
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialApanArgumentRequired, name));
                                    }
                                    if (apan < 0 || apan > 255)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialApanArgumentOutOfRange, name));
                                    }
                                    apan = (apan + 1) / 2;
                                    if (apan > 127)
                                        apan = 127;
                                }
                                if (command == "atune")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if ((double)l1.Value < -10800 || (double)l1.Value > 2400)
                                        {
                                            throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAtuneArgumentOutOfRange, name));
                                        }
                                        if (atune == null)
                                        {
                                            atune = Math.Pow(2, (double)l1.Value / 1200);
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAtuneArgumentRequired, name));
                                    }
                                }
                                if (command == "aspan")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (aspan == null)
                                        {
                                            aspan = (int)((double)l1.Value / 2);
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAspanArgumentRequired, name));
                                    }
                                    if (aspan < 0 || aspan > 127)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAspanArgumentOutOfRange, name));
                                    }
                                }
                                if (command == "amixa")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (amixa == null)
                                        {
                                            amixa = (int)((double)l1.Value / 2);
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAmixaArgumentRequired, name));
                                    }
                                    if (amixa < 0 || amixa > 127)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAmixaArgumentOutOfRange, name));
                                    }
                                }
                                if (command == "amixb")
                                {
                                    if (l1.Type == TokenType.Double)
                                    {
                                        if (amixb == null)
                                        {
                                            amixb = (int)((double)l1.Value / 2);
                                        }
                                    }
                                    else
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAmixbArgumentRequired, name));
                                    }
                                    if (amixb < 0 || amixb > 127)
                                    {
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialAmixbArgumentOutOfRange, name));
                                    }
                                }
                                // 必要なコマンドは読んだので、行末またはカッコ閉じまで読み飛ばす
                                for (;;)
                                {
#if ENABLE_COMMENT
                                    if (l1.Type == TokenType.Comment)
                                    {
                                        if (comment == null)
                                        {
                                            comment = l1.Value as string;
                                        }
                                    }
#endif
                                    if (l1.Type == TokenType.NewLine || l1.Type == TokenType.RightBrace)
                                        break;
                                    if (l1.Type == TokenType.EndOfStream)
                                        throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialUnexpectedEOF, name));
                                    l1 = Lexer.Read(false);
                                }
                                if (l1.Type == TokenType.RightBrace)
                                    break;
                            }
                            if (filename == null)
                            {
                                throw new Exception(string.Format(MessageResource.Stxt_SoundMaterialSampleCommandtRequired, name));
                            }
                            if (avol < 0)
                                avol = 127;
                            if (apri < 0)
                                apri = 10;
                            if (apan == null)
                                apan = 64;
                            if (atune == null)
                                atune = 1.0;
                            if (aspan == null)
                                aspan = 0;
                            if (amixa == null)
                                amixa = 0;
                            if (amixb == null)
                                amixb = 0;
#if ENABLE_COMMENT
                            if (comment == null)
                                comment = string.Empty;
#endif
                            atune = ((double)(int)(atune.Value * 100000 + 0.5)) / 100000;
                            data.Add(new StxtData(name, filename, apri, avol, apan.Value, atune.Value, aspan.Value, amixa.Value, amixb.Value
#if ENABLE_COMMENT
                                                  , comment
#endif
                                                  ));
                        }
                        else if (t.Type == TokenType.Comment)
                        {
                            continue;
                        }
                        else if (t.Type == TokenType.EndOfStream)
                        {
                            break;
                        }
                        else
                        {
                            throw new Exception(MessageResource.Stxt_UnexpectedElement);
                        }
                    }
                }
                catch (Exception e)
                {
                    throw new Exception(string.Format(MessageResource.Stxt_LineNo, Lexer.LineNumber) + "\n" + e.Message);
                }
                return data.ToArray(typeof(StxtData)) as StxtData[];
            }
        }
        public static StxtData[] Read(string StxtFile)
        {
            using (StreamReader sr = new StreamReader(StxtFile, Encoding.GetEncoding("shift-jis")))
            {
                Parser p = new Parser(sr);
                return p.Read(StxtFile);
            }
        }
    }
}
