﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.IO;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    using Resources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public class RequestFilterEventArgs : EventArgs
    {
        public string Filter { get; set; }
    }

    public delegate void RequestFilterHandler(RequestFilterEventArgs e);

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class FileQuester
    {
        public const string AllFilesFilter = "|All Files (*.*)|*.*";

        ///--------------------------------
        /// <summary>
        /// 指定パスからディレクトリを取得
        /// 確実に存在するパスを返す
        /// </summary>
        public static string GetValidateDirectory(string directory)
        {
            if (directory == null) { return null; }
            if (directory == String.Empty) { return null; }

            if (Directory.Exists(directory) == false)
            {
                return null;
            }

            return directory;
        }

        ///--------------------------------
        /// <summary>
        /// パスの取得
        /// </summary>
        public static string QueryWithFile(
                                           FileDialog dialog,
                                           string filter,
                                           string initialFilePath
                                           )
        {
            string directory = null;

            try
            {
                if ((directory =
                     GetValidateDirectory(Path.GetDirectoryName(initialFilePath))) != null)
                {
                    dialog.InitialDirectory = directory;
                }

                dialog.Filter = filter;
                dialog.RestoreDirectory = true;
                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return null;
                }
            }

            catch { return null; }
            finally { dialog.Dispose(); }
            return dialog.FileName;
        }

        ///--------------------------------
        /// <summary>
        /// パスの取得
        /// </summary>
        public static string Query(FileDialog dialog, string filter, string initialFilePath)
        {
            string directory = null;

            try
            {
                if ((directory = GetValidateDirectory
                     (initialFilePath)) != null)
                {
                    dialog.InitialDirectory = directory;
                }

                dialog.Filter = filter;
                dialog.RestoreDirectory = true;
                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return null;
                }

                //initialFilePath = Path.GetDirectoryName( dialog.FileName);
            }

            catch { return null; }
            finally { dialog.Dispose(); }
            return dialog.FileName;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// プロジェクトファイルの選択ダイアログ
    /// </summary>
    public class ProjectFileQuester : FileQuester
    {
        private const string _Filter = "Sound Project Files (*.{0})|*.{0}|All Files (*.*)|*.*";

        public static string Extension { get; set; }

        public static string Filter
        {
            get
            {
                return (Extension != null) ? string.Format(_Filter, Extension) : AllFilesFilter;
            }
        }

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの作成
        /// </summary>
        public static string CreateFilePath(string directory, string name)
        {
            string filePath = Path.Combine(directory, name);
            filePath += "." + Extension;

            return filePath.GetFullPath();
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のパスの取得
        /// </summary>
        public static string QueryLoad()
        {
            string filePath = null;

            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_File_OpenProject_Text;
            if ((filePath = Query(dialog, Filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }

        ///--------------------------------
        /// <summary>
        /// 書き込み用のパスの取得
        /// </summary>
        public static string QuerySave(string initialFileName)
        {
            FileDialog dialog = null;
            string filePath = null;

            dialog = new SaveFileDialog();
            dialog.FileName = initialFileName;
            if ((filePath = Query(dialog, Filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 別プラットフォームプロジェクトファイルの選択ダイアログ（インポート）
    /// </summary>
    public class AnotherPlatformProjectFileQuester : FileQuester
    {
        public static string Filter
        {
            get
            {
                return (
                        Resources.MessageResource.FileFilter_ProjectFilesforOtherPlatforms +
                        " (*.rspj)|*.rspj" +
                        AllFilesFilter
                        );
            }
        }

        ///
        public static event RequestFilterHandler RequestFilter;

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のパスの取得
        /// </summary>
        public static string QueryLoad()
        {
            string filePath = null;
            string filter = Filter;

            if (RequestFilter != null)
            {
                RequestFilterEventArgs args = new RequestFilterEventArgs();
                RequestFilter(args);
                filter = args.Filter;
            }

            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_File_ImportProject_Text;
            if ((filePath = Query(dialog, filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 別プラットフォームのバンクファイルの選択ダイアログ（インポート）
    /// </summary>
    public class AnotherPlatformBankFileQuester : FileQuester
    {
        public static string Filter
        {
            get
            {
                return (
                        Resources.MessageResource.FileFilter_BankFilesforOtherPlatforms +
                        " (*.rbnk;*.bnk)|*.rbnk;*.bnk" +
                        AllFilesFilter
                        );
            }
        }

        ///
        public static event RequestFilterHandler RequestFilter;

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のパスの取得
        /// </summary>
        public static string QueryLoad()
        {
            string filePath = null;
            string filter = Filter;

            if (RequestFilter != null)
            {
                RequestFilterEventArgs args = new RequestFilterEventArgs();
                RequestFilter(args);
                filter = args.Filter;
            }

            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_File_ImportBank_Text;
            if ((filePath = Query(dialog, filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 別プラットフォームのサウンドセットファイルの選択ダイアログ（インポート）
    /// </summary>
    public class AnotherPlatformSoundSetFileQuester : FileQuester
    {
        public static string Filter
        {
            get
            {
                return (
                        Resources.MessageResource.FileFilter_SoundSetFilesforOtherPlatforms +
                        " (*.rsst)|*.rsst" +
                        AllFilesFilter
                        );
            }
        }

        ///
        public static event RequestFilterHandler RequestFilter;

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のパスの取得
        /// </summary>
        public static string QueryLoad()
        {
            string filePath = null;
            string filter = Filter;

            if (RequestFilter != null)
            {
                RequestFilterEventArgs args = new RequestFilterEventArgs();
                RequestFilter(args);
                filter = args.Filter;
            }

            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_File_ImportSoundSet_Text;
            if ((filePath = Query(dialog, filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// サウンドセットファイルの選択ダイアログ
    /// </summary>
    public class SoundSetFileQuester : FileQuester
    {
        private const string _Filter = "Sound Set Files (*.{0})|*.{0}|All Files (*.*)|*.*";

        public static string Extension { get; set; }

        public static string Filter
        {
            get
            {
                return (Extension != null) ? string.Format(_Filter, Extension) : AllFilesFilter;
            }
        }

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの作成
        /// </summary>
        public static string CreateFilePath(string directory, string name)
        {
            string filePath = Path.Combine(directory, name);
            filePath += "." + Extension;

            return filePath.GetFullPath();
        }

        /// <summary>
        /// 読み込み用のパスの取得します。
        /// </summary>
        public static string QueryLoad()
        {
            return QueryLoad(true);
        }

        /// <summary>
        /// 読み込み用のパスの取得します。
        /// </summary>
        public static string QueryLoad(bool confirmFilePath)
        {
            string filePath = null;
            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_Project_AddSoundSet_Text;

            if (confirmFilePath != false)
            {
                dialog.FileOk += OnFileOk;
            }

            if ((filePath = Query(dialog, Filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static void OnFileOk(object sender, System.ComponentModel.CancelEventArgs e)
        {
            if (ProjectFilePathUtility.ConfirmFilePath(((OpenFileDialog)sender).FileName) == false)
            {
                e.Cancel = true;
                return;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// バンクファイルの選択ダイアログ
    /// </summary>
    public class BankFileQuester : FileQuester
    {
        private const string _Filter = "Bank Files (*.{0})|*.{0}|All Files (*.*)|*.*";

        public static string Extension { get; set; }

        public static string Filter
        {
            get
            {
                return (Extension != null) ? string.Format(_Filter, Extension) : AllFilesFilter;
            }
        }

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のバンクのパスの取得
        /// </summary>
        public static string QueryLoad()
        {
            string filePath = null;

            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = MessageResource.DialogTitle_File_OpenFile_Text;

            if ((filePath = Query(dialog, Filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }

        ///--------------------------------
        /// <summary>
        /// 書き込み用のバンクのパスの取得
        /// </summary>
        public static string QuerySave()
        {
            string filePath = null;

            if ((filePath = Query(new SaveFileDialog(), Filter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    /// シーケンスファイルの選択ダイアログ
    /// </summary>
    public class SeqFileQuester : FileQuester
    {
        public const string FilterSmf = "Standard MIDI Files (*.mid)|*.mid|All Files (*.*)|*.*";
        private const string FilterTextFormat =
            "Text Sequence Files (*.{0})|*.{0}|All Files (*.*)|*.*";

        public static string ExtensionText { get; set; }

        public static string FilterText
        {
            get
            {
                return (ExtensionText != null) ?
                    string.Format(FilterTextFormat, ExtensionText) : AllFilesFilter;
            }
        }

        ///--------------------------------
        private static string _InitialFilePath = null;

        ///--------------------------------
        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のシーケンスのパスの取得
        /// </summary>
        public static string QueryLoad(String fileFilter)
        {
            string filePath = null;

            if ((filePath = Query(new OpenFileDialog(), fileFilter, _InitialFilePath)) == null)
            {
                return null;
            }

            _InitialFilePath = Path.GetDirectoryName(filePath);
            return filePath;
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のシーケンスのパスの取得
        /// </summary>
        public static string QueryLoadSMF()
        {
            return QueryLoad(FilterSmf);
        }

        ///--------------------------------
        /// <summary>
        /// 読み込み用のシーケンスのパスの取得
        /// </summary>
        public static string QueryLoadText()
        {
            return QueryLoad(FilterText);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// 波形ファイルの選択ダイアログ
    /// </summary>
    public class WaveFileQuester : FileQuester
    {
        public const string Filter =
            "Wave Files (*.wav;*.aif;*.aiff)|*.wav;*.aif;*.aiff|All Files (*.*)|*.*";

        ///
        private static string _InitialFilePath = null;

        /// <summary>
        /// 初期ファイルパスの取得/設定
        /// </summary>
        public static string InitialFilePath
        {
            get { return _InitialFilePath; }
            set { _InitialFilePath = value; }
        }

        /// <summary>
        /// 読み込み用の波形のパスの取得
        /// </summary>
        public static string QueryLoad(Control owner)
        {
            return QueryLoad(owner, Filter);
        }

        /// <summary>
        /// 読み込み用の波形のパスの取得
        /// </summary>
        public static string QueryLoad(Control owner, string filter)
        {
            string result = null;

            QueryLoad(owner, null, filter, _InitialFilePath, out result);
            if (result != null)
            {
                _InitialFilePath = Path.GetDirectoryName(result);
            }

            return result;
        }

        /// <summary>
        /// 読み込み用の波形のパスの取得
        /// </summary>
        public static bool QueryLoad(Control owner, string title, string filter, string initialFilePath, out string result)
        {
            if (filter == null)
            {
                filter = Filter;
            }

            string directory = String.Empty;
            string fileName = String.Empty;
            if (string.IsNullOrEmpty(initialFilePath) == false)
            {
                if (Directory.Exists(initialFilePath) == true)
                {
                    directory = initialFilePath;
                }
                else if (File.Exists(initialFilePath) == true)
                {
                    fileName = Path.GetFileName(initialFilePath);
                    string dir = GetValidateDirectory(Path.GetDirectoryName(initialFilePath));
                    if (string.IsNullOrEmpty(dir) == false)
                    {
                        directory = dir;
                    }
                }
            }

            if (UsePreviewWaveChooseDialog != false)
            {
                WaveOpenFileDialog dialog = new WaveOpenFileDialog();

                try
                {
                    dialog.ValidatePreviewFilePathDelegate = ((path) => AACUtil.IsAACFile(path) == true || FileUtil.IsOpusFile(path) == true ? false : true);
                    dialog.Owner = owner;
                    if (string.IsNullOrEmpty(title) == false)
                    {
                        dialog.Title = title;
                    }
                    if (string.IsNullOrEmpty(filter) == false)
                    {
                        dialog.Filter = filter;
                    }
                    dialog.FilePath = fileName;
                    dialog.InitialDirectory = directory;

                    if (dialog.ShowDialog() != DialogResult.OK)
                    {
                        result = null;
                        return true;
                    }

                    result = dialog.FilePath;
                    return true;
                }

                catch
                {
                    result = null;
                    return true;
                }
                finally
                {
                    dialog.Dispose();
                }
            }
            else
            {
                OpenFileDialog dialog = new OpenFileDialog();

                try
                {
                    //dialog.Owner = this;
                    if (string.IsNullOrEmpty(title) == false)
                    {
                        dialog.Title = title;
                    }
                    if (string.IsNullOrEmpty(filter) == false)
                    {
                        dialog.Filter = filter;
                    }
                    dialog.FileName = fileName;
                    dialog.InitialDirectory = directory;

                    if (dialog.ShowDialog() != DialogResult.OK)
                    {
                        result = null;
                        return true;
                    }

                    result = dialog.FileName;
                    return true;
                }

                catch
                {
                    result = null;
                    return true;
                }
                finally
                {
                    dialog.Dispose();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static bool UsePreviewWaveChooseDialog
        {
            get
            {
                XmlOptionsApplicationGeneral general = null;
                general = ApplicationBase.Instance.AppConfiguration.Options.Application.General;
                return general.UsePreviewWaveChooseDialog;
            }
        }
    }
}
