﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Profile;

    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using Measure = NintendoWare.SoundMaker.Profile.ProjectProfileService.MeasureWaveReferenceCount;

    /// <summary>
    ///
    /// </summary>
    public partial class MeasureWaveReferenceCountPanel : DockingPage, IToolWindowPage
    {
        #region ** 固定値

        public const string PageName = "MeasureWaveReferenceCount";

        #endregion

        private Measure.ReferenceFileData referenceFileData = null;

        private CommonListAdapter fileListAdapter = null;
        private ComponentList fileComponentList = new ComponentList(null);

        private CommonListAdapter instrumentListAdapter = null;
        private ComponentList instrumentComponentList = new ComponentList(null);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MeasureWaveReferenceCountPanel() : base(PageName)
        {
            InitializeComponent();

            //
            this.Text = MessageResource.ToolWindowName_MeasureWaveReferenceCount;

            Bitmap icon = ImageResource.BitmapIconMeasureWaveReferenceCountWindow;
            icon.MakeTransparent(UIServiceBase.TransparentColor);
            this.Image = icon;

            //
            this.fileListAdapter = new CommonListAdapter(this.ListDecorationEvaluator);

            this.filePathList.ReadOnly = true;
            this.filePathList.AllowDragItem = false;
            this.filePathList.HeaderHeight = 28;
            this.filePathList.SelectChanged += OnSelectChanged;

            this.filePathList.PartDrawers.Replace(new ListPartRowHeaderDrawer());
            this.filePathList.SubItems.Replace(new ListSubItemFilePathWithoutButton());

            this.filePathList.HeaderSource =
                UIService.DefaultHeaderAdapters[ListTraits.ListName_MeasureReferenceCountFilePath];
            this.filePathList.ItemsSource = this.fileListAdapter;

            //
            this.instrumentListAdapter = new CommonListAdapter(this.ListDecorationEvaluator);

            this.instrumentList.ReadOnly = true;
            this.instrumentList.AllowDragItem = false;
            this.instrumentList.HeaderHeight = 28;
            this.instrumentList.ItemDoubleClicked += OnItemDoubleClicked;

            this.instrumentList.PartDrawers.Replace(new ListPartRowHeaderDrawer());
            this.instrumentList.SubItems.Replace(new ListSubItemFilePathWithoutButton());

            this.instrumentList.HeaderSource =
                UIService.DefaultHeaderAdapters[ListTraits.ListName_MeasureReferenceCountInstrument];
            this.instrumentList.ItemsSource = this.instrumentListAdapter;

            //
            WaveReferenceCountMeasurer.RequireGetReferenceFileData +=
                delegate (object sender, EventArgs e)
                    {
                        this.referenceFileData = WaveReferenceCountMeasurer.GetReferenceFileData();
                        UpdateFileList();
                        UpdateInstrumentList(null);
                    };

            //
            ProjectService.Closed +=
                delegate (object sender, EventArgs e)
                    {
                        Reset();
                    };
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        ///
        /// </summary>
        public void Reset()
        {
            this.fileComponentList.Clear();
            this.fileListAdapter.AddItems(this.fileComponentList);

            this.instrumentComponentList.Clear();
            this.instrumentListAdapter.AddItems(this.instrumentComponentList);
        }

        #region ** プロパティ

        /// <summary>
        ///
        /// </summary>
        private SoundProjectService ProjectService
        {
            get
            {
                return ApplicationBase.Instance.ProjectService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private FormsUIService UIService
        {
            get
            {
                return FormsApplication.Instance.UIService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private ProjectProfileService ProjectProfileService
        {
            get
            {
                return FormsApplication.Instance.ProjectProfileService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Measure WaveReferenceCountMeasurer
        {
            get
            {
                return ProjectProfileService.WaveReferenceCountMeasurer;
            }
        }

        #endregion

        /// <summary>
        /// ファイルリストの更新します。
        /// </summary>
        private void UpdateFileList()
        {
            this.fileComponentList.Clear();
            foreach (Measure.ReferenceFileData.FileData fileData in
                     this.referenceFileData.FileDatas.Values)
            {
                ListComponent listComponent = new ListComponent();
                listComponent.FilePath = fileData.WaveFilePath;
                listComponent.ReferenceCount = fileData.Count;
                this.fileComponentList.Add(listComponent);
            }

            this.fileListAdapter.AddItems(this.fileComponentList);

            this.filePathList.HeaderCtrl.SetSortState
                (ListTraits.ColumnName_ReferenceCount, SortOrder.Descending);
            this.filePathList.Sort();
        }

        /// <summary>
        /// インストルメントリストの更新します。
        /// </summary>
        private void UpdateInstrumentList(string filePath)
        {
            this.instrumentComponentList.Clear();

            if (filePath != null)
            {
                Measure.ReferenceFileData.FileData fileData = null;
                if (this.referenceFileData.FileDatas.TryGetValue(filePath, out fileData) == true)
                {
                    foreach (Measure.ReferenceFileData.FileData.InstrumentData instrumentData in
                             fileData.InstrumentDatas.Values)
                    {
                        ListComponent bbb = new ListComponent();
                        bbb.Name = instrumentData.Name;
                        bbb.ReferenceCount = instrumentData.Count;
                        bbb.FilePath = instrumentData.BankFilePath;
                        this.instrumentComponentList.Add(bbb);
                    }
                }
            }

            this.instrumentListAdapter.AddItems(this.instrumentComponentList);

            this.instrumentList.HeaderCtrl.SetSortState
                (ListTraits.ColumnName_ReferenceCount, SortOrder.Descending);
            this.instrumentList.Sort();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnClick_MeasureWaveReferenceCount(object sender, EventArgs e)
        {
            QueryAndMeasureWaveReferenceCountHandler.Execute(this.ProjectService);
        }

        /// <summary>
        /// アイテムの選択が変更された時に呼ばれます。
        /// </summary>
        private void OnSelectChanged(object sender, EventArgs e)
        {
            CommonListItem[] items = this.filePathList.GetSelectedItems();
            string filePath = null;

            if (items.Length == 1)
            {
                ListComponent listComponent = items[0].Target as ListComponent;
                filePath = listComponent.FilePath;
            }

            UpdateInstrumentList(filePath);
        }

        /// <summary>
        /// アイテムをダブルクリックした時に呼ばれます。
        /// </summary>
        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ComponentListItem item = e.Item as ComponentListItem;
            ListComponent listComponent = item.Target as ListComponent;

            UIService.MainWindow.ShowBankPageByFilePath
                (listComponent.FilePath, listComponent.Name);
        }

        #region ** メソッド

        #endregion

        /// <summary>
        /// 行ヘッダ
        /// </summary>
        private class ListPartRowHeaderDrawer : ListPartCommonRowHeaderDrawer
        {
            /// <summary>
            ///
            /// </summary>
            protected override bool DoShowErrorIcon(ListDrawDescriptor desc)
            {
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public class ListSubItemFilePathWithoutButton : ListSubItemFilePath
        {
            protected override IListPart[] GetPartsInternal(ListDrawDescriptor desc)
            {
                return new[] { CreateDefaultPart() };
            }
        }

        /// <summary>
        /// リスト用のComponent
        /// </summary>
        class ListComponent : Component
        {
            /// <summary>
            ///
            /// </summary>
            public string FilePath
            {
                get { return GetFilePathValue(ProjectParameterNames.FilePath); }
                set { SetFilePathValue(ProjectParameterNames.FilePath, value); }
            }

            /// <summary>
            ///
            /// </summary>
            public int ReferenceCount
            {
                get { return GetIntValue("ReferenceCount"); }
                set { SetIntValue("ReferenceCount", value); }
            }

            /// <summary>
            /// パラメータを初期化します。
            /// </summary>
            protected override void InitializeParameters()
            {
                base.InitializeParameters();

                SetInstanceFilePathValue(ProjectParameterNames.FilePath, string.Empty);
                SetInstanceIntValue("ReferenceCount", 0, 0, Int32.MaxValue);
            }
        }
    }
}
