﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Utilities;

    /// <summary>
    /// サウンドパラメータパネル
    /// </summary>
    public partial class SoundParameterPanel : UserControl
    {
        private Component[] targetItems = null;
        private List<NameSetList> targetParameterNames = new List<NameSetList>();

        public event OperationExecutedEventHandler OperationExecuted;

        ///
        public class NameSet
        {
            public string Name { get; set; }
            public string Text { get; set; }

            public NameSet(string name, string text)
            {
                Name = name;
                Text = text;
            }
        }

        ///
        private class NameSetList : List<NameSet>
        {
            public Type TargetType { get; set; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SoundParameterPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        ///
        /// </summary>
        private MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        /// サウンドを取得/設定します。
        /// </summary>
        public Component[] Items
        {
            get { return this.targetItems; }
            set
            {
                if (this.targetItems != null &&
                    this.targetItems.Length == 1)
                {
                    this.targetItems[0].Parameters.ParameterValueChanged -= OnParameterValueChanged;
                }

                bool buildUI = true;
                if (this.targetItems != null && value != null)
                {
                    if (this.targetItems[0].GetType() == value[0].GetType())
                    {
                        buildUI = false;
                    }
                }

                this.targetItems = value;

                if (this.targetItems != null &&
                    this.targetItems.Length == 1)
                {
                    this.targetItems[0].Parameters.ParameterValueChanged += OnParameterValueChanged;
                }

                if (buildUI != false)
                {
                    BuildUI();
                }
                else
                {
                    AttachToUI();
                }

                UpdateUI();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void AddParameterName(Type type, string name, string text)
        {
            NameSetList list = GetNameSetList(type);
            if (list == null)
            {
                list = new NameSetList();
                list.TargetType = type;
                this.targetParameterNames.Add(list);
            }

            // 2行になっているテキストを1行にします。
            text = text.Replace("\r\n", String.Empty);

            list.Add(new NameSet(name, text));
        }

        /// <summary>
        ///
        /// </summary>
        public void InsertParameterName(Type type, string previousName, string name, string text)
        {
            NameSetList list = GetNameSetList(type);
            if (list == null)
            {
                list = new NameSetList();
                list.TargetType = type;
                this.targetParameterNames.Add(list);
            }

            // 2行になっているテキストを1行にします。
            text = text.Replace("\r\n", String.Empty);

            //
            int index = 0;
            foreach (NameSet nameSet in list)
            {
                index++;
                if (nameSet.Name == previousName)
                {
                    break;
                }
            }

            list.Insert(index, new NameSet(name, text));
        }

        /// <summary>
        ///
        /// </summary>
        public OperationHistory OperationHistory { get; set; }

        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            FitControlSizeInFlowLayout();
        }

        /// <summary>
        ///
        /// </summary>
        private NameSetList GetNameSetList(Type targetType)
        {
            foreach (NameSetList list in this.targetParameterNames)
            {
                if (TypeComparer.HasAncestor(list.TargetType, targetType) != false)
                {
                    return list;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private void AttachToUI()
        {
            foreach (UserControl control in this.flowLayoutPanel.Controls)
            {
                IParameterEditPanel panel = control as IParameterEditPanel;
                Debug.Assert(panel != null, "Control is not IParameterEditPanel");
                panel.Components = this.targetItems;
            }
        }

        /// <summary>
        /// UIの構築を行います。
        /// </summary>
        private void BuildUI()
        {
            RemoveAllPanels();

            //
            if (this.targetItems == null ||
                this.targetItems.Length <= 0)
            {
                return;
            }

            //
            Component targetItem = this.targetItems[0];
            NameSetList list = GetNameSetList(targetItem.GetType());
            if (list == null)
            {
                return;
            }

            //
            foreach (NameSet nameSet in list)
            {
                IParameterValue value = targetItem.Parameters[nameSet.Name];

                // Int
                if (value is IntParameterValue)
                {
                    IntParameterValue intValue = value as IntParameterValue;

                    IntParameterEditPanel panel = new IntParameterEditPanel();
                    panel.Components = this.targetItems;
                    panel.ParameterName = nameSet.Name;
                    panel.Label = nameSet.Text;
                    panel.Minimum = intValue.MinValue;
                    panel.Maximum = intValue.MaxValue;
                    panel.Value = intValue.Value;
                    panel.ValueChanging += OnIntValueChanging;
                    panel.ValueChanged += OnIntValueChanged;

                    this.flowLayoutPanel.Controls.Add(panel);
                }

                // Float
                if (value is FloatParameterValue)
                {
                    FloatParameterValue floatValue = value as FloatParameterValue;

                    FloatParameterEditPanel panel = new FloatParameterEditPanel();
                    panel.Components = this.targetItems;
                    panel.ParameterName = nameSet.Name;
                    panel.Label = nameSet.Text;
                    panel.Minimum = floatValue.MinValue;
                    panel.Maximum = floatValue.MaxValue;
                    panel.Value = floatValue.Value;
                    panel.ValueChanging += OnFloatValueChanging;
                    panel.ValueChanged += OnFloatValueChanged;

                    this.flowLayoutPanel.Controls.Add(panel);
                }

                // Bool
                if (value is BoolParameterValue)
                {
                    BoolParameterValue boolValue = value as BoolParameterValue;

                    BoolParameterEditPanel panel = new BoolParameterEditPanel();
                    panel.Components = this.targetItems;
                    panel.ParameterName = nameSet.Name;
                    panel.Label = nameSet.Text;
                    panel.Value = boolValue.Value;
                    panel.ValueChanged += OnBoolValueChanged;

                    this.flowLayoutPanel.Controls.Add(panel);
                }
            }

            FitControlSizeInFlowLayout();
        }

        /// <summary>
        /// 全てのパネルの破棄を行います。
        /// </summary>
        private void RemoveAllPanels()
        {
            foreach (UserControl control in this.flowLayoutPanel.Controls)
            {
                if (control is IntParameterEditPanel)
                {
                    IntParameterEditPanel panel = control as IntParameterEditPanel;
                    panel.ValueChanging -= OnIntValueChanging;
                    panel.ValueChanged -= OnIntValueChanged;
                }

                if (control is FloatParameterEditPanel)
                {
                    FloatParameterEditPanel panel = control as FloatParameterEditPanel;
                    panel.ValueChanging -= OnFloatValueChanging;
                    panel.ValueChanged -= OnFloatValueChanged;
                }

                if (control is BoolParameterEditPanel)
                {
                    BoolParameterEditPanel panel = control as BoolParameterEditPanel;
                    panel.ValueChanged -= OnBoolValueChanged;
                }
            }
            this.flowLayoutPanel.Controls.Clear();
        }

        /// <summary>
        /// UIの更新を行います。
        /// </summary>
        private void UpdateUI()
        {
            string name = string.Empty;
            if (this.targetItems != null)
            {
                name = this.targetItems[0].Name;
                if (this.targetItems[0] is StreamSoundTrackBase)
                {
                    StreamSoundTrackBase track = this.targetItems[0] as StreamSoundTrackBase;
                    name = string.Format("{0}/Track {1} ({2})",
                                         track.Parent.Name,
                                         track.Parent.Children.IndexOf(track),
                                         Path.GetFileName(track.FilePath));
                }
            }

            tbx_Name.Enabled = string.IsNullOrEmpty(name) == false;
            tbx_Name.Text = name;

            foreach (UserControl control in this.flowLayoutPanel.Controls)
            {
                IParameterEditPanel panel = control as IParameterEditPanel;
                if (panel != null)
                {
                    panel.UpdateUI();
                }
            }

#if false
            if (InvokeRequired == true)
            {
                Invoke(new UpdatePanelDelegate(UpdatePanel));
                return;
            }
            UpdateControlsStatus();
            UpdateControlsValue();
            UpdateReadOnlyControls();
#endif
        }

        /// <summary>
        /// FlowLayoutPanel内のコントロールサイズをFlowLayoutPanelのサイズに合わせます。
        /// </summary>
        private void FitControlSizeInFlowLayout()
        {
            int totalHeight = this.flowLayoutPanel.Location.Y;
            int panelClientWidth = this.flowLayoutPanel.ClientSize.Width;

            foreach (Control control in this.flowLayoutPanel.Controls)
            {
                control.Size = new System.Drawing.Size(panelClientWidth, control.Height);
                totalHeight += control.Height;
                totalHeight += control.Margin.Vertical;
            }

            //
            System.Drawing.Size size = new System.Drawing.Size(this.Size.Width, totalHeight);
            //this.MinimumSize = size;
            this.Size = size;
        }

        /// <summary>
        ///
        /// </summary>
        private void OnParameterValueChanged(object sender, EventArgs e)
        {
            if (InvokeRequired != false)
            {
                Invoke((MethodInvoker)delegate
              {
                  UpdateUI();
                  MainWindow.BuildCommandUI();
              });
            }
            else
            {
                UpdateUI();
                MainWindow.BuildCommandUI();
            }
        }

        ///
        public event ValueChangingEventHandler ValueChanging;
        public event ValueChangedEventHandler ValueChanged;

        /// <summary>
        ///
        /// </summary>
        private void OnIntValueChanging(object sender, ValueChangeEventArgs e)
        {
            if (ValueChanging != null)
            {
                ValueChanging(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFloatValueChanging(object sender, ValueChangeEventArgs e)
        {
            if (ValueChanging != null)
            {
                ValueChanging(sender, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnIntValueChanged(object sender, ValueChangeEventArgs e)
        {
            SetValue(this.targetItems, e.Name, e.Value);

            //
            if (ValueChanged != null)
            {
                ValueChanged
                    (this,
                      new IntValueChangedEventArgs(this.targetItems, e.Name, (int)e.Value));
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnFloatValueChanged(object sender, ValueChangeEventArgs e)
        {
            SetValue(this.targetItems, e.Name, e.Value);

            //
            if (ValueChanged != null)
            {
                ValueChanged
                    (this,
                      new FloatValueChangedEventArgs(this.targetItems, e.Name, (float)e.Value));
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBoolValueChanged(object sender, ValueChangeEventArgs e)
        {
            SetValue(this.targetItems, e.Name, e.Value);

            //
            if (ValueChanged != null)
            {
                ValueChanged
                    (this,
                      new BoolValueChangedEventArgs(this.targetItems, e.Name, (bool)e.Value));
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationExecuted(Operation operation)
        {
            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        /// <summary>
        ///
        /// </summary>
        private void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(e.Operation);
            }

            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValue(Component[] components, string name, object value)
        {
            foreach (Component component in components)
            {
                if (component.Parameters.ContainsKey(name) == false)
                {
                    continue;
                }

                SetParameterOperation operation =
                    new SetParameterOperation(component.Parameters, name, value);
                operation.Execute();
                OnOperationExecuted(operation);
            }
        }
    }
}
