﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;
    using Framework.Commands;
    using Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Projects;
    using SoundFoundation.Commands;
    using SoundFoundation.Core;
    using SoundFoundation.Core.Parameters;
    using SoundFoundation.FileFormats.Wave;
    using SoundFoundation.Operations;
    using SoundFoundation.Parameters;
    using SoundFoundation.Projects;
    using SoundFoundation.Utilities;
    using SoundFoundation.Windows.Forms;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public partial class PercussionListPanel : UserControl, IInnerBankPanel
    {
        private PercussionListAdapter _ListAdapter = null;

        private ToolStripAdapter _percussionListMenuAdapter;
        private ContextMenuStrip _ContextMenuStrip = null;

        private Instrument _Instrument = null;

        private BankPanel _BankPanel = null;

        private ToolStripMenuItem _HeaderContextMenuItem_AddPresetListColumns = null;
        private ToolStripMenuItem _HeaderContextMenuItem_ApplyPresetListColumns = null;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PercussionListPanel(BankPanel bankPanel, ContextMenuStrip contextMenuStrip)
        {
            InitializeComponent();

            _BankPanel = bankPanel;
            _ContextMenuStrip = contextMenuStrip;

            //
            _ListAdapter = new PercussionListAdapter(this.ListDecorationEvaluator);
            _ListAdapter.OperationHistory = _BankPanel.BankDocument.OperationHistory;

            _ListCtrl.HeaderHeight = 28;
            _ListCtrl.ItemsSource = _ListAdapter;
            _ListCtrl.SelectChanged += OnSelectChangedListCtrl;
            _ListCtrl.ItemDoubleClicked += OnItemDoubleClicked;
            _ListCtrl.QueryFileDropped += OnQueryFileDropped;
            _ListCtrl.FileDropped += OnFileDropped;
            _ListCtrl.EditBegan += OnEditBegan;
            _ListCtrl.EditEnded += OnEditEnded;
            _ListCtrl.AttachedContextMenuStrip = _ContextMenuStrip;
            _ListCtrl.Enter += OnEnter;
            _ListCtrl.Leave += OnLeave;

            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;
            _HeaderContextMenuItem_ApplyPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_ApplyPresetListColumns);
            _ListCtrl.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            _ListCtrl.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Instrument Instrument
        {
            get { return _Instrument; }
            set
            {
                if (_Instrument != null)
                {
                    _Instrument.Children.CollectionChanged -= OnCollectionChanged;
                }

                _Instrument = value;

                if (_Instrument != null)
                {
                    _Instrument.Children.CollectionChanged += OnCollectionChanged;
                }

                _ListAdapter.SetupPercussionList(_Instrument);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public CommonListCtrl PercussionList
        {
            get
            {
                return _ListCtrl;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public CommonListCtrl ActiveListCtrl
        {
            get
            {
                if (_ListCtrl.ContainsFocus) { return _ListCtrl; }
                return null;
            }
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        ///
        /// </summary>
        public PercussionListAdapter ListAdapter
        {
            get
            {
                return this._ListAdapter;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public PercussionListAdapter ActiveListAdapter
        {
            get
            {
                if (_ListCtrl.ContainsFocus) { return _ListAdapter; }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public void UpdatedClipboard()
        {
            _ListCtrl.ClearDashLine();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RedrawControls()
        {
            _ListCtrl.Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdatedOptions()
        {
            _ListCtrl.UpdatePartDrawerOptions();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool CanSplitKeyRegion()
        {
            CommonListCtrl listCtrl = null;

            if ((listCtrl = ActiveListCtrl) == null)
            {
                return false;
            }

            ComponentListItem[] items = listCtrl.GetSelectedItems();

            if (items.Length <= 0)
            {
                return false;
            }

            foreach (PercussionListItem item in items)
            {
                if (item.KeyRegion == null ||
                    item.Shadow == false)
                {
                    return false;
                }
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool SplitKeyRegion()
        {
            List<int> list = new List<int>();
            CommonListCtrl listCtrl = null;

            if ((listCtrl = ActiveListCtrl) == null)
            {
                return false;
            }

            foreach (PercussionListItem item in listCtrl.GetSelectedItems())
            {
                if (item.Shadow == false)
                {
                    return false;
                }
                list.Add(item.Index);
            }

            ActiveListAdapter.DivideKeyRegion(Instrument, list.ToArray());
            return true;

        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            _ListCtrl.UpdateInfos();
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public void UpdatePresetListColumns()
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.Shown()
        {
            _ListAdapter.UpdatePercussionList(_Instrument);
            _ListCtrl.UpdateInfos();
            SelectChangedPercussionList();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.UpdateReadOnly(bool readOnly)
        {
            _ListCtrl.ReadOnly = readOnly;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.CommandExecuted(Command command)
        {
            ListCtrl listCtrl = null;

            if ((listCtrl = ActiveListCtrl) != null)
            {
                if (command == EditCommands.CopyCell)
                {
                    listCtrl.SetDashLineBySelectedSubItem();
                }
            }

            _ListCtrl.Invalidate();

            MainWindow.BuildCommandUI();
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.UndoExecuted()
        {
            _ListAdapter.UpdatePercussionList(_Instrument);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void IInnerBankPanel.RedoExecuted()
        {
            _ListAdapter.UpdatePercussionList(_Instrument);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected BankDocument BankDocument
        {
            get { return _BankPanel.BankDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {

                case NotifyCollectionChangedAction.Add:
                    foreach (KeyRegion keyRegion in e.NewItems)
                    {
                        _ListAdapter.ReplacePercussionList(keyRegion);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (KeyRegion keyRegion in e.OldItems)
                    {
                        _ListAdapter.ReplacePercussionList(_Instrument,
                                                            keyRegion.KeyMin, keyRegion.KeyMax);
                    }
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateStatusBar(CommonListCtrl listCtrl)
        {
            CommonListItem[] items = listCtrl.GetSelectedItems();
            MainWindow.UpdateStatusText(items.Length == 1 ? items[0].FilePath : String.Empty);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectChangedListCtrl(object sender, EventArgs e)
        {
            SelectChangedPercussionList();
            UpdateStatusBar(_ListCtrl);
            MainWindow.BuildCommandUI();
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SelectChangedPercussionList()
        {
            HashSet<Component> dictionary = new HashSet<Component>();

            foreach (PercussionListItem item in _ListCtrl.GetSelectedItems())
            {
                if (dictionary.Contains(item.Target) == false)
                {
                    dictionary.Add(item.Target);
                }
            }

            VelocityRegion[] velRegions = null;

            velRegions = dictionary.Cast<VelocityRegion>().ToArray();
            ParameterPanel.SetBankPanel(_BankPanel);
            if (velRegions.Length == 1 && velRegions[0] != null)
            {
                ParameterPanel.Instrument = velRegions[0].Parent.Parent as Instrument;
                ParameterPanel.VelocityRegion = velRegions[0];
            }
            else
            {
                ParameterPanel.VelocityRegion = null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップできるかどうか判断する時に呼ばれる
        /// </summary>
        private void OnQueryFileDropped(object sender, FileDroppedEventArgs e)
        {
            if (BankDocument.IsReadOnly != false ||
                e.FilePaths.Length != 1)
            {
                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDropped(object sender, FileDroppedEventArgs e)
        {
            BeginInvoke(new ListCtrl.FileDroppedEventHandler(OnFileDroppedInternal),
                        new object[] { sender, e });
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDroppedInternal(object sender, FileDroppedEventArgs e)
        {
            if (e.DropTarget == null || e.FilePaths.Length != 1)
            {
                return;
            }

            MainWindow.Activate();

            PercussionListItem item = e.DropTarget as PercussionListItem;
            string filePath = e.FilePaths[0];
            Operation operation = null;

            if (item.KeyRegion == null)
            {
                return;
            }

            try
            {
                WaveFileReader.CreateInstance(filePath);

                if (item.Shadow == false)
                {
                    operation = new SetParameterOperation
                        (item.Target.Parameters,
                          ProjectParameterNames.FilePath, filePath);

                    operation.Execute();
                    BankDocument.OperationHistory.AddOperation(operation);
                }
                else
                {
                    _ListAdapter.DivideKeyRegion(item.Instrument, item.Index, filePath);
                }
            }
            catch
            {
                string message = Resources.MessageResource.Message_CanNotUseUnsupportedFileWithAAC;

                TextDisplayMessageBox dialog = new TextDisplayMessageBox
                    (message, filePath,
                      TextDisplayMessageBoxStyle.OKButton);
                dialog.ShowDialog();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            UpdateStatusBar(_ListCtrl);
            MainWindow.EnableAllCommands(true);
            MainWindow.RedrawPanels();
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnLoad(object sender, EventArgs e)
        {
            InitializeMenu();

            _ListCtrl.HeaderSource = _BankPanel.HeaderAdapters[ListTraits.ListName_Percussion];

            _ListCtrl.GetItemSelecteds().Clear();
            _ListCtrl.InitializeCaret();
        }

        ///--------------------------------
        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != _percussionListMenuAdapter) { return; }

            _percussionListMenuAdapter = ToolStripAdapter.FromToolStrip
                (FormsApplication.Instance.CommandService, _ContextMenuStrip);
            _percussionListMenuAdapter.CommandTarget = FormsApplication.Instance;
            _percussionListMenuAdapter.CommandExecuted += OnPercussionListCommandExecuted;

            BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUI()
        {
            if (null != _percussionListMenuAdapter)
            {
                _percussionListMenuAdapter.BuildUI();
            }
        }

        /// <summary>
        /// パーカッションリストコマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnPercussionListCommandExecuted(object sender, CommandEventArgs e)
        {
            ((IInnerBankPanel)this).CommandExecuted(e.Command);
        }

        ///--------------------------------
        /// <summary>
        /// リスト項目関連
        /// </summary>
        private void OnHeaderMenuItemAddPresetListColumnsClick(object sender, EventArgs e)
        {
            string listName = ListTraits.ListName_Percussion;
            ListHeaderAdapterDictionary headerAdapters = _BankPanel.HeaderAdapters as ListHeaderAdapterDictionary;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.AddPresetListColumns(listName, applier);
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnHeaderMenuItemApplyPresetListColumnsClick(object sender, EventArgs e)
        {
            ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
            string listName = ListTraits.ListName_Percussion;
            ListHeaderAdapterDictionary headerAdapters = _BankPanel.HeaderAdapters as ListHeaderAdapterDictionary;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(headerAdapters);

            ListHeaderHelper.ApplyPresetListColumns(applier, listName, menuItem.Text);
        }

        private void OnHeaderCtrlContextMenuOpening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            string listName = ListTraits.ListName_Percussion;

            ListHeaderHelper.CreateMeunApplyPresetListColumns(_HeaderContextMenuItem_ApplyPresetListColumns, listName, OnHeaderMenuItemApplyPresetListColumnsClick);
        }

        private void OnEnter(object sender, EventArgs e)
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnLeave(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void UpdatePresetListColumnsSplitButton()
        {
            string listName = ListTraits.ListName_Percussion;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// パーカッションリスト用のアダプタ
    /// </summary>
    public class PercussionListAdapter : CommonListAdapter
    {
        public PercussionListAdapter(CommonListDecorationEvaluator decorationEvaluator)
            : base(decorationEvaluator)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetupPercussionList(Instrument instrument)
        {
            if (instrument == null)
            {
                Items.Clear();
                return;
            }

            //
            ComponentListItem item = null;
            int currentKey = 0;
            int minimumKey = 0;
            int maximumKey = 0;

            SuspendUpdatedEvent();
            Items.Clear();
            InterlockComponent = false;

            foreach (KeyRegion keyRegion in instrument.Children)
            {

                minimumKey = keyRegion.KeyMin;
                maximumKey = keyRegion.KeyMax;

                //
                while (currentKey < minimumKey)
                {
                    Items.Add(new PercussionListItem(currentKey, instrument));
                    currentKey++;
                }

                //
                while (currentKey <= maximumKey)
                {
                    item = new PercussionListItem(currentKey, instrument, keyRegion,
                                                   currentKey != minimumKey);
                    Items.Add(item);
                    currentKey++;
                }
            }

            //
            while (currentKey < 128)
            {
                Items.Add(new PercussionListItem(currentKey, instrument));
                currentKey++;
            }

            //
            InterlockComponent = true;
            ResumeUpdatedEvent();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdatePercussionList(Instrument instrument)
        {
            if (instrument == null)
            {
                return;
            }

            ReplacePercussionList(instrument, 0, 127);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ReplacePercussionList(KeyRegion keyRegion)
        {
            PercussionListItem item = null;
            int minimumKey = keyRegion.KeyMin;
            int maximumKey = keyRegion.KeyMax;

            for (int key = minimumKey; key <= maximumKey; key++)
            {
                item = Items[key] as PercussionListItem;
                item.Replace(keyRegion, key != minimumKey);
            }

            if (maximumKey + 1 < 128)
            {
                item = Items[maximumKey + 1] as PercussionListItem;
                item.Replace(item.KeyRegion, false);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ReplacePercussionList(Instrument instrument, int minimumKey, int maximumKey)
        {
            PercussionListItem item = null;
            KeyRegion keyRegion = null;
            bool shadow = false;

            for (int key = minimumKey; key <= maximumKey; key++)
            {
                item = Items[key] as PercussionListItem;

                keyRegion = FindKeyRegion(instrument, key);
                shadow = keyRegion != null ? keyRegion.KeyMin != key : false;

                item.Replace(keyRegion, shadow);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DivideKeyRegion(Instrument instrument, int[] keys)
        {
            BeginTransaction();

            foreach (int key in keys)
            {
                if (DivideKeyRegionInternal(instrument, key, null) == false)
                {
                    CancelTransaction();
                    return;
                }
            }

            EndTransaction();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DivideKeyRegion(Instrument instrument, int key)
        {
            DivideKeyRegion(instrument, key, null);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DivideKeyRegion(Instrument instrument, int key, string filePath)
        {
            BeginTransaction();

            if (DivideKeyRegionInternal(instrument, key, filePath) == false)
            {
                CancelTransaction();
                return;
            }

            EndTransaction();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool DivideKeyRegionInternal(Instrument instrument, int key, string filePath)
        {
            Operation operation = null;
            KeyRegion keyRegion = null;
            KeyRegion newKeyRegion = null;
            VelocityRegion newVelRegion = null;
            Component parent = null;
            Component nextSibling = null;
            int index = 0;

            if ((keyRegion = FindKeyRegion(instrument, key)) == null)
            {
                return false;
            }

            newKeyRegion = DuplicateKeyRegion(keyRegion);
            newKeyRegion.KeyMin = key;
            newKeyRegion.KeyMax = keyRegion.KeyMax;

            if (filePath != null)
            {
                newVelRegion = newKeyRegion.Children[0] as VelocityRegion;
                newVelRegion.FilePath = filePath;
            }

            //
            operation = new SetParameterOperation(keyRegion.Parameters,
                                                   ProjectParameterNames.KeyRegion.KeyMax, key - 1);
            operation.Execute();
            OnOperationExecuted(operation);

            //
            parent = keyRegion.Parent;
            if ((index = parent.Children.IndexOf(keyRegion) + 1) < parent.Children.Count)
            {
                nextSibling = parent.Children[index];
            }

            operation = new InsertComponentOperation(parent, nextSibling,
                                                        new[] { newKeyRegion });
            operation.Execute();
            OnOperationExecuted(operation);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegion DuplicateKeyRegion(KeyRegion keyRegion)
        {
            KeyRegion newKeyRegion = new KeyRegion();
            VelocityRegion newVelRegion = null;

            foreach (VelocityRegion velRegion in keyRegion.Children)
            {
                newVelRegion = new VelocityRegion();
                newVelRegion.VelocityMin = velRegion.VelocityMin;
                newVelRegion.VelocityMax = velRegion.VelocityMax;
                newVelRegion.FilePath = velRegion.FilePath;
                newVelRegion.OriginalKey = velRegion.OriginalKey;

                newKeyRegion.Children.Add(newVelRegion);
            }

            return newKeyRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegion FindKeyRegion(Instrument instrument, int key)
        {
            foreach (KeyRegion keyRegion in instrument.Children)
            {
                if (key >= keyRegion.KeyMin && key <= keyRegion.KeyMax)
                {
                    return keyRegion;
                }
            }
            return null;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// パーカッションリスト用のリストコントロール
    /// </summary>
    public class PercussionListCtrl : CommonListCtrl
    {
        private static Brush _TextBrushSt = null;
        private static Brush _TextBrushSf = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        static PercussionListCtrl()
        {
            _TextBrushSt = new SolidBrush(Color.FromArgb(192, 192, 192));
            _TextBrushSf = new SolidBrush(Color.FromArgb(192, 192, 192));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public PercussionListCtrl()
        {
            HeaderCtrl.SortEnabled = false;
            AllowDragItem = false;
            BackgroundDrawer = new PercussionListBackgroundDrawer();

            //AutoDropFiles = false;
            DecideFileDropPosition = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void SetupDrawDescriptor(IListItem listItem, string name, ListDrawDescriptor desc)
        {
            PercussionListItem item = listItem as PercussionListItem;
            Debug.Assert(item != null, "listItem is not PercussionListitem");

            if (item.Shadow != false && name != "Key")
            {
                desc.TextBrush = desc.Selected != false ? _TextBrushSt : _TextBrushSf;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムが参照している Componentを取得
        /// </summary>
        public override Component[] GetComponentsBySelectedItem()
        {
            return SelectedItems
                .Cast<PercussionListItem>() /*.Where( i => i.Shadow == false )*/
                .Select(i => i.KeyRegion).ToArray();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class PercussionListBackgroundDrawer : ListBackgroundDrawer
    {
        private static Brush _HeaderBrush = null;
        private static Brush _HeaderBrushFtSt = null;
        private static Brush _HeaderBrushFfSt = null;

        ///
        static PercussionListBackgroundDrawer()
        {
            Color baseColor1;
            Color baseColor2;

            //
            baseColor1 = SystemColors.Control;
            baseColor2 = SystemColors.ControlDark;

            Color headerColor = Color.FromArgb
                ((baseColor1.R * 192 / 255) + (baseColor2.R * (255 - 192) / 255),
                 (baseColor1.G * 192 / 255) + (baseColor2.G * (255 - 192) / 255),
                 (baseColor1.B * 192 / 255) + (baseColor2.B * (255 - 192) / 255));

            //
            baseColor1 = Color.FromArgb(SystemColors.Highlight.R * 7 / 8,
                                         SystemColors.Highlight.G * 7 / 8,
                                         SystemColors.Highlight.B * 7 / 8);
            baseColor2 = headerColor;

            Color headerFtSt = Color.FromArgb
                ((baseColor1.R * 224 / 255) + (baseColor2.R * (255 - 224) / 255),
                 (baseColor1.G * 224 / 255) + (baseColor2.G * (255 - 224) / 255),
                 (baseColor1.B * 224 / 255) + (baseColor2.B * (255 - 224) / 255));

            //
            baseColor1 = SystemColors.ControlDark;
            baseColor2 = headerColor;

            Color headerFfSt = Color.FromArgb
                ((baseColor1.R * 224 / 255) + (baseColor2.R * (255 - 224) / 255),
                 (baseColor1.G * 224 / 255) + (baseColor2.G * (255 - 224) / 255),
                 (baseColor1.B * 224 / 255) + (baseColor2.B * (255 - 224) / 255));

            //
            _HeaderBrush = new SolidBrush(headerColor);
            _HeaderBrushFtSt = new SolidBrush(headerFtSt);
            _HeaderBrushFfSt = new SolidBrush(headerFfSt);

        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void DrawInternal(ListDrawDescriptor desc)
        {
            if (desc.Name != "Key")
            {
                base.DrawInternal(desc);
                return;
            }

            //
            Brush brush = null;

            if (desc.Selected != false)
            {
                brush = desc.Focused != false ? _HeaderBrushFtSt : _HeaderBrushFfSt;
            }
            else
            {
                brush = _HeaderBrush;
            }

            desc.Gc.FillRectangle(brush, desc.Bounds);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// パーカッションリスト用のアイテム
    /// </summary>
    public class PercussionListItem : CommonListItem
    {
        private bool _EnabledShadow = false;
        private int _Index = -1;
        private Instrument _Instrument = null;
        private KeyRegion _KeyRegion = null;
        private VelocityRegion _VelRegion = null;
        private string _Label = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public PercussionListItem(int index, Instrument instrument, KeyRegion keyRegion, bool shadow) : base(null)
        {
            Length = 20;

            _Index = index;
            _EnabledShadow = shadow;
            _Instrument = instrument;
            _KeyRegion = keyRegion;

            if (keyRegion != null)
            {
                _VelRegion = keyRegion.Children[0] as VelocityRegion;
                Debug.Assert(_VelRegion != null, "KeyRegion does not have VelocityRegion");
            }

            _Label = String.Format("{0} ({1,4})", KeyNoteConverter.ToNote(_Index),
                                    _Index.ToString());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public PercussionListItem(int index, Instrument instrument) : this(index, instrument, null, false)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int OriginalKey
        {
            get { return _VelRegion != null ? _VelRegion.OriginalKey : 0; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Replace(KeyRegion keyRegion, bool shadow)
        {
            _EnabledShadow = shadow;

            _KeyRegion = keyRegion;
            _VelRegion = null;

            if (keyRegion != null)
            {
                _VelRegion = keyRegion.Children[0] as VelocityRegion;
                Debug.Assert(_VelRegion != null, "KeyRegion does not have VelocityRegion");
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component Target
        {
            get { return _VelRegion; }
            set { }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Shadow
        {
            get { return _EnabledShadow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Index
        {
            get { return _Index; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Instrument Instrument
        {
            get { return _Instrument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public KeyRegion KeyRegion
        {
            get { return _KeyRegion; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override bool CanEdit
        {
            get
            {
                if (_VelRegion == null ||
                    _EnabledShadow != false)
                {
                    return false;
                }
                return true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override IParameterValue GetValue(string name)
        {
            if (_VelRegion == null ||
                _VelRegion.Parameters.ContainsKey(name) == false)
            {
                return null;
            }

            //
            if (_Instrument.IsInstListEnvelope != false)
            {
                switch (name)
                {
                    case ProjectParameterNames.Envelope.Attack:
                    case ProjectParameterNames.Envelope.Decay:
                    case ProjectParameterNames.Envelope.Hold:
                    case ProjectParameterNames.Envelope.Sustain:
                    case ProjectParameterNames.Envelope.Release:
                        return null;
                }
            }

            //
            switch (name)
            {
                case "OriginalKey":
                    ParameterEditor parameterEditor = null;

                    parameterEditor = new OriginalKeyParameterEditor
                        (name, _VelRegion.Parameters[name]);
                    parameterEditor.ValueChanged += OnValueChanged;
                    return parameterEditor;

                case "DataSize":
                    return null;
            }

            return CreateParameterEditor(name, _VelRegion.Parameters[name], OnValueChanged);
        }

        /// <summary>
        ///
        /// </summary>
        public override IConstParameterValue GetConstValue(string name)
        {
            //
            if (name == "Key")
            {
                return new TextParameterValue(_Label);
            }

            //
            if (_VelRegion == null)
            {
                return new TextParameterValue("*");
            }

            //
            if (_Instrument.IsInstListEnvelope != false)
            {
                switch (name)
                {
                    case ProjectParameterNames.Envelope.Attack:
                    case ProjectParameterNames.Envelope.Decay:
                    case ProjectParameterNames.Envelope.Hold:
                    case ProjectParameterNames.Envelope.Sustain:
                    case ProjectParameterNames.Envelope.Release:
                        return new TextParameterValue("*");
                }
            }

            //
            switch (name)
            {
                case ProjectParameterNames.FilePath:
                    if (Shadow != false)
                    {
                        //TextParameterValueを返すことにより "..."ボタンの表示を消している
                        return new TextParameterValue((string)_VelRegion.Parameters[name].Value);
                    }
                    break;

                case ProjectParameterNames.WaveTime:
                    _VelRegion.UpdateWaveFile();
                    string times = WaveFileUtility.GetWaveTimeStringWithLoop(_VelRegion.WaveFile);
                    return new TextParameterValue(times);

                case ProjectParameterNames.WaveTick:
                    _VelRegion.UpdateWaveFile();
                    string tick = WaveFileUtility.GetWaveTickString(_VelRegion.WaveFile);
                    return new TextParameterValue(tick);

                case ProjectParameterNames.SampleRate:
                    WaveFile waveFile = GetComponentReferenceWaveFile(_VelRegion);
                    string text = WaveFileUtility.GetSampleRateString(waveFile);
                    return new TextParameterValue(text);

                case ProjectParameterNames.WaveBitRate:
                    waveFile = GetComponentReferenceWaveFile(_VelRegion);
                    text = WaveFileUtility.GetWaveBitRateString(waveFile);
                    return new TextParameterValue(text);

                case ProjectParameterNames.WaveSampleBit:
                    waveFile = GetComponentReferenceWaveFile(_VelRegion);
                    text = WaveFileUtility.GetWaveSampleBitString(waveFile);
                    return new TextParameterValue(text);

                case ProjectParameterNames.WaveChannel:
                    waveFile = GetComponentReferenceWaveFile(_VelRegion);
                    text = WaveFileUtility.GetChannelString(waveFile);
                    return new TextParameterValue(text);

                case ProjectParameterNames.VelocityRegion.OriginalKey:
                    int key = (int)_VelRegion.Parameters[name].Value;
                    return new TextParameterValue(KeyNoteConverter.ToNote(key));

                case ProjectParameterNames.VelocityRegion.InterpolationType:
                    return new InterpolationTypeParameterValue(_VelRegion.InterpolationType);
            }

            return _VelRegion.Parameters[name];
        }

        /// <summary>
        ///
        /// </summary>
        public bool ContainsKeyRange(int key)
        {
            if (Shadow != false || _KeyRegion == null)
            {
                return false;
            }

            return key >= _KeyRegion.KeyMin && key <= _KeyRegion.KeyMax;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnValueChanged(object sender, EventArgs e)
        {
            ParameterEditor parameterEditor = sender as ParameterEditor;

            if (parameterEditor.Name == "FilePath" && _VelRegion != null)
            {
                _VelRegion.WaveFile = null;
            }

            Adapter.SetValue(_VelRegion, parameterEditor.Name, parameterEditor.Value);
        }

        private WaveFile GetComponentReferenceWaveFile(VelocityRegion velRegion)
        {
            if (velRegion != null)
            {
                velRegion.UpdateWaveFile();

                return velRegion.WaveFile;
            }

            return null;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// オリジナルキー編集用の ParameterEditor
    /// </summary>
    public class OriginalKeyParameterEditor : ParameterEditor
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OriginalKeyParameterEditor(string name, IParameterValue targetParameter) : base(name, targetParameter)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object ParseValue(string text)
        {
            return KeyNoteConverter.ToKey(text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override ValidationResult ValidateValue(object value)
        {
            int key = KeyNoteConverter.ToKey((string)value);
            return base.ValidateValue(key);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override object Value
        {
            set { base.Value = KeyNoteConverter.ToKey((string)value); }
        }

    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class KeyNoteConverter
    {
        private static string[] _NoteLabels = new string[] {
            "cnm1", "csm1", "dnm1", "dsm1", "enm1", "fnm1", "fsm1",
            "gnm1", "gsm1", "anm1", "asm1", "bnm1",

            "cn0", "cs0", "dn0", "ds0", "en0", "fn0", "fs0",
            "gn0", "gs0", "an0", "as0", "bn0",

            "cn1", "cs1", "dn1", "ds1", "en1", "fn1", "fs1",
            "gn1", "gs1", "an1", "as1", "bn1",

            "cn2", "cs2", "dn2", "ds2", "en2", "fn2", "fs2",
            "gn2", "gs2", "an2", "as2", "bn2",

            "cn3", "cs3", "dn3", "ds3", "en3", "fn3", "fs3",
            "gn3", "gs3", "an3", "as3", "bn3",

            "cn4", "cs4", "dn4", "ds4", "en4", "fn4", "fs4",
            "gn4", "gs4", "an4", "as4", "bn4",

            "cn5", "cs5", "dn5", "ds5", "en5", "fn5", "fs5",
            "gn5", "gs5", "an5", "as5", "bn5",

            "cn6", "cs6", "dn6", "ds6", "en6", "fn6", "fs6",
            "gn6", "gs6", "an6", "as6", "bn6",

            "cn7", "cs7", "dn7", "ds7", "en7", "fn7", "fs7",
            "gn7", "gs7", "an7", "as7", "bn7",

            "cn8", "cs8", "dn8", "ds8", "en8", "fn8", "fs8",
            "gn8", "gs8", "an8", "as8", "bn8",

            "cn9", "cs9", "dn9", "ds9", "en9", "fn9", "fs9",
            "gn9",
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static string[] Notes
        {
            get { return _NoteLabels; }
        }

        ///--------------------------------
        /// <summary>
        /// Keyから Noteに変換する
        /// </summary>
        public static string ToNote(int key)
        {
            Debug.Assert(key >= 0 && key <= 127, "Index is out of a range");
            return _NoteLabels[key];
        }

        ///--------------------------------
        /// <summary>
        /// Noteから Keyに変換する
        /// </summary>
        public static int ToKey(String note)
        {
            try { return ParseScale(note.ToUpper()); }
            catch (IndexOutOfRangeException) { return -1; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static int ParseScale(string value)
        {
            int index = 0;
            int key = 0;
            int resultKey = 0;

            switch (value[index])
            {
                case 'C': key = 0; break;
                case 'D': key = 2; break;
                case 'E': key = 4; break;
                case 'F': key = 5; break;
                case 'G': key = 7; break;
                case 'A': key = 9; break;
                case 'B': key = 11; break;
                default: return -1;
            }

            index++;

            //
            switch (value[index])
            {
                case 'S':
                    key++;
                    index++;
                    break;

                case 'N':
                    index++;
                    break;

                default:
                    return -1;
            }

            //
            if ((resultKey = ParseOctave(value.Substring(index))) < 0)
            {
                return -1;
            }

            key += resultKey;
            if (key < 0 || key > 127)
            {
                return -1;
            }

            return key;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static int ParseOctave(string value)
        {
            int index = 0;
            int key = 0;
            bool minus = false;

            if (value[index] == 'M')
            {
                minus = true;
                index++;
            }

            while (index < value.Length)
            {
                if (value[index] < '0' || value[index] > '9')
                {
                    return -1;
                }

                key = key * 10 + (int)(value[index] - '0');
                index++;
            }

            if (minus != false)
            {
                key = -key;
            }

            if ((key = (key + 1) * 12) > 127)
            {
                key = 127;
            }
            return key;
        }
    }
}
