﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Reflection;
    using System.Resources;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.FileFormats.Audio;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    ///--------------------------------------------------------------------------
    /// <summary>
    /// CreateStreamSoundDialog の概要の説明です。
    /// </summary>
    public partial class CreateStreamSoundDialog : CreateSoundDialog
    {
        private object[] baseItems;

        private readonly StreamSoundBase _sound = ApplicationBase.Instance.CreateComponentService.Create<StreamSoundBase>();

        public static bool VisibleWaveEncodingPCM8
        {
            get; set;
        }

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateStreamSoundDialog(SoundProjectService projectService, SoundSetDocument soundSetDocument, string defaultPlayerName)
            : base(projectService, soundSetDocument, defaultPlayerName)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == defaultPlayerName) { throw new ArgumentNullException("defaultPlayerName"); }

            InitializeComponent();

            //
            SetUniqueNameToTextBox(textBox_Label, projectService.Project,
                                    ProjectService.ComponentDictionary,
                                    ProjectService.Project.StreamSoundNamePrefix);

            //
            if (VisibleWaveEncodingPCM8 == true)
            {
                comboBox_Type.Items.Add("PCM8");
            }

            this.FileType = _sound.Encoding;
            this.Volume = _sound.Volume;
            this.PlayerPriority = _sound.PlayerPriority;

            //プレイヤーの設定
            comboBox_Player.Items.AddRange(PlayerControlItems);
            comboBox_Player.SelectedIndex = 0;
            this.PlayerReference = _sound.PlayerReference;

            //
            InspectFilePath();
            UpdateControls();

            //
            //ResumeInitialParameter();

            List<object> items = new List<object>();
            foreach (object item in this.WaveEncodingComboBox.Items)
            {
                items.Add(item);
            }
            this.baseItems = items.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの取得
        /// </summary>
        private string Label
        {
            get { return textBox_Label.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルタイプの取得
        /// </summary>
        private WaveEncoding FileType
        {
            get
            {
                if (AACUtil.IsAACFile(this.FilePath) == true ||
                    FileUtil.IsOpusFile(this.FilePath) == true)
                {
                    return WaveEncoding.NoConvert;
                }
                else
                {
                    switch (comboBox_Type.SelectedIndex)
                    {
                        case 0:
                            return WaveEncoding.Adpcm;
                        case 1:
                            return WaveEncoding.Pcm16;
                        case 2:
                            return WaveEncoding.Pcm8;
                    }
                }

                throw new Exception("unexpected error");
            }
            set
            {
                foreach (var item in comboBox_Type.Items)
                {
                    if (item.ToString().ToLower() == value.ToString().ToLower())
                    {
                        comboBox_Type.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get { return textBox_FilePath.Text; }
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの取得/設定
        /// </summary>
        private int Volume
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_Volume.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_Volume.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの取得/設定
        /// </summary>
        private int PlayerPriority
        {
            get
            {
                int value = 0;
                try { value = int.Parse(textBox_PlayerPriority.Text); }
                catch { Debug.Assert(false); }
                return value;
            }
            set
            {
                textBox_PlayerPriority.Text = _sound.PlayerPriority.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤーを取得します。
        /// </summary>
        private string PlayerReference
        {
            get
            {
                return comboBox_Player.SelectedItem.ToString();
            }
            set
            {
                foreach (var item in comboBox_Player.Items)
                {
                    if (item.ToString() == value)
                    {
                        comboBox_Player.SelectedItem = item;
                        break;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// １トラックのチャンネル数を取得します。
        /// </summary>
        private int ChannelCount
        {
            get
            {
                if (AppConfiguration.EnabledMultiChannelAAC == true &&
                    AACUtil.IsAACFile(FilePath) == true)
                {
                    if (this.radioButton_Channel1.Checked == true)
                    {
                        return 1;
                    }
                    else if (this.radioButton_Channel2.Checked == true)
                    {
                        return 2;
                    }
                }

                return 0;
            }
        }


        /// <summary>
        /// 作成したサウンドを取得します。
        /// </summary>
        public StreamSoundBase NewSound
        {
            get
            {
                if (0 < _sound.Children.Count)
                {
                    return _sound;
                }

                _sound.Name = Label;
                _sound.Encoding = FileType;
                _sound.Volume = Volume;
                _sound.PlayerReference = PlayerReference;
                _sound.PlayerPriority = PlayerPriority;
                _sound.FilePath = FilePath;

                if (AppConfiguration.EnabledMultiChannelAAC == true &&
                    AACUtil.IsAACFile(FilePath) == true)
                {
                    using (Stream stream = File.OpenRead(FilePath))
                    {
                        AdtsHeader adtsHeader = AdtsHeader.Parse(stream);
                        if (adtsHeader != null && adtsHeader.IsValid() == true)
                        {
                            int trackCount = (adtsHeader.Channel + (adtsHeader.Channel % ChannelCount)) / ChannelCount;
                            for (int i = 0; i < trackCount; i++)
                            {
                                StreamSoundTrackBase track = ApplicationBase.Instance.CreateComponentService.Create<StreamSoundTrackBase>();
                                track.FilePath = FilePath;
                                if (i * ChannelCount <= adtsHeader.Channel)
                                {
                                    track.ChannelCount = ChannelCount;
                                }
                                else
                                {
                                    track.ChannelCount = 1;
                                }
                                _sound.Children.Add(track);
                            }
                        }
                    }
                }

                if (_sound.Children.Count <= 0)
                {
                    StreamSoundTrackBase track = ApplicationBase.Instance.CreateComponentService.Create<StreamSoundTrackBase>();
                    track.FilePath = FilePath;
                    _sound.Children.Add(track);
                }
                else
                {
                    ((StreamSoundTrackBase)_sound.Children[0]).FilePath = FilePath;
                }

                return _sound;
            }
        }

        protected ComboBox WaveEncodingComboBox
        {
            get
            {
                return this.comboBox_Type;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReference(object sender, System.EventArgs e)
        {
            string filePath = null;
            string filter = FormsApplication.Instance.AppConfiguration.StreamWaveFileFilter;

            if ((filePath = WaveFileQuester.QueryLoad(this, filter)) != null)
            {
                textBox_FilePath.Text = filePath;
            }

            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                try
                {
                    bool label = InspectLabel();
                    bool filePath = InspectFilePath();
                    bool volume = InspectVolume();
                    bool playerPriority = InspectPlayerPriority();

                    if (label != false &&
                         filePath != false &&
                         volume != false &&
                         playerPriority != false)
                    {
                        return true;
                    }
                }
                catch { }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ラベルの検査
        /// </summary>
        private bool InspectLabel()
        {
            return ValidateName(textBox_Label.Text, textBox_Label);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの検査
        /// </summary>
        private bool InspectFilePath()
        {
            if (File.Exists(textBox_FilePath.Text) == false)
            {
                button_CreateLabel.Enabled = false;
                ShowError(textBox_FilePath, MessageResource.Message_FileNotFound);
                return false;
            }

            button_CreateLabel.Enabled = true;
            HideError(textBox_FilePath);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの検査
        /// </summary>
        private bool InspectVolume()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Volume,
                                         textBox_Volume.Text, textBox_Volume);
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤープライオリティの検査
        /// </summary>
        private bool InspectPlayerPriority()
        {
            return ValidateIntParameter(_sound, ProjectParameterNames.Sound.PlayerPriority,
                                         textBox_PlayerPriority.Text, textBox_PlayerPriority);
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnTextChangedFilePath(object sender, EventArgs e)
        {
            TextBox textBox = sender as TextBox;
            string filePath = textBox.Text;
            int index = this.WaveEncodingComboBox.SelectedIndex;

            this.label_ChannelCount.Visible = false;
            this.label_ChannelCount.Enabled = false;
            this.radioButton_Channel1.Visible = false;
            this.radioButton_Channel1.Enabled = false;
            this.radioButton_Channel2.Visible = false;
            this.radioButton_Channel2.Enabled = false;

            if (AACUtil.IsAACFile(filePath) == true || FileUtil.IsOpusFile(filePath) == true)
            {
                this.WaveEncodingComboBox.Items.Clear();
                this.WaveEncodingComboBox.Items.Add(MessageResource.Label_WaveEncoding_NoConvert);
                this.WaveEncodingComboBox.SelectedIndex = 0;

                if (FileUtil.IsOpusFile(filePath) == true)
                {
                    return;
                }

                if (AppConfiguration.EnabledMultiChannelAAC == true)
                {
                    this.label_ChannelCount.Visible = true;
                    this.label_ChannelCount.Enabled = true;
                    this.radioButton_Channel1.Visible = true;
                    this.radioButton_Channel1.Enabled = true;
                    this.radioButton_Channel2.Visible = true;
                    this.radioButton_Channel2.Enabled = true;
                }
            }
            else
            {
                this.WaveEncodingComboBox.Items.Clear();
                this.WaveEncodingComboBox.Items.AddRange(this.baseItems);
                this.WaveEncodingComboBox.SelectedIndex = index;
            }

            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedLabel(object sender, System.EventArgs e)
        {
            InspectLabel();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFilePath(object sender, System.EventArgs e)
        {
            InspectFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedVolume(object sender, System.EventArgs e)
        {
            InspectVolume();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedPlayerPriority(object sender, System.EventArgs e)
        {
            InspectPlayerPriority();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCreateLabel(object sender, EventArgs e)
        {
            textBox_Label.Text = CreateSoundSetItemNameFromFilePath(
                                    ProjectService.Project.StreamSoundNamePrefix, textBox_FilePath.Text);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }
    }
}
