﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Collections;
using System.Linq;
using System.Windows.Forms;
using System.Resources;
using System.Reflection;
using System.Diagnostics;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Resources;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    /// アイテム作成ダイアログの基本クラス
    /// </summary>
    public partial class CreateItemDialog : System.Windows.Forms.Form
    {
        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateItemDialog()
        {
            InitializeComponent();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public AppMessageBoxResult InquireYesNo(string text)
        {
            if (null == text) { throw new ArgumentNullException("text"); }
            return WarningMessageBox(text, AppMessageBoxButton.OKCancel);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public AppMessageBoxResult WarningMessageBox(string text, AppMessageBoxButton button)
        {
            if (null == text) { throw new ArgumentNullException("text"); }
            return FormsApplication.Instance.UIService.ShowMessageBox(text, button, AppMessageBoxImage.Warning);
        }


        ///--------------------------------
        /// <summary>
        /// エラーの表示
        /// </summary>
        protected void ShowErrorText(Control control, string message)
        {
            Debug.Assert(message != null);
            _ErrorProvider.SetError(control, message);
        }

        ///--------------------------------
        /// <summary>
        /// エラーの表示
        /// </summary>
        protected void ShowError(Control control, string text)
        {
            if (null == text) { throw new ArgumentNullException("text"); }
            _ErrorProvider.SetError(control, text);
        }

        ///--------------------------------
        /// <summary>
        /// エラーの非表示
        /// </summary>
        protected void HideError(Control control)
        {
            _ErrorProvider.SetError(control, null);
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected virtual Control GetAcceptButton()
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// コントロールの更新
        /// </summary>
        protected void UpdateControls()
        {
            Control acceptButton = GetAcceptButton();
            Debug.Assert(acceptButton != null);
            acceptButton.Enabled = CanCreate;
        }

        ///--------------------------------
        /// <summary>
        /// アイテムを作成することが可能な状態なのか調べる
        /// ダイアログが閉じられる時に呼ばれる、falseを返すと、ダイアログが閉じなく
        /// なります。
        /// </summary>
        protected virtual bool CanCreate
        {
            get { Debug.Assert(false); return false; }
        }

        /// <summary>
        /// 名前の検証
        /// </summary>
        protected bool ValidateName(ComponentService componentService, string name, Control control)
        {
            if (null == componentService) { throw new ArgumentNullException("projectService"); }
            if (null == name) { throw new ArgumentNullException("name"); }
            if (null == control) { throw new ArgumentNullException("control"); }

            ValidationResult result = ItemNameValidator.ValidateSoundProjectItemName(name);
            if (!result.IsValid)
            {
                ShowError(control, result.ToString());
                return false;
            }

            if (componentService.ComponentDictionary.Contains(name) != false)
            {
                Component[] components = componentService.ComponentDictionary[name]
                    .Where(c => !(c is SoundSet))
                    .Where(c => !(c is FolderComponent))
                    .Where(c => !(c is SoundProject))
                    .ToArray();

                if (components.Length > 0)
                {
                    ShowError(control, MessageResource.Message_ItemNameAlreadyExisting);
                    return false;
                }
            }

            HideError(control);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ダイアログが閉じられる直前に呼ばれる
        /// </summary>
        private void OnClosing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (CanCreate == false)
                {
                    e.Cancel = true;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected void SetUniqueNameToTextBox(TextBox textBox, SoundProject project, ComponentDictionary componentDictionary, string prefix)
        {
            ItemNamingSettings setting = new ItemNamingSettings(project);

            if (setting.HasPrefix != false)
            {
                textBox.Text = ItemNamer.CreateUniqueName(componentDictionary, prefix);
                textBox.SelectionStart = prefix.Length;
                textBox.SelectionLength = textBox.Text.Length - prefix.Length;
            }
            else
            {
                textBox.Text = ItemNamer.CreateUniqueName(componentDictionary, String.Empty);
                textBox.SelectAll();
            }
        }
    }
}
