﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Windows.Forms;
using NintendoWare.SoundMaker.Framework.Configurations.Schemas;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    /// <summary>
    /// ウィンドウ設定を適用または抽出します。
    /// </summary>
    public class WindowConfigurationApplier
    {
        private Control _control;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="control">対象コントロール。</param>
        public WindowConfigurationApplier(Control control)
        {
            if (null == control) { throw new ArgumentNullException("control"); }

            _control = control;
        }

        /// <summary>
        /// ウィンドウ設定を適用します。
        /// </summary>
        /// <param name="control">適用先のコントロール。</param>
        /// <param name="xmlWindow">ウィンドウ設定。</param>
        public static void Apply(Control control, XmlWindow xmlWindow)
        {
            new WindowConfigurationApplier(control).Apply(xmlWindow, false);
        }

        /// <summary>
        /// ウィンドウ設定を適用します。
        /// </summary>
        /// <param name="control">適用先のコントロール。</param>
        /// <param name="xmlWindow">ウィンドウ設定。</param>
        /// <param name="applyVisible">Visible プロパティの有効無効設定。</param>
        public static void Apply(Control control, XmlWindow xmlWindow, bool applyVisible)
        {
            new WindowConfigurationApplier(control).Apply(xmlWindow, applyVisible);
        }

        /// <summary>
        /// ウィンドウ設定を抽出します。
        /// </summary>
        /// <param name="control">対象コントロール。</param>
        /// <param name="xmlWindow">抽出したデータの格納先。</param>
        public static void Extract(Control control, XmlWindow xmlWindow)
        {
            new WindowConfigurationApplier(control).Extract(xmlWindow, false);
        }

        /// <summary>
        /// ウィンドウ設定を抽出します。
        /// </summary>
        /// <param name="control">対象コントロール。</param>
        /// <param name="xmlWindow">抽出したデータの格納先。</param>
        /// <param name="extractVisible">Visible プロパティの有効無効設定。</param>
        public static void Extract(Control control, XmlWindow xmlWindow, bool extractVisible)
        {
            new WindowConfigurationApplier(control).Extract(xmlWindow, extractVisible);
        }

        /// <summary>
        /// ウィンドウ設定を適用します。
        /// </summary>
        /// <param name="xmlWindow">ウィンドウ設定。</param>
        /// <param name="applyVisible">Visible プロパティの有効無効設定。</param>
        public void Apply(XmlWindow xmlWindow, bool applyVisible)
        {
            if (null == xmlWindow) { throw new ArgumentNullException("xmlWindow"); }

            _control.SuspendLayout();

            //
            Rectangle windowBounds = new Rectangle(xmlWindow.X, xmlWindow.Y,
                                                    xmlWindow.Width, xmlWindow.Height);
            Screen cardinalScreen = null;

            foreach (Screen screen in Screen.AllScreens)
            {
                if (Rectangle.Intersect(screen.Bounds, windowBounds) != Rectangle.Empty)
                {
                    cardinalScreen = screen;
                    break;
                }
            }

            bool modifiedLocation = false;
            if (cardinalScreen == null)
            {
                Rectangle bounds = Screen.PrimaryScreen.Bounds;
                int limitX = bounds.X + bounds.Width;
                int limitY = bounds.Y + bounds.Height;
                int x = xmlWindow.X;
                int y = xmlWindow.Y;

                if (x > limitX)
                {
                    x = limitX - xmlWindow.Width;
                }

                if (y > limitY)
                {
                    y = limitY - xmlWindow.Height;
                }

                _control.Location = new Point(x, y);
                modifiedLocation = true;
            }
            else
            {
                if (-1 < xmlWindow.X && -1 < xmlWindow.Y)
                {
                    _control.Location = new Point(xmlWindow.X, xmlWindow.Y);
                    modifiedLocation = true;
                }
            }

            if (_control is Form && modifiedLocation != false)
            {
                (_control as Form).StartPosition = FormStartPosition.Manual;
            }

            //
            if (-1 < xmlWindow.Width && -1 < xmlWindow.Height)
            {
                _control.Size = new Size(xmlWindow.Width, xmlWindow.Height);
            }

            if (_control is Form)
            {

                Form form = _control as Form;

                switch (xmlWindow.State)
                {
                    case XmlWindowState.Normal:
                        form.WindowState = FormWindowState.Normal;
                        break;

                    case XmlWindowState.Maximized:
                        form.WindowState = FormWindowState.Maximized;
                        break;

                    case XmlWindowState.Minimized:
                        form.WindowState = FormWindowState.Minimized;
                        break;
                }

            }

            if (applyVisible)
            {
                _control.Visible = xmlWindow.Visible;
            }

            _control.ResumeLayout();
        }


        /// <summary>
        /// ウィンドウ設定を抽出します。
        /// </summary>
        /// <param name="xmlWindow">抽出したデータの格納先。</param>
        /// <param name="extractVisible">Visible プロパティの有効無効設定。</param>
        public void Extract(XmlWindow xmlWindow, bool extractVisible)
        {
            if (null == xmlWindow) { throw new ArgumentNullException("xmlWindow"); }

            bool extractBounds = true;

            if (_control is Form)
            {

                Form form = _control as Form;

                switch (form.WindowState)
                {
                    case FormWindowState.Normal:
                        xmlWindow.State = XmlWindowState.Normal;
                        break;

                    case FormWindowState.Maximized:
                        xmlWindow.State = XmlWindowState.Maximized;
                        extractBounds = false;
                        break;

                    case FormWindowState.Minimized:
                        xmlWindow.State = XmlWindowState.Minimized;
                        extractBounds = false;
                        break;
                }

            }
            else
            {
                xmlWindow.State = XmlWindowState.Normal;
            }

            if (extractVisible)
            {
                xmlWindow.Visible = _control.Visible;
            }

            if (!extractBounds) { return; }

            xmlWindow.X = _control.Left;
            xmlWindow.Y = _control.Top;
            xmlWindow.Width = _control.Width;
            xmlWindow.Height = _control.Height;
        }
    }
}
