﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// ファイルパスを問い合わせてプロジェクトを開きます。
    /// </summary>
    public class QueryAndOpenProjectHandler : OpenProjectHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public QueryAndOpenProjectHandler(IQueryCommandParameter queryParameter) : base(queryParameter) { }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            SoundProjectService projectService = GetTargetProjectService(parameters);
            if (null == projectService) { return false; }

            string filePath = null;
            bool isMruProject = false;

            if (null != parameters &&
                 parameters.ContainsParameter(CommandParameterNames.TargetFilePath))
            {
                filePath = parameters.GetParameter(CommandParameterNames.TargetFilePath) as string;
                isMruProject = true;
            }

            if (null == filePath)
            {
                filePath = ProjectFileQuester.QueryLoad();
            }
            if (null == filePath) { return false; }


            if (null != projectService.ProjectDocument)
            {
                if (!CloseProjectHandler.Execute()) { return false; }
            }

            try
            {
                return base.ExecuteInternal(projectService, filePath);
            }
            catch (DirectoryNotFoundException)
            {

                // MruProject の場合は処理を続ける。
                if (!isMruProject)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                        string.Format(Resources.MessageResource.Message_TargetFileNotFound,
                                       filePath));
                    return false;
                }

            }
            catch (FileNotFoundException exception)
            {

                // MruProject の場合は処理を続ける。
                if (!isMruProject || exception.FileName != filePath)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox(
                        string.Format(Resources.MessageResource.Message_TargetFileNotFound,
                                       exception.FileName));
                    return false;
                }

            }
            catch (FileNotSupportedException exception)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                                exception.Message,
                                                                                exception.FilePath));
                return false;
            }
            catch (FileNotSupportedVersionException exception)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                                exception.Message,
                                                                                exception.FilePath));
                return false;
            }
            catch (Exception e)
            {
                string message = e.Message;
                if (string.IsNullOrEmpty(message) == true)
                {
                    message = Resources.MessageResource.Message_UnknownError;
                }
                ApplicationBase.Instance.UIService.ShowMessageBox(message);
                return false;
            }

            DeleteMruProject(filePath);

            return true;
        }

        private void DeleteMruProject(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            if (!ApplicationBase.Instance.AppConfiguration.MruProjects.Contains(filePath)) { return; }

            if (File.Exists(filePath) == true) { return; }

            if (ApplicationBase.Instance.UIService.ShowMessageBox(
                    string.Format(Resources.MessageResource.Message_ConfirmDeleteMruProject,
                                   filePath),
                    null,
                    AppMessageBoxButton.YesNo,
                    AppMessageBoxImage.Question,
                    AppMessageBoxResult.Yes) != AppMessageBoxResult.Yes)
            {
                return;
            }

            ApplicationBase.Instance.AppConfiguration.MruProjects.Delete(filePath);
        }
    }
}
