﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Utilities
{
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Projects;
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Xml;

    public static class BankUtility
    {
        public static string[] GetAllWaveFilePath(string bankFilePath)
        {
            return new FilePathExtractor().GetAllFilePaths(bankFilePath).ToArray();
        }

        public static string[] GetAllWaveFilePath(Bank bank)
        {
            List<string> waveFilePaths = new List<string>();

            GetAllVelocityFilePath(bank, ref waveFilePaths);

            return waveFilePaths.ToArray();
        }

        private static void GetAllVelocityFilePath(Component component, ref List<string> waveFilePaths)
        {
            if (component != null)
            {
                if (component is VelocityRegion == true)
                {
                    VelocityRegion velocityRegion = component as VelocityRegion;
                    waveFilePaths.Add(velocityRegion.FilePath);
                }
                else
                {
                    foreach (Component child in component.Children)
                    {
                        GetAllVelocityFilePath(child, ref waveFilePaths);
                    }
                }
            }
        }

        /// <summary>
        /// バンクファイル(*.fbnk)からベロシティリージョンのファイルパスを取得するクラスです。
        /// </summary>
        private class FilePathExtractor
        {
            private class ActionDictionary : Dictionary<string, Action<FilePathExtractor, XmlReader, string>> { }

            private static readonly Dictionary<string, ActionDictionary> NodeActionDictionary = new Dictionary<string, ActionDictionary>();
            private readonly List<string> filePaths = new List<string>();

            static FilePathExtractor()
            {
                NodeActionDictionary.Add(string.Empty,
                    new ActionDictionary()
                        {
                            { "Bank", (self, reader, nodeName) => self.Read(reader, nodeName, "Body") },
                        });

                NodeActionDictionary.Add("Bank",
                    new ActionDictionary()
                        {
                            { "Body",  (self, reader, nodeName) => self.Read(reader, nodeName, "Bank") },
                            { "Items", (self, reader, nodeName) => self.Read(reader, nodeName, "Instrument") },
                        });

                NodeActionDictionary.Add("Body",
                    new ActionDictionary()
                        {
                            { "Bank", (self, reader, nodeName) => self.Read(reader, nodeName, "Items") },
                        });

                NodeActionDictionary.Add("Items",
                    new ActionDictionary()
                        {
                            { "Instrument",      (self, reader, nodeName) => self.Read(reader, nodeName, "Items") },
                            { "KeyRegion",       (self, reader, nodeName) => self.Read(reader, nodeName, "Items") },
                            { "VelocityRegion",  (self, reader, nodeName) => self.Read(reader, nodeName, "Parameters") },
                        });

                NodeActionDictionary.Add("Instrument",
                    new ActionDictionary()
                        {
                            { "Items", (self, reader, nodeName) => self.Read(reader, nodeName, "KeyRegion") },
                        });

                NodeActionDictionary.Add("KeyRegion",
                    new ActionDictionary()
                        {
                            { "Items", (self, reader, nodeName) => self.Read(reader, nodeName, "VelocityRegion") },
                        });

                NodeActionDictionary.Add("VelocityRegion",
                    new ActionDictionary()
                        {
                            { "Parameters", (self, reader, nodeName) => self.ReadParameters(reader) },
                        });
            }

            public IEnumerable<string> GetAllFilePaths(string bankFilePath)
            {
                try
                {
                    using (var stream = new FileStream(bankFilePath, FileMode.Open))
                    {
                        XmlReaderSettings settings = new XmlReaderSettings();
                        settings.IgnoreWhitespace = true;
                        settings.IgnoreComments = true;

                        using (XmlReader reader = XmlReader.Create(stream, settings))
                        {
                            this.Read(reader, string.Empty, "Bank");
                        }

                        var baseDirectory = Path.GetDirectoryName(bankFilePath);
                        return filePaths
                            .Select(f => Path.IsPathRooted(f) == true ? f : Path.Combine(baseDirectory, f).GetFullPath());
                    }
                }
                catch (Exception)
                {
                    return new string[0];
                }
            }

            private void Read(XmlReader reader, string currentNodeName, string targetNodeName)
            {
                while (reader.Read())
                {
                    if (reader.NodeType == XmlNodeType.EndElement &&
                        reader.Name == currentNodeName)
                    {
                        return;
                    }

                    if (reader.NodeType == XmlNodeType.Element &&
                        reader.Name == targetNodeName)
                    {
                        var actionDictionary = NodeActionDictionary[currentNodeName];
                        actionDictionary[targetNodeName](this, reader, targetNodeName);
                    }
                }
            }

            private void ReadParameters(XmlReader reader)
            {
                while (reader.Read())
                {
                    if (reader.NodeType == XmlNodeType.EndElement &&
                        reader.Name == "Parameters")
                    {
                        return;
                    }

                    if (reader.NodeType == XmlNodeType.Element)
                    {
                        switch (reader.Name)
                        {
                            case "FilePath":
                                var filePath = this.ReadValue(reader);
                                if (filePath != null)
                                {
                                    this.filePaths.Add(filePath);
                                }
                                break;
                        }
                    }
                }
            }

            private string ReadValue(XmlReader reader)
            {
                if (reader.IsEmptyElement == false)
                {
                    reader.Read();
                    if (reader.NodeType != XmlNodeType.Text)
                    {
                        throw new XmlException();
                    }

                    var value = reader.Value;

                    reader.Read();
                    if (reader.NodeType != XmlNodeType.EndElement)
                    {
                        // タグが閉じていない
                        throw new XmlException();
                    }

                    return value;
                }
                return null;
            }
        }
    }
}
