﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Profile
{
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileManagement;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NW4R.ProtocolSound;

    /// <summary>
    ///
    /// </summary>
    public class ProjectProfileService : IDisposable
    {
        private MeasureWaveReferenceCount waveReferenceCountMeasurer = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ProjectProfileService(DocumentService documentService, BankServiceManager bankServiceManager, ParameterValueChangedProxyService parameterValueChangedProxyService, FileWatcherService fileWatcherService, SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            this.waveReferenceCountMeasurer = new MeasureWaveReferenceCount
                (documentService, bankServiceManager,
                  parameterValueChangedProxyService, fileWatcherService,
                  intermediateOutputTraits);
        }

        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            this.waveReferenceCountMeasurer.Dispose();
        }

        /// <summary>
        ///
        /// </summary>
        public MeasureWaveReferenceCount WaveReferenceCountMeasurer
        {
            get
            {
                return this.waveReferenceCountMeasurer;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public class MeasureWaveReferenceCount : IDisposable
        {
            private const int MaxKeyCount = 128;
            private const int MaxVelocityCount = 128;

            private SoundIntermediateOutputTraits intermediateOutputTraits;

            private BankServiceManager bankServiceManager = null;
            private DocumentService documentService = null;

            private ParameterValueChangedProxyService parameterValueChangedProxyService = null;
            private FileWatcherService fileWatcherService = null;

            private Dictionary<SoundSetItem, Data> datas = null;
            private Dictionary<Tuple<string, int>, ReferenceCountData> referenceCountDatas = null;
            private Dictionary<SoundSetItem, string> monitoringFiles = null;

            public event EventHandler RequireRecalculated;
            public event EventHandler RequireGetReferenceFileData;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public MeasureWaveReferenceCount(DocumentService documentService, BankServiceManager bankServiceManager, ParameterValueChangedProxyService parameterValueChangedProxyService, FileWatcherService fileWatcherService, SoundIntermediateOutputTraits intermediateOutputTraits)
            {
                this.documentService = documentService;
                this.fileWatcherService = fileWatcherService;

                this.bankServiceManager = bankServiceManager;
                this.bankServiceManager.Added += OnBankServiceAdded;
                this.bankServiceManager.Removed += OnBankServiceRemoved;

                this.parameterValueChangedProxyService = parameterValueChangedProxyService;
                this.parameterValueChangedProxyService.Add
                    (this,
                      OnParameterValueChanged,
                      OnComponentAdded,
                      OnComponentRemoved,
                      ParameterValueChangedProxyService.Kinds.SequenceSound |
                      ParameterValueChangedProxyService.Kinds.SoundSetBank);

                this.datas = new Dictionary<SoundSetItem, Data>();
                this.referenceCountDatas = new Dictionary<Tuple<string, int>, ReferenceCountData>();
                this.monitoringFiles = new Dictionary<SoundSetItem, string>();

                this.intermediateOutputTraits = intermediateOutputTraits;
            }

            /// <summary>
            ///
            /// </summary>
            public void Dispose()
            {
                this.documentService = null;
                this.fileWatcherService = null;

                this.bankServiceManager.Added -= OnBankServiceAdded;
                this.bankServiceManager.Removed -= OnBankServiceRemoved;
                this.bankServiceManager = null;

                this.parameterValueChangedProxyService.Remove(this);
                this.parameterValueChangedProxyService = null;
            }

            /// <summary>
            /// 参照データの収集を開始します。
            /// </summary>
            public void Begin()
            {
                ClearAllData();
            }

            /// <summary>
            /// 参照データの収集を終了します。
            /// </summary>
            public void End()
            {
                DispatchWaveFilePath();

                if (RequireGetReferenceFileData != null)
                {
                    RequireGetReferenceFileData(this, EventArgs.Empty);
                }
            }

            /// <summary>
            /// 全ての参照データを破棄します。
            /// </summary>
            public void ClearAllData()
            {
                this.datas.Clear();
                this.referenceCountDatas.Clear();
            }

            /// <summary>
            /// 参照データのインスタンスを作成します。
            /// </summary>
            public Data CreateData(Sound sound)
            {
                return new Data(sound);
            }

            /// <summary>
            /// 参照データを追加します。
            /// </summary>
            public void Add(Sound sound, Data data)
            {
                if (this.datas.ContainsKey(sound) == false)
                {
                    this.datas.Add(sound, data);
                }
            }

            /// <summary>
            /// 全てのファイル監視を終了します。
            /// </summary>
            public void TerminateAllMonitoringFile()
            {
                foreach (string filePath in this.monitoringFiles.Values)
                {
                    this.fileWatcherService.Remove(filePath, OnFileUpdated);
                }
                this.monitoringFiles.Clear();
            }

            /// <summary>
            ///
            /// </summary>
            public void DispatchWaveFilePath()
            {
                using (ReferenceBankDocumentManager manager = CreateReferenceBankDocumentManager())
                {
                    string[] bankFilePaths = this.datas.Values
                        .SelectMany(d => d.GetBankFilePaths())
                        .Distinct()
                        .ToArray();
                    manager.Add(bankFilePaths);

                    this.datas.Values
                        .ToList()
                        .ForEach((d) => { d.DispatchWaveFilePath(manager); });
                }
            }

            /// <summary>
            ///
            /// </summary>
            public void DispatchWaveFilePath(string bankFilePath)
            {
                using (ReferenceBankDocumentManager manager = CreateReferenceBankDocumentManager())
                {
                    manager.Add(bankFilePath);

                    this.datas.Values
                        .ToList()
                        .ForEach((d) => { d.DispatchWaveFilePath(manager); });
                }
            }

            /// <summary>
            ///
            /// </summary>
            public ReferenceFileData GetReferenceFileData()
            {
                using (ReferenceBankDocumentManager manager = CreateReferenceBankDocumentManager())
                {
                    foreach (Data data in this.datas.Values)
                    {
                        string[] bankFilePaths = data.GetBankFilePaths();
                        manager.Add(bankFilePaths);
                    }

                    ///
                    ReferenceFileData referenceFileData = new ReferenceFileData();
                    foreach (Data data in this.datas.Values)
                    {
                        data.AddReferenceFileData(referenceFileData, manager);
                    }

                    return referenceFileData;
                }
            }

            /// <summary>
            /// 追加されている参照データから集計された参照データを取得します。
            /// </summary>
            public ReferenceCountData GetReferenceCountData(string filePath, int programNo)
            {
                if (this.Calculated == false)
                {
                    return null;
                }

                ReferenceCountData resultData = null;
                Tuple<string, int> key = Tuple.Create(filePath, programNo);
                if (this.referenceCountDatas.TryGetValue(key, out resultData) == true)
                {
                    return resultData;
                }

                try
                {
                    resultData = new ReferenceCountData();
                    foreach (Data data in this.datas.Values)
                    {
                        data.AddReferenceCountData(resultData, filePath, programNo);
                    }
                }
                catch
                {
                }

                this.referenceCountDatas.Add(key, resultData);
                return resultData;
            }

            /// <summary>
            ///
            /// </summary>
            private string BankFileExtension
            {
                get
                {
                    return FormsApplication.Instance.Traits.IntermediateOutputTraits.BankFileExtension;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private string SequenceFileExtension
            {
                get
                {
                    return FormsApplication.Instance.Traits.IntermediateOutputTraits.TextSequenceSoundFileExtension;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private bool Calculated
            {
                get
                {
                    return this.datas.Count > 0 ? true : false;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private ReferenceBankDocumentManager CreateReferenceBankDocumentManager()
            {
                return new ReferenceBankDocumentManager
                    (this.documentService, this.bankServiceManager, this.intermediateOutputTraits);
            }

            /// <summary>
            /// 参照データに無効フラグを設定します。
            /// </summary>
            private void SetRequireCalculateFlag(string filePath)
            {
                foreach (KeyValuePair<SoundSetItem, Data> pair in this.datas)
                {
                    if (pair.Key is SequenceSoundBase)
                    {
                        SequenceSoundBase sequenceSound = pair.Key as SequenceSoundBase;
                        if (String.Compare(filePath, sequenceSound.FilePath, true) == 0)
                        {
                            SetRequireCalculateFlag(pair.Value);
                        }
                    }
                }
            }

            /// <summary>
            /// 参照データに無効フラグを設定します。
            /// </summary>
            private void SetRequireCalculateFlag(SoundSetItem sound)
            {
                Data data = null;
                if (this.datas.TryGetValue(sound, out data) == true)
                {
                    SetRequireCalculateFlag(data);
                }
            }

            /// <summary>
            /// 参照データに無効フラグを設定します。
            /// </summary>
            private void SetRequireCalculateFlag(Data data)
            {
                data.RequireCalculateFlag = true;
                this.referenceCountDatas.Clear();

                if (RequireRecalculated != null)
                {
                    RequireRecalculated(this, EventArgs.Empty);
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void OnBankServiceAdded(object sender, BankServiceEventArgs e)
            {
                e.BankService.Reloaded += OnBankReloaded;

            }

            /// <summary>
            ///
            /// </summary>
            private void OnBankServiceRemoved(object sender, BankServiceEventArgs e)
            {
                e.BankService.Reloaded -= OnBankReloaded;
            }

            /// <summary>
            ///
            /// </summary>
            private void OnBankReloaded(object sender, EventArgs e)
            {
                BankService bankService = sender as BankService;
                string bankFilePath = bankService.BankDocument.Resource.Key;

                DispatchWaveFilePath(bankFilePath);

                if (RequireGetReferenceFileData != null)
                {
                    RequireGetReferenceFileData(this, EventArgs.Empty);
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void OnComponentAdded(object sender, ComponentEventArgs e)
            {
                if (sender is SequenceSoundBase)
                {
                    SequenceSoundBase sequenceSound = sender as SequenceSoundBase;
                    this.AddMonitoringFile(sequenceSound, sequenceSound.FilePath);
                }
                else if (sender is SoundSetBankBase)
                {
                    SoundSetBankBase soundSetBank = sender as SoundSetBankBase;
                    this.AddMonitoringFile(soundSetBank, soundSetBank.FilePath);
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void OnComponentRemoved(object sender, ComponentEventArgs e)
            {
                if (sender is SequenceSoundBase)
                {
                    SequenceSoundBase sequenceSound = sender as SequenceSoundBase;
                    this.RemoveMonitoringFile(sequenceSound, sequenceSound.FilePath);
                }
                else if (sender is SoundSetBankBase)
                {
                    SoundSetBankBase soundSetBank = sender as SoundSetBankBase;
                    this.RemoveMonitoringFile(soundSetBank, soundSetBank.FilePath);
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void OnParameterValueChanged(object sender, ParameterEventArgs e)
            {
                switch (e.Key)
                {
                    case ProjectParameterNames.FilePath:
                        if (sender is SequenceSoundBase)
                        {
                            SequenceSoundBase sequenceSound = sender as SequenceSoundBase;
                            this.ReplaceMonitoringFile(sequenceSound, sequenceSound.FilePath);
                            this.SetRequireCalculateFlag(sequenceSound);
                        }
                        else if (sender is SoundSetBankBase)
                        {
                            // TODO: 集計データも古いパスから新しいパスに書き換える必要があります。

                            SoundSetBankBase soundSetBank = sender as SoundSetBankBase;
                            this.ReplaceMonitoringFile(soundSetBank, soundSetBank.FilePath);

                            DispatchWaveFilePath(soundSetBank.FilePath);

                            if (RequireGetReferenceFileData != null)
                            {
                                RequireGetReferenceFileData(this, EventArgs.Empty);
                            }
                        }
                        break;
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void AddMonitoringFile(SoundSetItem sound, string filePath)
            {
                this.fileWatcherService.Add(filePath, OnFileUpdated);
                this.monitoringFiles.Add(sound, filePath);
            }

            /// <summary>
            ///
            /// </summary>
            private void RemoveMonitoringFile(SoundSetItem sound, string filePath)
            {
                this.fileWatcherService.Remove(filePath, OnFileUpdated);
                this.monitoringFiles.Remove(sound);
            }

            /// <summary>
            ///
            /// </summary>
            private void ReplaceMonitoringFile(SoundSetItem sound, string newFilePath)
            {
                string oldFilePath = null;

                if (this.monitoringFiles.TryGetValue(sound, out oldFilePath) == true &&
                    String.Compare(oldFilePath, newFilePath, true) != 0)
                {
                    this.RemoveMonitoringFile(sound, oldFilePath);
                    this.AddMonitoringFile(sound, newFilePath);
                }
            }

            /// <summary>
            ///
            /// </summary>
            private void OnFileUpdated(string filePath)
            {
                string extension = Path.GetExtension(filePath).Replace(".", String.Empty);

                if (String.Compare(extension, this.SequenceFileExtension, true) == 0)
                {
                    this.SetRequireCalculateFlag(filePath);
                }
                else if (String.Compare(extension, this.BankFileExtension, true) == 0)
                {
                    if (this.bankServiceManager.Contains(filePath) == false)
                    {
                        DispatchWaveFilePath(filePath);

                        if (RequireGetReferenceFileData != null)
                        {
                            RequireGetReferenceFileData(this, EventArgs.Empty);
                        }
                    }
                }
            }

            /// <summary>
            /// バンクドキュメントを参照するクラスです。
            ///
            /// 既にバンクドキュメントが開いている場合には開かれているバンクドキュメントを
            /// 開かれていない場合には、開いたバンクドキュメントを保持しています。
            /// </summary>
            public class ReferenceBankDocumentManager : IDisposable
            {
                private SoundIntermediateOutputTraits intermediateOutputTraits = null;

                private BankServiceManager bankServiceManager = null;
                private DocumentService documentService = null;

                Dictionary<string, ReferenceBankDocument> referenceDocuments = new Dictionary<string, ReferenceBankDocument>();

                ///
                public ReferenceBankDocumentManager(DocumentService documentService, BankServiceManager bankServiceManager, SoundIntermediateOutputTraits intermediateOutputTraits)
                {
                    this.intermediateOutputTraits = intermediateOutputTraits;
                    this.documentService = documentService;
                    this.bankServiceManager = bankServiceManager;
                }

                ///
                public Bank GetBank(string filePath)
                {
                    if (this.referenceDocuments.ContainsKey(filePath) == false)
                    {
                        return null;
                    }
                    return this.referenceDocuments[filePath].BankDocument.Bank;
                }

                ///
                public void Add(string filePath)
                {
                    Add(new string[] { filePath }, true);
                }

                ///
                public void Add(string[] filePaths)
                {
                    Add(filePaths, false);
                }

                ///
                public void Add(string[] filePaths, bool forceLoad)
                {
                    foreach (string filePath in filePaths)
                    {
                        if (this.referenceDocuments.ContainsKey(filePath) == false)
                        {
                            ReferenceBankDocument referenceDocument = new ReferenceBankDocument
                                (this.documentService, this.bankServiceManager,
                                  this.intermediateOutputTraits,
                                  filePath, forceLoad);
                            if (referenceDocument.BankDocument != null)
                            {
                                this.referenceDocuments.Add(filePath, referenceDocument);
                            }
                        }
                    }
                }

                ///
                public void Dispose()
                {
                    foreach (ReferenceBankDocument referenceDocument in this.referenceDocuments.Values)
                    {
                        referenceDocument.Dispose();
                    }

                    this.referenceDocuments.Clear();
                }

                ///
                public class ReferenceBankDocument : IDisposable
                {
                    private DocumentReference docRef = null;

                    ///
                    public ReferenceBankDocument(DocumentService documentService, BankServiceManager bankServiceManager, SoundIntermediateOutputTraits intermediateOutputTraits, string filePath, bool forceLoad)
                    {
                        if (bankServiceManager.Contains(filePath) == true &&
                            forceLoad == false)
                        {
                            BankService bankService = bankServiceManager[filePath];
                            this.BankDocument = bankService.BankDocument;
                        }
                        else
                        {
                            try
                            {
                                FileResource resource = new FileResource(filePath);
                                this.docRef = documentService.OpenDocument(resource);
                                if (this.docRef != null &&
                                    this.docRef.Document is BankDocument == true &&
                                    this.docRef.Document.TypeName == intermediateOutputTraits.BankDocumentTypeName)
                                {
                                    this.BankDocument = this.docRef.Document as BankDocument;
                                }
                            }
                            catch (Exception)
                            {
                            }
                        }
                    }

                    ///
                    public void Dispose()
                    {
                        if (this.docRef != null)
                        {
                            this.docRef.Close();
                            this.docRef = null;
                        }
                    }

                    ///
                    public BankDocument BankDocument
                    {
                        get;
                        private set;
                    }
                }
            }

            /// <summary>
            /// 集計されたファイルの参照データを入れるクラスです。
            /// </summary>
            public sealed class ReferenceFileData
            {
                ///
                public ReferenceFileData()
                {
                    this.FileDatas = new Dictionary<string, FileData>();
                }

                ///
                public Dictionary<string, FileData> FileDatas
                {
                    get;
                    private set;
                }

                ///
                public void Add(string waveFilePath, string instrumentName, int count, string bankFilePath)
                {
                    FileData data = null;

                    if (this.FileDatas.TryGetValue(waveFilePath, out data) == false)
                    {
                        data = new FileData(waveFilePath);
                        this.FileDatas.Add(waveFilePath, data);
                    }

                    data.Add(instrumentName, count, bankFilePath);
                }

                /// <summary>
                ///
                /// </summary>
                public sealed class FileData
                {
                    private Dictionary<string, InstrumentData> instrumentDatas = null;

                    ///
                    public FileData(string waveFilePath)
                    {
                        this.WaveFilePath = waveFilePath;
                        this.instrumentDatas = new Dictionary<string, InstrumentData>();

                    }

                    ///
                    public string WaveFilePath
                    {
                        get;
                        private set;
                    }

                    ///
                    public int Count
                    {
                        get
                        {
                            return this.instrumentDatas.Values
                                .Select(d => d.Count)
                                .Sum();
                        }
                    }

                    ///
                    public Dictionary<string, InstrumentData> InstrumentDatas
                    {
                        get
                        {
                            return this.instrumentDatas;
                        }
                    }

                    ///
                    public void Add(string instrumentName, int count, string bankFilePath)
                    {
                        InstrumentData data = null;

                        if (this.instrumentDatas.TryGetValue(instrumentName, out data) == false)
                        {
                            data = new InstrumentData();
                            data.Name = instrumentName;
                            data.BankFilePath = bankFilePath;
                            this.instrumentDatas.Add(instrumentName, data);
                        }

                        data.Count += count;
                    }

                    ///
                    public sealed class InstrumentData
                    {
                        public string Name
                        {
                            get;
                            set;
                        }

                        public string BankFilePath
                        {
                            get;
                            set;
                        }

                        public int Count
                        {
                            get;
                            set;
                        }
                    }
                }
            }

            /// <summary>
            /// 集計された参照データを入れるクラスです。
            /// </summary>
            public sealed class ReferenceCountData
            {
                private int[,] data = new int[MaxKeyCount, MaxVelocityCount];

                ///
                public ReferenceCountData()
                {
                    this.Incorrect = false;
                }

                ///
                public int TotalCount
                {
                    get;
                    internal set;
                }

                ///
                public bool Incorrect
                {
                    get;
                    internal set;
                }

                ///
                public int this[int key, int velocity]
                {
                    get
                    {
                        return this.data[key, velocity];
                    }
                    set
                    {
                        this.data[key, velocity] = value;
                    }
                }

                ///
                public int GetCount(int keyMin, int keyMax, int velocityMin, int velocityMax)
                {
                    int count = 0;

                    for (int key = keyMin; key <= keyMax; key++)
                    {
                        for (int velocity = velocityMin; velocity <= velocityMax; velocity++)
                        {
                            count += this.data[key, velocity];
                        }
                    }
                    return count;
                }
            }

            /// <summary>
            /// 参照データを入れるクラスです。
            /// </summary>
            public sealed class Data
            {
                private Sound sound = null;
                private Dictionary<int, BankData> datas = new Dictionary<int, BankData>();

                ///
                public Data(Sound sound)
                {
                    this.sound = sound;
                }

                ///
                public bool RequireCalculateFlag
                {
                    get;
                    set;
                }

                ///
                public void Count(int bankNo, int programNo, int key, int velocity)
                {
                    if (this.datas.ContainsKey(bankNo) == false)
                    {
                        this.datas.Add(bankNo, new BankData());
                    }
                    this.datas[bankNo].Count(programNo, key, velocity);
                }

                ///
                public string[] GetBankFilePaths()
                {
                    if (this.sound is SequenceSoundBase)
                    {
                        SequenceSoundBase sequenceSound = this.sound as SequenceSoundBase;

                        return this.datas.Keys
                            .Select(i => sequenceSound.SoundSetBanks[i])
                            .Where(s => s != null)
                            .Select(s => s.FilePath)
                            .ToArray();
                    }
                    return null;
                }

                /// ReferenceBankDocumentManagerに登録されているバンクファイルを使い
                /// キー、ベロシティから参照される波形ファイルを決定します。
                public void DispatchWaveFilePath(ReferenceBankDocumentManager manager)
                {
                    if (this.sound is SequenceSoundBase)
                    {
                        SequenceSoundBase sequenceSound = this.sound as SequenceSoundBase;

                        foreach (KeyValuePair<int, BankData> pair in this.datas)
                        {
                            SoundSetBankBase soundSetBank =
                                sequenceSound.SoundSetBanks[pair.Key];

                            if (soundSetBank != null)
                            {
                                string bankFilePath = soundSetBank.FilePath;
                                Bank bank = manager.GetBank(bankFilePath);
                                if (bank != null)
                                {
                                    pair.Value.DispatchWaveFilePath(bank);
                                }
                            }
                        }
                    }
                }

                ///
                public void AddReferenceFileData(ReferenceFileData referenceFileData, ReferenceBankDocumentManager manager)
                {
                    if (this.sound is SequenceSoundBase)
                    {
                        SequenceSoundBase sequenceSound = this.sound as SequenceSoundBase;

                        foreach (KeyValuePair<int, BankData> pair in this.datas)
                        {
                            SoundSetBankBase soundSetBank =
                                sequenceSound.SoundSetBanks[pair.Key];

                            if (soundSetBank != null)
                            {
                                Bank bank = manager.GetBank(soundSetBank.FilePath);
                                pair.Value.AddReferenceFileData
                                    (referenceFileData, soundSetBank.FilePath, bank);
                            }
                        }
                    }
                }

                ///
                public void AddReferenceCountData(ReferenceCountData referenceCountData, string filePath, int programNo)
                {
                    if (this.sound is SequenceSoundBase)
                    {
                        if (this.RequireCalculateFlag == true)
                        {
                            referenceCountData.Incorrect = true;
                        }

                        SequenceSoundBase sequenceSound = this.sound as SequenceSoundBase;

                        foreach (int bankNoDataIndex in this.datas.Keys)
                        {
                            SoundSetBankBase soundSetBank =
                                sequenceSound.SoundSetBanks[bankNoDataIndex];

                            if (soundSetBank != null)
                            {
                                if (String.Compare(soundSetBank.FilePath, filePath, true) == 0)
                                {
                                    BankData bankNoData = this.datas[bankNoDataIndex];
                                    bankNoData.AddReferenceCountData(referenceCountData, programNo);
                                }
                            }
                        }
                    }
                }

                /// <summary>
                ///
                /// </summary>
                class BankData
                {
                    private Dictionary<int, InstrumentData> datas = new Dictionary<int, InstrumentData>();

                    ///
                    public void Count(int programNo, int key, int velocity)
                    {
                        if (this.datas.ContainsKey(programNo) == false)
                        {
                            this.datas.Add(programNo, new InstrumentData());
                        }
                        this.datas[programNo].Count(key, velocity);
                    }

                    ///
                    public void DispatchWaveFilePath(Bank bank)
                    {
                        foreach (KeyValuePair<int, InstrumentData> pair in this.datas)
                        {
                            int programNo = pair.Key;
                            pair.Value.DispatchWaveFilePath(bank, programNo);
                        }
                    }

                    ///
                    public void AddReferenceFileData(ReferenceFileData referenceFileData, string bankFilePath, Bank bank)
                    {
                        foreach (KeyValuePair<int, InstrumentData> pair in this.datas)
                        {
                            int programNo = pair.Key;
                            pair.Value.AddReferenceFileData
                                (referenceFileData, bankFilePath, bank, programNo);
                        }
                    }

                    ///
                    public void AddReferenceCountData(ReferenceCountData keyVelocityData, int programNo)
                    {
                        if (this.datas.ContainsKey(programNo) == true)
                        {
                            this.datas[programNo].AddReferenceCountData(keyVelocityData);
                        }
                    }

                    /// <summary>
                    ///
                    /// </summary>
                    class InstrumentData
                    {
                        private Dictionary<Tuple<int, int>, CountData> datas = new Dictionary<Tuple<int, int>, CountData>();

                        ///
                        class CountData
                        {
                            public CountData()
                            {
                                WaveFilePath = String.Empty;
                                Count = 0;
                            }

                            public string WaveFilePath { get; set; }
                            public int Count { get; set; }
                        }

                        ///
                        public void Count(int key, int velocity)
                        {
                            Tuple<int, int> pairKey = Tuple.Create(key, velocity);
                            if (this.datas.ContainsKey(pairKey) == false)
                            {
                                this.datas.Add(pairKey, new CountData());
                            }
                            this.datas[pairKey].Count++;
                        }


                        ///
                        public void DispatchWaveFilePath(Bank bank, int programNo)
                        {
                            Instrument instrument = BankUtility.GetInstrument(bank, programNo);

                            foreach (KeyValuePair<Tuple<int, int>, CountData> pair in this.datas)
                            {
                                int key = pair.Key.Item1;
                                int velocity = pair.Key.Item2;
                                string waveFilePath = BankUtility.GetFilePath
                                    (instrument, key, velocity);

                                CountData countData = pair.Value;
                                countData.WaveFilePath = waveFilePath;
                            }
                        }

                        ///
                        public void AddReferenceFileData(ReferenceFileData referenceFileData, string bankFilePath, Bank bank, int programNo)
                        {
                            Instrument instrument = BankUtility.GetInstrument(bank, programNo);

                            foreach (KeyValuePair<Tuple<int, int>, CountData> pair in this.datas)
                            {
                                string waveFilePath = pair.Value.WaveFilePath;
                                referenceFileData.Add(waveFilePath, instrument.Name,
                                                       pair.Value.Count, bankFilePath);
                            }
                        }

                        ///
                        public void AddReferenceCountData(ReferenceCountData referenceCountData)
                        {
                            foreach (KeyValuePair<Tuple<int, int>, CountData> pair in this.datas)
                            {
                                int key = pair.Key.Item1;
                                int velocity = pair.Key.Item2;
                                referenceCountData[key, velocity] += pair.Value.Count;
                                referenceCountData.TotalCount += pair.Value.Count;
                            }
                        }
                    }
                }
            }
        }
    }


    /// <summary>
    ///
    /// </summary>
    internal static class BankUtility
    {
        public static string GetFilePath(Bank bank, int programNo, int key, int velocity)
        {
            return GetFilePath(GetInstrument(bank, programNo), key, velocity);
        }

        ///
        public static string GetFilePath(Instrument instrument, int key, int velocity)
        {
            if (instrument != null)
            {
                KeyRegion keyRegion = GetKeyRegion(instrument, key);
                if (keyRegion != null)
                {
                    VelocityRegion velocityRegion = GetVelocityRegion(keyRegion, velocity);
                    if (velocityRegion != null)
                    {
                        return velocityRegion.FilePath;
                    }
                }
            }
            return null;
        }


        ///
        public static Instrument GetInstrument(Bank bank, int programNo)
        {
            foreach (Instrument instrument in bank.Children)
            {
                if (instrument.ProgramNo == programNo)
                {
                    return instrument;
                }
            }
            return null;
        }

        ///
        private static KeyRegion GetKeyRegion(Instrument instrument, int key)
        {
            foreach (KeyRegion keyRegion in instrument.Children)
            {
                if (key >= keyRegion.KeyMin && key <= keyRegion.KeyMax)
                {
                    return keyRegion;
                }
            }
            return null;
        }

        ///
        private static VelocityRegion GetVelocityRegion(KeyRegion keyRegion, int velocity)
        {
            foreach (VelocityRegion velocityRegion in keyRegion.Children)
            {
                if (velocity >= velocityRegion.VelocityMin && velocity <= velocityRegion.VelocityMax)
                {
                    return velocityRegion;
                }
            }
            return null;
        }
    }
}
