﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Net;
using System.Threading;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;

    //-------------------------------------------------------------------------
    // 接続状態変更イベント
    #region ConnectionChangedEvent

    public enum ConnectionState
    {
        Disconnected = 0,   /// 非接続
        Connecting,         /// 接続処理中
        Connected,          /// 接続中
        Disconnecting,      /// 切断処理中
    }

    /// <summary>
    /// 接続状態変更イベントハンドラ

    /// <para>
    /// 接続状態が変化したときに発生するイベントのハンドラです。
    /// </para>
    /// </summary>
    public delegate void ConnectionChangedEventHandler(ConnectionChangedEventArgs args);

    /// <summary>
    /// 接続状態変更イベント引数
    /// <para>
    /// このイベント引数により接続したのか切断したのかを判別できます。
    /// </para>
    /// </summary>
    public sealed class ConnectionChangedEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ConnectionChangedEventArgs(ConnectionState state)
        {
            _state = state;
        }
        public ConnectionChangedEventArgs(ConnectionState state, string errorText)
        {
            _state = state;
            _errorText = errorText;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public bool IsConnected
        {
            get { return (ConnectionState.Connected == _state); }
        }

        /// <summary>
        /// 接続状態
        /// </summary>
        public ConnectionState State
        {
            get { return _state; }
        }

        /// <summary>
        /// エラーが発生したかどうか
        /// </summary>
        public bool IsError
        {
            get { return (_errorText.Length > 0); }
        }

        /// <summary>
        /// エラーテキスト
        /// </summary>
        public string ErrorText
        {
            get { return _errorText; }
        }

        // 接続しているか
        private readonly ConnectionState _state;
        private readonly string _errorText = String.Empty;
    }
    #endregion

    //-------------------------------------------------------------------------
    // コンソールインターフェイス
    #region ICommConsole
    /// <summary>
    /// コンソールインターフェイス
    /// <para>
    /// メッセージ出力先を指定するためのインターフェイス。
    /// このインターフェイスは他のスレッドから呼ばれることに注意する必要があります。
    /// </para>
    /// </summary>
    public interface ICommConsole
    {
        /// <summary>
        /// メッセージ書き出し
        /// </summary>
        void WriteMessage(string format, params object[] args);

        /// <summary>
        /// エラー書き出し
        /// </summary>
        void WriteError(string format, params object[] args);

    }
    #endregion

    /// <summary>
    /// マネージャ
    /// <para>
    /// 管理を行う静的なクラスです。ツールからビューアへパケットを送る接続
    /// （ツール接続）、ビューアからツールへパケットを送る接続（ビューア接続）、
    /// ビューアへのPingを行う接続（Ping接続）の三種類の接続を管理します。
    /// </para>
    /// </summary>
    public class CommManager
    {
        // Lock
        private Object lockObject = new Object();

        // Lock
        private Object lockSndEditObject = new Object();

        ///
        private delegate void SetStateDelegate(ConnectionState state);

        /// <summary>
        ///
        /// </summary>
        public static CommManager Instance
        {
            get;
            set;
        }

        /// <summary>
        /// 接続状態変更イベントハンドラ
        /// </summary>
        public event ConnectionChangedEventHandler ConnectionChanged;

        /// <summary>
        /// SndEdit接続状態変更イベントハンドラ
        /// </summary>
        public event ConnectionChangedEventHandler SndEditConnectionChanged;

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="console">メッセージ出力先となるコンソール</param>
        public virtual void Initialize(Control invokeControl, ICommConsole console)
        {
            if (null == invokeControl)
            {
                this.invokeControl = new Control();
                this.invokeControl.CreateControl();
            }
            else
            {
                this.invokeControl = invokeControl;
            }

            this.console = console;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Uninitialize()
        {
            Disconnect();

            //
            this.invokeControl = null;
            this.console = null;

            //
            ToolConnection = null;
            ViewerConnection = null;
            PingConnection = null;

            SyncConnection = null;
            FuncConnection = null;
            CtrlConnection = null;
        }

        /// <summary>
        /// 接続
        /// </summary>
        public virtual bool Connect()
        {
            return false;
        }

        /// <summary>
        /// SoundPlayer(Viewer)と接続します。
        /// </summary>
        public virtual bool ConnectToSoundPlayer()
        {
            return false;
        }

        /// <summary>
        /// 接続処理中なのか調べます。
        /// </summary>
        public bool IsConnecting
        {
            get { return (ConnectionState.Connecting == this.connectionState); }
        }

        /// <summary>
        /// 接続中なのか調べます。
        /// </summary>
        public bool IsConnected
        {
            get { return (ConnectionState.Connected == this.connectionState); }
        }

        /// <summary>
        /// 切断処理中なのか調べます。
        /// </summary>
        public bool IsDisconnecting
        {
            get
            {
                lock (this.lockObject)
                {
                    return (ConnectionState.Disconnecting == this.connectionState);
                }
            }
        }

        /// <summary>
        /// SndEditが接続処理中なのか調べます。
        /// </summary>
        public bool IsConnectingSndEdit
        {
            get { return (this.connectionStateSndEdit == ConnectionState.Connecting); }
        }

        /// <summary>
        /// SndEditが接続中なのか調べます。
        /// </summary>
        public bool IsConnectedSndEdit
        {
            get { return (this.connectionStateSndEdit == ConnectionState.Connected); }
        }

        /// <summary>
        /// SndEdit切断処理中なのか調べます。
        /// </summary>
        public bool IsDisconnectingSndEdit
        {
            get
            {
                lock (this.lockObject)
                {
                    return (this.connectionStateSndEdit == ConnectionState.Disconnecting);
                }
            }
        }

        /// <summary>
        /// 接続を閉じる
        /// </summary>
        public void Disconnect()
        {
            if (null == InvokeControl) { return; }

            if (InvokeControl.InvokeRequired)
            {
                InvokeControl.BeginInvoke(new MethodInvoker(OnDisconnect));
            }
            else
            {
                OnDisconnect();
            }
        }

        /// <summary>
        /// 強制切断する
        /// </summary>
        public void Terminate()
        {
            if (InvokeControl.InvokeRequired)
            {
                InvokeControl.BeginInvoke(new MethodInvoker(OnTerminate));
            }
            else
            {
                OnTerminate();
            }
        }

        /// <summary>
        /// 接続状態を取得します。
        /// </summary>
        public virtual ConnectionState GetState()
        {
            return this.connectionState;
        }

        /// <summary>
        /// SndEdit接続状態を取得します。
        /// </summary>
        public ConnectionState GetStateSndEdit()
        {
            return this.connectionStateSndEdit;
        }

        /// <summary>
        /// 接続状態を設定します。
        /// </summary>
        public void SetState(ConnectionState state)
        {
            if (InvokeControl.InvokeRequired)
            {
                IAsyncResult result = InvokeControl.BeginInvoke
                    (new SetStateDelegate(OnSetState), state);
            }
            else
            {
                OnSetState(state);
            }
        }

        /// <summary>
        /// SndEdit接続状態を設定します。
        /// </summary>
        public void SetStateSndEdit(ConnectionState state)
        {
            if (InvokeControl.InvokeRequired)
            {
                IAsyncResult result = InvokeControl.BeginInvoke
                    (new SetStateDelegate(OnSetStateSndEdit), state);
            }
            else
            {
                OnSetStateSndEdit(state);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string LastError
        {
            get { return _errorText; }
            set { _errorText = value; }
        }

        /// <summary>
        /// コンソール
        /// </summary>
        public ICommConsole Console
        {
            get
            {
                Debug.Assert(this.console != null, "Console is null");
                return this.console;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Control InvokeControl
        {
            get
            {
                //Debug.Assert( this.invokeControl != null);
                return this.invokeControl;
            }
        }

        /// <summary>
        /// パケット送信
        /// </summary>
        public virtual void SendPacket(CommPacket packet)
        {
            if (ToolConnection.IsConnected == false)
            {
                return;
            }

            ToolPacket toolPacket = packet as ToolPacket;
            if (toolPacket == null)
            {
                string message = String.Format("Illigal packet found {0}", packet.ToString());
                Debug.WriteLine(message);
                return;
            }

            ToolConnection.SendPacket(toolPacket);
        }

        /// <summary>
        /// パケット送信
        /// </summary>
        public void SendPacketByCtrlConnection(CommPacket packet)
        {
            if (CtrlConnection.IsConnected == false)
            {
                return;
            }

            CtrlPacket ctrlPacket = packet as CtrlPacket;
            if (ctrlPacket == null)
            {
                string message = String.Format("Illigal packet found {0}", packet.ToString());
                Debug.WriteLine(message);
                return;
            }

            CtrlConnection.SendPacket(ctrlPacket);
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public virtual bool IsWantToExit
        {
            get
            {
                return true;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection ToolConnection
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection ViewerConnection
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection PingConnection
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection SyncConnection
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection FuncConnection
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommConnection CtrlConnection
        {
            get;
            set;
        }

        /// <summary>
        /// 切断された時に呼ばれます。
        /// </summary>
        protected virtual void OnDisconnected()
        {
        }

        /// <summary>
        /// 接続が変更された時に呼ばれます。
        /// </summary>
        protected void OnConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (ConnectionChanged != null)
            {
#if false
                InvokeControl.BeginInvoke
                    ( ConnectionChanged,
                      new object[] { e });
#else
                ConnectionChanged(e);
#endif
            }
        }

        /// <summary>
        /// SndEdit接続が変更された時に呼ばれます。
        ///
        /// </summary>
        protected void OnSndEditConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (SndEditConnectionChanged != null)
            {
                SndEditConnectionChanged(e);
            }
        }

        /// <summary>
        /// 切断される時に呼ばれます。
        /// </summary>
        private void OnDisconnect()
        {
            DisconnectInternal(Timeout.Infinite);
        }

        /// <summary>
        /// 強制切断される時に呼ばれます。
        /// </summary>
        private void OnTerminate()
        {
            DisconnectInternal(0);
        }

        /// <summary>
        /// 切断処理です。
        /// </summary>
        private void DisconnectInternal(int timeout)
        {
            bool isDisconnected = false;

            //
            ConnectionState state = GetState();
            if (state != ConnectionState.Disconnecting && state != ConnectionState.Disconnected)
            {
                SetState(ConnectionState.Disconnecting);
                ToolConnection.Disconnect(timeout);
                ViewerConnection.Disconnect(timeout);
                PingConnection.Disconnect(timeout);
                SetState(ConnectionState.Disconnected);
                isDisconnected = true;
            }

            state = GetStateSndEdit();
            if (state != ConnectionState.Disconnecting && state != ConnectionState.Disconnected)
            {
                SetStateSndEdit(ConnectionState.Disconnecting);
                SyncConnection.Disconnect(timeout);
                FuncConnection.Disconnect(timeout);
                CtrlConnection.Disconnect(timeout);
                SetStateSndEdit(ConnectionState.Disconnected);
                isDisconnected = true;
            }

            //
            if (isDisconnected == false)
            {
                return;
            }

            OnDisconnected();
        }

        /// <summary>
        /// 接続状態が変更された時に呼ばれます。
        /// </summary>
        private void OnSetState(ConnectionState state)
        {
            if (state == this.connectionState)
            {
                return;
            }

            Debug.Assert(!InvokeControl.InvokeRequired, "InvokeRequired is not false");
            lock (this.lockObject)
            {
                if (state == this.connectionState)
                {
                    return;
                }

                this.connectionState = state;
                OnConnectionChanged
                    (new ConnectionChangedEventArgs(this.connectionState, LastError));
                LastError = String.Empty;
            }
        }

        /// <summary>
        /// SndEdit接続状態が変更された時に呼ばれます。
        /// </summary>
        private void OnSetStateSndEdit(ConnectionState state)
        {
            if (state == this.connectionStateSndEdit)
            {
                return;
            }

            Debug.Assert(!InvokeControl.InvokeRequired, "InvokeRequired is not false");
            lock (this.lockSndEditObject)
            {
                if (state == this.connectionStateSndEdit)
                {
                    return;
                }

                this.connectionStateSndEdit = state;

                OnSndEditConnectionChanged
                    (new ConnectionChangedEventArgs(this.connectionStateSndEdit, LastError));
                LastError = String.Empty;
            }
        }

        // Invoke用コントロール
        private Control invokeControl;

        // メッセージ出力用コンソール
        private ICommConsole console;

        // 接続状態
        private ConnectionState connectionState = ConnectionState.Disconnected;

        // SndEdit接続状態
        private ConnectionState connectionStateSndEdit = ConnectionState.Disconnected;

        // Last Error
        private string _errorText = String.Empty;
    }
}
