﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Projects;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// サウンドセットにサウンドを追加します。
    /// </summary>
    public abstract class AddSoundHandler : AddSoundSetItemHandler
    {
        private string _defaultPlayerName = string.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">対象コマンドID。</param>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        /// <param name="defaultPlayerName">デフォルトプレイヤー名。</param>
        protected AddSoundHandler(string commandID, bool insert,
                                   IQueryCommandParameter queryParameter, string defaultPlayerName)
            : base(commandID, insert, queryParameter)
        {
            if (null == defaultPlayerName) { throw new ArgumentNullException("defaultPlayerName"); }
            _defaultPlayerName = defaultPlayerName;
        }

        /// <summary>
        /// デフォルトプレイヤー名を取得します。
        /// </summary>
        public string DefaultPlayerName
        {
            get { return _defaultPlayerName; }
        }

        /// <summary>
        /// サウンドセットにサウンドを追加します。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="document">追加先のサウンドセットドキュメント。</param>
        /// <param name="parentItem">追加先のサウンドセットアイテム。</param>
        /// <param name="insertTargetSound">挿入位置のサウンド。</param>
        /// <param name="newSound">追加するサウンド。</param>
        /// <param name="defaultPlayerName">デフォルトプレイヤー名。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundProjectService projectService,
                                    SoundSetDocument document, SoundSetItem parentItem,
                                    Sound insertTargetSound, Sound newSound, string defaultPlayerName)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == parentItem) { throw new ArgumentNullException("parentItem"); }
            if (null == newSound) { throw new ArgumentNullException("newSound"); }
            if (null == defaultPlayerName) { throw new ArgumentNullException("defaultPlayerName"); }

            try
            {

                document.OperationHistory.BeginTransaction();

                Execute(document, parentItem, insertTargetSound, newSound);

                if (newSound.PlayerReference == defaultPlayerName)
                {
                    AddDefaultPlayer(projectService, document, defaultPlayerName);
                }

                //
                EventRouter router = null;

                router = ((IEventRoutable)newSound).Router;
                router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(),
                                      newSound));
                document.OperationHistory.EndTransaction();
            }
            catch
            {
                document.OperationHistory.CancelTransaction();
            }

            return true;
        }

        /// <summary>
        /// デフォルトプレイヤーを追加します。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="document">追加先のサウンドセットドキュメント。</param>
        /// <param name="defaultPlayerName">デフォルトプレイヤー名。</param>
        public static void AddDefaultPlayer(SoundProjectService projectService,
                                              SoundSetDocument document, string defaultPlayerName)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == document.SoundSet) { throw new ArgumentException("target soundset not found."); }
            if (null == defaultPlayerName) { throw new ArgumentNullException("defaultPlayerName"); }

            string playerName = defaultPlayerName;

            if (projectService.ComponentDictionary.Contains(defaultPlayerName))
            {

                if ((from Component component in projectService.ComponentDictionary[defaultPlayerName]
                     where component is PlayerBase
                     select component).FirstOrDefault() != null) { return; }

                throw new ApplicationException("default player name is already existed.");

            }

            PlayerPack playerPack = (from Component component in document.SoundSet.Children
                                     where component is PlayerPack
                                     select component as PlayerPack).FirstOrDefault();

            AddSoundSetItemHandler.Execute(document, playerPack, null, new PlayerBase() { Name = playerName });
        }

        /// <summary>
        /// サウンドセットにアイテムを追加します。
        /// </summary>
        /// <param name="newItem">追加するサウンドセットアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool Execute(IQueryCommandParameter parameters, SoundSetItem newItem)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }
            if (!(newItem is Sound)) { throw new ArgumentException("invalid item type."); }

            SoundSetDocument targetDocument = GetTargetDocument(parameters);
            Component targetComponent = GetInsertParentComponent(parameters);

            if (!(targetComponent is SoundSetItem) ||
                 (targetComponent as SoundSetItem).SoundSet != targetDocument.SoundSet)
            {
                throw new ArgumentException("target component is invalid.");
            }

            return Execute(Application.ProjectService, targetDocument,
                            targetComponent as SoundSetItem,
                            GetInsertTargetComponent(parameters) as Sound,
                            newItem as Sound,
                            _defaultPlayerName);
        }
    }
}
