﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;

    /// <summary>
    /// プロジェクトを開きます。
    /// </summary>
    public abstract class OpenProjectHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected OpenProjectHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.OpenProject.ID; }
        }

        /// <summary>
        /// File.OpenProject コマンドを実行します。
        /// </summary>
        /// <param name="projectService">プロジェクトサービス。</param>
        /// <param name="filePath">ファイルパス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected bool ExecuteInternal(SoundProjectService projectService, string filePath)
        {
            if (null != projectService.ProjectDocument &&
                projectService.ProjectDocument.Resource.Key == filePath) { return true; }

            List<DocumentReference> docRefList = null;
            try
            {
                // ファイルのバージョンを確認します。
                bool oldFileAllForceSave = false;
                bool isReadOnly = false;
                List<string> newVersionFiles = new List<string>();
                List<string> oldVersionFiles = new List<string>();
                docRefList = FileVersionUtility.GetDifferentVersionFilesWithProjectFile(filePath, ref newVersionFiles, ref oldVersionFiles);

                if (0 < newVersionFiles.Count)
                {
                    string text = string.Empty;
                    foreach (string newVersionFile in newVersionFiles)
                    {
                        text += newVersionFile + "\n";
                    }

                    TextDisplayMessageBox messageBox =
                        new TextDisplayMessageBox(MessageResource.Message_UnsupportedFileVersion,
                                                  text,
                                                  TextDisplayMessageBoxStyle.OKButton);
                    messageBox.ShowDialog();

                    return false;
                }
                else if (0 < oldVersionFiles.Count)
                {
                    UpgradeFileDialog dialog = new UpgradeFileDialog();
                    switch (dialog.ShowDialog())
                    {
                        case DialogResult.Cancel:
                            return false;

                        case DialogResult.No: // 読み取り専用で開く
                            isReadOnly = true;
                            break;

                        case DialogResult.OK:
                            oldFileAllForceSave = true;
                            isReadOnly = false;
                            break;
                    }
                }

                try
                {
                    ApplicationBase.Instance.FileWatcherService.BeginDelayAdd();
                    try
                    {
                        // プロジェクトファイルに付随したコンポーネントのパラメータ初期値設定ファイルを読み込みます。
                        var defaultparamFilePath = ApplicationBase.Instance.GetDefaultParametersFilePath(filePath);
                        ApplicationBase.Instance.CreateComponentService.LoadDefaultParametersFile(defaultparamFilePath);
                    }
                    catch (Exception e)
                    {
                        var title = Resources.MessageResource.Message_ReadErrorDefaultParameterFile;
                        var messageBox = new TextDisplayMessageBox(title, e.Message, TextDisplayMessageBoxStyle.OKButton);
                        messageBox.ShowDialog();
                    }
                    projectService.Open(filePath, isReadOnly);
                }
                catch
                {
                    ApplicationBase.Instance.FileWatcherService.CancelDelayAdd();
                    throw;
                }
                finally
                {
                    ApplicationBase.Instance.FileWatcherService.EndDelayAdd();
                }

                // 旧バージョンを全て保存する指示があったなら保存する。
                if (oldFileAllForceSave == true)
                {
                    SaveHelper.SaveForce(oldVersionFiles.ToArray());
                }

                DocumentService documentService = ApplicationBase.Instance.DocumentService;

                // 全ての読み取り専用をクリア
                foreach (Document document in documentService.Documents)
                {
                    if (document is SoundDocument == true)
                    {
                        (document as SoundDocument).IsReadOnly = false;
                    }
                }

                if (isReadOnly == true)
                {
                    // 旧バージョンのみ読み取り専用に設定
                    foreach (string oldFilePath in oldVersionFiles)
                    {
                        SoundDocument document = documentService.Documents.Where(a => a.Resource.Key == oldFilePath).FirstOrDefault() as SoundDocument;

                        if (document != null)
                        {
                            document.IsReadOnly = true;
                        }
                    }
                }
            }
            finally
            {
                if (docRefList != null)
                {
                    foreach (DocumentReference docRef in docRefList)
                    {
                        docRef.Close();
                    }
                }
            }

            ApplicationBase.Instance.SetSndEditConfiguration();

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(
                                                             Command command,
                                                             IQueryCommandParameter parameters
                                                             )
        {
            return (null != GetTargetProjectService(parameters)) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 対象プロジェクトを取得します。
        /// </summary>
        protected SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }
    }
}
