﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    // HACK : ★バンクファイルを開くことだけを考えた暫定実装です。
    /// <summary>
    /// ファイルを開きます。
    /// </summary>
    public abstract class OpenFileHandler : CommandHandler
    {
        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.OpenFile.ID; }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        protected ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        /// <summary>
        /// バンクサービスディクショナリを取得します。
        /// </summary>
        protected BankServiceManager BankServices
        {
            get { return Application.BankServices; }
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <param name="filePath">ファイルパスを指定します。</param>
        /// <param name="bankServiceRef">バンクサービスへの参照があれば返します。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(string filePath, out BankServiceReference bankServiceRef)
        {
            bankServiceRef = null;

            if (ApplicationBase.Instance.BankServices.Contains(filePath))
            {
                return true;
            }

            try
            {
                bankServiceRef = ApplicationBase.Instance.BankServices.OpenItem(filePath);
            }
            catch (FileNotSupportedException exception)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                                exception.Message,
                                                                                exception.FilePath));
                return false;
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                    string.Format(Resources.MessageResource.Message_UnknownError, filePath));
                return false;
            }

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <param name="filePath">ファイルパス。</param>
        /// <param name="bankServiceRef">バンクサービスへの参照があれば返します。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected bool OpenFile(string filePath, out BankServiceReference bankServiceRef)
        {
            return Execute(filePath, out bankServiceRef);
        }
    }
}
