﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Linq;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Midi;
    using NintendoWare.SoundFoundation.Core.Wave;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.SoundMakerPlugin;

    public partial class OptionDialogCommon : OptionDialog
    {
        public static bool VisibleSoundPlayerTab;

        private MidiDevice[] midiInputDevices = new MidiDevice[0];

        public OptionDialogCommon()
        {
            InitializeComponent();

            if (VisibleSoundPlayerTab == false)
            {
                optionTabCtrl.TabPages.Remove(tabPage_SoundPlayer);
            }

            this.InitializeAudioOutputDeviceList();
            this.InitializeMidiInputDeviceList();

            bool supportVisualStyle = SupportVisualStyle;

            //
            midiSequencerInputDeviceListView.OwnerDraw = true;
            midiSequencerInputDeviceListView.DrawItem +=
                delegate (object sender, DrawListViewItemEventArgs e)
                {
                    ListView listView = sender as ListView;
                    Graphics gc = e.Graphics;
                    Rectangle bounds = listView.GetItemRect(e.ItemIndex,
                                                             ItemBoundsPortion.Entire);
                    Rectangle textBounds = listView.GetItemRect(e.ItemIndex,
                                                                 ItemBoundsPortion.Label);
                    Rectangle iconBounds = new Rectangle(bounds.X, bounds.Y,
                                                          textBounds.X - bounds.X, bounds.Height);
                    ButtonState state;
                    Brush brush = null;

                    MidiDevice targetDevice = null;
                    targetDevice = midiKeyboardInputDeviceComboBox.SelectedItem as MidiDevice;

                    //
                    iconBounds.Inflate(-1, -1);
                    if (supportVisualStyle == false)
                    {
                        if (e.Item.Tag == targetDevice)
                        {
                            state = ButtonState.Inactive;
                            brush = new SolidBrush(SystemColors.GrayText);
                        }
                        else
                        {
                            state = ButtonState.Normal;
                            brush = new SolidBrush(e.Item.ForeColor);
                        }

                        if (e.Item.Checked != false)
                        {
                            state |= ButtonState.Checked;
                        }

                        ControlPaint.DrawCheckBox(gc, iconBounds, state);
                        gc.DrawString(e.Item.Text, listView.Font, brush, textBounds);
                    }
                    else
                    {
                        VisualStyleRenderer renderer = null;
                        VisualStyleElement style = null;
                        bool disabled = false;

                        if (e.Item.Tag == targetDevice)
                        {
                            style = e.Item.Checked != false ?
                                VisualStyleElement.Button.CheckBox.CheckedDisabled :
                                VisualStyleElement.Button.CheckBox.UncheckedDisabled;
                            disabled = true;
                        }
                        else
                        {
                            style = e.Item.Checked != false ?
                                VisualStyleElement.Button.CheckBox.CheckedNormal :
                                VisualStyleElement.Button.CheckBox.UncheckedNormal;
                        }

                        renderer = new VisualStyleRenderer(style);
                        renderer.DrawBackground(gc, iconBounds);
                        renderer.DrawText(gc, textBounds, e.Item.Text, disabled,
                                          TextFormatFlags.Left);
                    }
                };
            midiSequencerInputDeviceListView.ItemCheck +=
                delegate (object sender, ItemCheckEventArgs e)
                {
                    ListView listView = sender as ListView;
                    ListViewItem item = listView.Items[e.Index];
                    Assertion.Operation.ObjectNotNull(item);

                    MidiDevice targetDevice = null;
                    targetDevice = midiKeyboardInputDeviceComboBox.SelectedItem as MidiDevice;

                    if (item.Tag == targetDevice)
                    {
                        e.NewValue = e.CurrentValue;
                    }
                };
        }

        //-----------------------------------------------------------------
        // サウンドタブパラメータへのアクセサ
        //-----------------------------------------------------------------

        public string AudioOutputDeviceName
        {
            get { return this.audioOutputDeviceComboBox.Text; }
            set { this.audioOutputDeviceComboBox.SelectedItem = value; }
        }

        //-----------------------------------------------------------------
        // MIDIタブパラメータへのアクセサ
        //-----------------------------------------------------------------

        public MidiDevice MidiKeyboardInputDevices
        {
            get
            {
                return this.midiKeyboardInputDeviceComboBox.SelectedItem as MidiDevice;
            }
            set
            {
                if (value == null)
                {
                    return;
                }

                int i = 0;

                foreach (object obj in this.midiKeyboardInputDeviceComboBox.Items)
                {
                    MidiDevice device = obj as MidiDevice;

                    if (device != null && device.Name == value.Name)
                    {
                        this.midiKeyboardInputDeviceComboBox.SelectedIndex = i;
                        break;
                    }

                    i++;
                }
            }
        }

        public IEnumerable<MidiDevice> midiSequencerInputDevices
        {
            get
            {
                List<MidiDevice> list = new List<MidiDevice>();
                foreach (ListViewItem item in this.midiSequencerInputDeviceListView.Items)
                {
                    if (item.Checked != false)
                    {
                        list.Add(item.Tag as MidiDevice);
                    }
                }
                return list.ToArray();
            }
            set
            {
                Ensure.Argument.NotNull(value);

                HashSet<string> targetDeviceNames = new HashSet<string>();

                foreach (MidiDevice device in value)
                {
                    if (!targetDeviceNames.Contains(device.Name))
                    {
                        targetDeviceNames.Add(device.Name);
                    }
                }

                for (int i = 0; i < this.midiSequencerInputDeviceListView.Items.Count; i++)
                {
                    ListViewItem item = this.midiSequencerInputDeviceListView.Items[i];
                    MidiDevice device = item.Tag as MidiDevice;

                    if (targetDeviceNames.Contains(device.Name))
                    {
                        item.Checked = true;
                    }
                }
            }
        }

        //-----------------------------------------------------------------
        // SoundPlayerタブパラメータへのアクセサ
        //-----------------------------------------------------------------

        public int SoundPlayerPort0
        {
            get { return (int)this.numericUpDown_Port0.Value; }
            set { this.numericUpDown_Port0.Value = value; }
        }

        public int SoundPlayerPort1
        {
            get { return (int)this.numericUpDown_Port1.Value; }
            set { this.numericUpDown_Port1.Value = value; }
        }

        public int SoundPlayerPort2
        {
            get { return (int)this.numericUpDown_Port2.Value; }
            set { this.numericUpDown_Port2.Value = value; }
        }

        public int SoundPlayerPort3
        {
            get { return (int)this.numericUpDown_Port3.Value; }
            set { this.numericUpDown_Port3.Value = value; }
        }

        public int SoundPlayerPort4
        {
            get { return (int)this.numericUpDown_Port4.Value; }
            set { this.numericUpDown_Port4.Value = value; }
        }

        public int SoundPlayerPort5
        {
            get { return (int)this.numericUpDown_Port5.Value; }
            set { this.numericUpDown_Port5.Value = value; }
        }

        public string SoundPlayerDvdRoot
        {
            get { return this.soundPlayerDvdRootTextBox.Text; }
            set { this.soundPlayerDvdRootTextBox.Text = value; }
        }

        //-----------------------------------------------------------------

        private void InitializeAudioOutputDeviceList()
        {
            foreach (var deviceName in SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.EnumerateAudioDeviceNames())
            {
                this.audioOutputDeviceComboBox.Items.Add(deviceName);
            }

            if (this.audioOutputDeviceComboBox.Items.Count > 0)
            {
                this.audioOutputDeviceComboBox.SelectedIndex = 0;
            }
        }

        private void InitializeMidiInputDeviceList()
        {
            List<MidiDevice> midiInputDevicesWork = new List<MidiDevice>();
            List<object> objList = new List<object>();

            objList.Add(Resources.MessageResource.MenuText_MIDI_None);

            int deviceID = 0;

            foreach (MidiInputDevice device in MidiDeviceManager.InputDevices)
            {
                MidiDevice midiDevice = new MidiDevice()
                {
                    DeviceID = deviceID,
                    Name = device.Name,
                };
                midiInputDevicesWork.Add(midiDevice);
                objList.Add(midiDevice);
                deviceID++;
            }

            this.midiInputDevices = midiInputDevicesWork.ToArray();

            if (this.midiInputDevices.Length > 0)
            {
                this.midiKeyboardInputDeviceComboBox.Items.AddRange(objList.ToArray());
                this.midiKeyboardInputDeviceComboBox.SelectedIndex = 0;

                foreach (MidiDevice device in this.midiInputDevices)
                {
                    ListViewItem item = new ListViewItem();
                    item.Text = device.ToString();
                    item.Tag = device;
                    this.midiSequencerInputDeviceListView.Items.Add(item);
                }
            }
            else
            {
                this.midiKeyboardInputDeviceComboBox.Enabled = false;
                this.midiSequencerInputDeviceListView.Enabled = false;
            }
        }

        private void ValidateSoundPlayerDvdRoot()
        {
            ValidationResult result = FilePathValidator.ValidateFilePath(this.SoundPlayerDvdRoot);

            if (!result.IsValid)
            {
                this.errorProvider.SetError(this.soundPlayerDvdRootTextBox, result.ToString());
                this.okButton.Enabled = false;
                return;
            }

            if (Regex.Match(this.SoundPlayerDvdRoot, @"^[a-zA-Z]:\\$").Success)
            {
                this.errorProvider.SetError(
                    this.soundPlayerDvdRootTextBox,
                    Resources.MessageResource.Message_MustNotBeRootDirectory);
                this.okButton.Enabled = false;
                return;
            }

            this.errorProvider.SetError(this.soundPlayerDvdRootTextBox, string.Empty);
            this.okButton.Enabled = true;
        }

        ///
        private bool SupportVisualStyle
        {
            get
            {
                if (Application.RenderWithVisualStyles != false &&
                    VisualStyleRenderer.IsElementDefined
                    (VisualStyleElement.Button.CheckBox.UncheckedDisabled) != false &&
                    VisualStyleRenderer.IsElementDefined
                    (VisualStyleElement.Button.CheckBox.CheckedDisabled) != false &&
                    VisualStyleRenderer.IsElementDefined
                    (VisualStyleElement.Button.CheckBox.UncheckedNormal) != false &&
                    VisualStyleRenderer.IsElementDefined
                    (VisualStyleElement.Button.CheckBox.CheckedNormal) != false)
                {
                    return true;
                }
                return false;
            }
        }

        //-----------------------------------------------------------------

        private void OnSoundPlayerDvdRootChanged(object sender, EventArgs e)
        {
            this.ValidateSoundPlayerDvdRoot();
        }

        private void OnSoundPlayerDvdRootReferClicked(object sender, EventArgs e)
        {
            FolderBrowserDialog dialog = new FolderBrowserDialog();
            dialog.SelectedPath = this.SoundPlayerDvdRoot;

            if (dialog.ShowDialog(this) == DialogResult.OK)
            {
                this.SoundPlayerDvdRoot = dialog.SelectedPath;
            }
        }

        //-----------------------------------------------------------------

        public class MidiDevice
        {
            public MidiDevice()
            {
                this.DeviceID = -1;
                this.Name = string.Empty;
            }

            public int DeviceID { get; set; }

            public string Name { get; set; }

            public override string ToString()
            {
                return this.Name;
            }
        }

        ///
        private void OnMidiKeyboardInput_SelectedIndexChanged(object sender, EventArgs e)
        {
            //MidiDevice device = midiKeyboardInputDeviceComboBox.SelectedItem as MidiDevice;
            midiSequencerInputDeviceListView.Invalidate();
        }

        private void OnClick_button_DefaultPort(object sender, EventArgs e)
        {
            this.numericUpDown_Port0.Value = 54086;
            this.numericUpDown_Port1.Value = 54087;
            this.numericUpDown_Port2.Value = 54088;
            this.numericUpDown_Port3.Value = 54084;
            this.numericUpDown_Port4.Value = 54085;
            this.numericUpDown_Port5.Value = 54099;
        }
    }
}
