﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Windows.Forms;
    using System.Xml;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Communications;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using MCS = NintendoWare.SoundMaker.Preview.MCS;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    /// <summary>
    /// プレビュープレイヤーコントロール
    /// </summary>
    public partial class PreviewPlayerControl : NUserControl
    {
        private const string clipboardDataName = "NintendoWareSoundMakerData";

        PreviewPlayer _previewPlayer;

        Color _headerForeColor;
        Color _headerBackColor;
        Color _headerForeColorSelected;
        Color _headerBackColorSelected;
        Color _headerForeColorSelectedNoFocus;
        Color _headerBackColorSelectedNoFocus;

        Color _itemForeColor;
        Color _itemBackColor;
        Color _itemForeColorSelected;
        Color _itemBackColorSelected;
        Color _itemForeColorSelectedNoFocus;
        Color _itemBackColorSelectedNoFocus;

        string _targetSoundName = string.Empty;
        Sound _targetSound = null;

        bool _selected = false;
        bool _focused = false;

        public PreviewPlayerControl(PreviewPlayer player)
        {
            Debug.Assert(null != player);
            InitializeComponent();

            // ボタンの初期イメージを設定する
            _play.Image = ButtonImages.Images[(int)Images.ButtonID.Play_Disabled];
            _pause.Image = ButtonImages.Images[(int)Images.ButtonID.Pause_Disabled];
            _stop.Image = ButtonImages.Images[(int)Images.ButtonID.Stop_Disabled];
            _repeat.Image = ButtonImages.Images[(int)Images.ButtonID.Repeat_Disabled];

            // 状態アイコンの初期イメージを設定する
            _state.Image = StateImages.Images[(int)Images.StateID.Disabled];

            // コンテキストメニューの初期イメージを設定する
            _toolStripMenuItemDelete.Image = FrameworkResources.ImageResource.BitmapIconDelete;

            // インデックスを設定する（以後、読み取り専用）
            _previewPlayer = player;
            _index.Text = player.Index.ToString();

            // イベントハンドラを設定する
            player.SoundChanged += OnSoundChanged;
            player.PlayerStateChanged += OnPreviewPlayerStateChanged;
            player.RepeatChanged += OnPreviewPlayerRepeatChanged;
            player.MuteChanged += delegate (object sender, EventArgs e)
                {
                    UpdateName();
                };

            ProjectService.Closed += OnProjectClosed;

            FormsApplicationCommon.Instance.AppActivated += OnAppActivated;
            FormsApplicationCommon.Instance.AppDeactivated += OnAppDeactivated;

            //
            this.lbl_Mute.Image = FrameworkResources.ImageResource.BitmapIconPreviewMute;
            this.lbl_Mute.Visible = false;
            this.lbl_SoloPlay.Image = FrameworkResources.ImageResource.BitmapIconPreviewSoloPlay;
            this.lbl_SoloPlay.Visible = false;

            _toolStripMenuItemDelete.Enabled = false;
        }

        #region ** プロパティ

        /// <summary>
        /// プレビュープレイヤーのインデックス
        /// </summary>
        public uint Index
        {
            get { return _previewPlayer.Index; }
        }

        /// <summary>
        /// プレビュープレイヤーを取得します。
        /// </summary>
        public PreviewPlayer PreviewPlayer
        {
            get { return _previewPlayer; }
        }

        /// <summary>
        /// 関連付けられたアイテム
        /// </summary>
        public Sound Item
        {
            get { return _previewPlayer.Item; }
            set
            {
                if (null == value)
                {
                    Reset();
                }
                else
                {
                    Attach(value.Name);
                }
            }
        }

        /// <summary>
        /// 選択状態
        /// </summary>
        public bool Selected
        {
            get { return _selected; }
            set
            {
                if (value == _selected)
                    return;

                UpdateState(value);

                if (!(_focused) && value)
                {
                    Focus();
                }

                UpdateColors();
            }
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplicationCommon.Instance.ProjectService; }
        }

        private CommunicationService CommunicationService
        {
            get
            {
                return FormsApplicationCommon.Instance.CommunicationService;
            }
        }

        private Image IconImage
        {
            get
            {
                switch (_previewPlayer.State)
                {
                    case PreviewPlayerState.Stopped:
                        return StateImages.Images[(int)Images.StateID.Stopped];

                    case PreviewPlayerState.Preparing:
                        return StateImages.Images[(int)Images.StateID.Preparing];

                    case PreviewPlayerState.Playing:
                        return StateImages.Images[(int)Images.StateID.Playing];

                    case PreviewPlayerState.Paused:
                        return StateImages.Images[(int)Images.StateID.Paused];
                }

                return StateImages.Images[(int)Images.StateID.Disabled];
            }
        }

        private Images.IconID IconID
        {
            get
            {
                if (null == Item)
                    return Images.IconID.None;

                if (Item is StreamSoundBase)
                {
                    return Images.IconID.Stream;
                }
                if (Item is WaveSoundBase)
                {
                    return Images.IconID.WaveSoundCo;
                }
                if (Item is SequenceSoundBase)
                {
                    return Images.IconID.Seq;
                }

                return Images.IconID.None;
            }
        }

        private Rectangle ImageRect
        {
            get
            {
                return new Rectangle(_name.Left - 3 - 16, (ClientRectangle.Height - 16) / 2, 16, 16);
            }
        }

        private ImageList IconImages
        {
            get { return Images.IconImages; }
        }

        private ImageList ButtonImages
        {
            get { return Images.ButtonImages; }
        }

        private ImageList StateImages
        {
            get { return Images.StateImages; }
        }

        #endregion

        /// <summary>
        ///
        /// </summary>
        public void Play()
        {
            this._play.PerformClick();
        }

        /// <summary>
        ///
        /// </summary>
        public void Stop()
        {
            this._stop.PerformClick();
        }

        #region ** イベント

        public event EventHandler SelectedChanged;

        public delegate bool DialogKeyEventHandler(object sender, KeyEventArgs e);
        public event DialogKeyEventHandler DialogKeyProcessed;

        #endregion

        #region ** イベントハンドラ

        protected override bool ProcessDialogKey(Keys keyData)
        {
            if (null != DialogKeyProcessed)
            {
                bool br = DialogKeyProcessed(this, new KeyEventArgs(keyData));
                if (br) { return true; }
            }

            // プレビュープレイヤーが無効な場合は、←→で次のアイテムに選択が移るのを防ぐ
            if (PreviewPlayerState.Disabled == _previewPlayer.State)
            {

                switch (keyData)
                {
                    case Keys.Left:
                    case Keys.Right:
                        return true;
                }

            }

            return base.ProcessDialogKey(keyData);
        }

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_MOUSEACTIVATE:
                    // コントロール上をクリックされたらフォーカスを設定する
                    Focus();
                    m.Result = (IntPtr)Win32.MA.MA_ACTIVATE;
                    return;
            }

            base.WndProc(ref m);
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            InitializeColor();

            FormsApplicationCommon.Instance.UIService.MainWindow.ProjectReloaded += OnProjectReloaded;
        }

        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);
            _focused = true;

            UpdateState(true);
            UpdateColors();
        }

        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);
            _focused = false;

            UpdateColors();
        }

        private void OnAppActivated(object sender, EventArgs e)
        {
            _focused = ContainsFocus;
            UpdateColors();
        }

        private void OnAppDeactivated(object sender, EventArgs e)
        {
            _focused = false;
            UpdateColors();
        }

        private void OnSoundChanged(PreviewPlayer sender, EventArgs e)
        {
            UpdateItem(false);
        }

        private void OnSoundParameterChanged(object sender, ParameterEventArgs e)
        {
            if (e.Key != ProjectParameterNames.Name &&
                e.Key != ProjectParameterNames.FilePath)
            {
                return;
            }

            if (e.Key == ProjectParameterNames.FilePath)
            {
                Stop();
            }

            UpdateName();
        }

        private void OnPreviewPlayerStateChanged(PreviewPlayer sender, EventArgs e)
        {
            // ボタンの状態を更新する
            UpdateButtons();

            // 状態アイコンを更新する
            _state.Image = IconImage;
        }

        private void OnPreviewPlayerRepeatChanged(PreviewPlayer sender, EventArgs e)
        {
            _repeat.Checked = sender.Repeat;
        }

        private void OnProjectReloaded(object sender, EventArgs e)
        {
            UpdateItem();
        }

        private void OnProjectClosed(object sender, EventArgs e)
        {
            UpdateItem();
        }

        private void OnDragEnter(object sender, DragEventArgs e)
        {
            if (FormsApplicationCommon.Instance.UIService.MainWindow.CanDropFiles(e))
            {
                e.Effect = DragDropEffects.Copy;
                return;
            }

            byte[] data = null;

            if (!e.Data.GetDataPresent(clipboardDataName)) { return; }
            if ((data = e.Data.GetData(clipboardDataName) as byte[]) == null) { return; }

            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(new MemoryStream(data));

            if (GetTargetSoundName(xmlDoc) != null && PreviewPlayer.Stopped)
            {
                e.Effect = DragDropEffects.Link;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        private void OnDragOver(object sender, DragEventArgs e)
        {
            OnDragEnter(sender, e);
        }

        private void OnDragDrop(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {

                if (FormsApplicationCommon.Instance.UIService.MainWindow.CanDropFiles(e))
                {
                    string[] filePaths = e.Data.GetData(DataFormats.FileDrop, false) as string[];
                    FormsApplicationCommon.Instance.UIService.MainWindow.DropFiles(filePaths);
                    return;
                }

            }

            byte[] data = null;

            if (!e.Data.GetDataPresent(clipboardDataName)) { return; }
            if ((data = e.Data.GetData(clipboardDataName) as byte[]) == null) { return; }

            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(new MemoryStream(data));

            Attach(GetTargetSoundName(xmlDoc));
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPlayClick(object sender, EventArgs e)
        {
            FormsApplicationCommon.Instance.LoudnessService.Stop();
            PreviewPlayerOperator.CheckExceptionForPlay(() =>
            {
                var streamSound = _targetSound as StreamSoundBase;
                try
                {
                    // ストリームサウンドの場合は Mute, Solo 設定をプレビューに反映させます。
                    // （ラウドネス計算など、他のプレビュー再生に作用しないようにするため、設定は再生開始後ただちに削除します。）
                    PreviewStreamSound.AddPlayMuteParameter(streamSound);
                    _previewPlayer.Play();
                }
                catch
                {
                    this.Item = null;
                    throw;
                }
                finally
                {
                    PreviewStreamSound.RemovePlayMuteParameter(streamSound);
                }
            });
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPauseClick(object sender, EventArgs e)
        {
            _previewPlayer.Pause();
        }

        private void OnStopClick(object sender, EventArgs e)
        {
            _previewPlayer.Stop();
        }

        private void OnRepeatCheckedChanged(object sender, EventArgs e)
        {
            CheckBox target = sender as CheckBox;
            _previewPlayer.Repeat = target.Checked;

            UpdateButtons();
        }

        private void OnChildlenEnter(object sender, EventArgs e)
        {
            _focused = true;
            UpdateColors();
        }

        private void OnChildlenLeave(object sender, EventArgs e)
        {
            _focused = false;
            UpdateColors();
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnPaintBackground(PaintEventArgs e) { }

        protected override void OnSystemColorsChanged(EventArgs e)
        {
            base.OnSystemColorsChanged(e);

            InitializeColor();
        }

        private void OnPaint(object sender, PaintEventArgs e)
        {
            // グリッドを描画する
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Top,
                                                         ClientRectangle.Right, ClientRectangle.Top);
            e.Graphics.DrawLine(SystemPens.ControlDark, ClientRectangle.Left, ClientRectangle.Bottom - 1,
                                                         ClientRectangle.Right, ClientRectangle.Bottom - 1);

            // アイコンを描画する
            if (Images.IconID.None == IconID) { return; }
            IconImages.Draw(e.Graphics, ImageRect.Location, (int)IconID);
        }

        private void toolStripMenuItemDelete_Click(object sender, EventArgs e)
        {
            this.Item = null;
        }

        #endregion

        #region ** メソッド

        public void Attach(string soundName)
        {
            _targetSoundName = (null == soundName) ? string.Empty : soundName;

            SetItemInternal(GetValidSound(soundName, null));
        }

        public void UpdateItem()
        {
            UpdateItem(true);
        }

        public void UpdateItem(bool checkItem)
        {
            if (checkItem)
            {
                SetItemInternal(GetValidSound(_targetSoundName, Item));
            }

            if (_targetSound != _previewPlayer.Item)
            {

                if (null != _targetSound)
                {
                    _targetSound.Parameters.ParameterValueChanged -= OnSoundParameterChanged;
                    if (_targetSound is StreamSoundBase)
                    {
                        foreach (StreamSoundTrackBase track in _targetSound.Children)
                        {
                            track.Parameters.ParameterValueChanged -= OnSoundParameterChanged;
                        }
                    }
                }

                _targetSound = _previewPlayer.Item;

                if (null != _targetSound)
                {
                    _targetSound.Parameters.ParameterValueChanged += OnSoundParameterChanged;
                    if (_targetSound is StreamSoundBase)
                    {
                        foreach (StreamSoundTrackBase track in _targetSound.Children)
                        {
                            track.Parameters.ParameterValueChanged += OnSoundParameterChanged;
                        }
                    }
                }

                UpdateName();
                Invalidate(ImageRect);
            }
            else if (_targetSound is StreamSoundBase == true &&
                     this._name.Text != _previewPlayer.Text)
            {
                UpdateName();
                Invalidate(ImageRect);
            }

            _toolStripMenuItemDelete.Enabled = this.Item != null;
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateIcon()
        {
            if (this.CommunicationService.IsConnected != false)
            {
                this.lbl_Mute.Visible = false;
                this.lbl_SoloPlay.Visible = false;
            }
            else
            {
                this.lbl_Mute.Visible = this._previewPlayer.ContainsMuteTrack;
                this.lbl_SoloPlay.Visible = this._previewPlayer.ContainsSoloPlayTrack;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateColors()
        {
            this.UpdateColors(true);
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateColors(bool isActive)
        {
            // 選択色 or 通常色 の設定
            if (this._selected)
            {
                if (this._focused && isActive)
                {
                    this._index.ForeColor = this._headerForeColorSelected;
                    this._itemHeader.BackColor = this._headerBackColorSelected;
                    this._name.ForeColor = this._itemForeColorSelected;
                    this.BackColor = this._itemBackColorSelected;
                }
                else
                {
                    this._index.ForeColor = this._headerForeColorSelectedNoFocus;
                    this._itemHeader.BackColor = this._headerBackColorSelectedNoFocus;
                    this._name.ForeColor = this._itemForeColorSelectedNoFocus;
                    this.BackColor = this._itemBackColorSelectedNoFocus;
                }
            }
            else
            {
                this._index.ForeColor = this._headerForeColor;
                this._itemHeader.BackColor = this._headerBackColor;
                this._name.ForeColor = this._itemForeColor;
                this.BackColor = this._itemBackColor;
            }

            this.Invalidate(true);
        }

        public void SetActive(bool isActive)
        {
            if (this._focused == isActive)
            {
                return;
            }

            this._focused = isActive;
            this.UpdateColors();
        }

        private void InitializeColor()
        {
            _headerForeColor = SystemColors.WindowText;
            _headerBackColor = SystemColors.Control;
            _itemForeColor = SystemColors.WindowText;
            _itemBackColor = SystemColors.Window;

            _headerForeColorSelected = SystemColors.HighlightText;
            _headerBackColorSelected =
                    Color.FromArgb(
                         SystemColors.Highlight.R * 7 / 8,
                         SystemColors.Highlight.G * 7 / 8,
                         SystemColors.Highlight.B * 7 / 8
                         );
            _itemForeColorSelected = SystemColors.HighlightText;
            _itemBackColorSelected =
                    Color.FromArgb(
                         (SystemColors.Highlight.R * 192 / 255) + (Color.White.R * (255 - 192) / 255),
                         (SystemColors.Highlight.G * 192 / 255) + (Color.White.G * (255 - 192) / 255),
                         (SystemColors.Highlight.B * 192 / 255) + (Color.White.B * (255 - 192) / 255)
                         );

            _headerForeColorSelectedNoFocus = SystemColors.HighlightText;
            _headerBackColorSelectedNoFocus = SystemColors.ControlDark;
            _itemForeColorSelectedNoFocus = SystemColors.HighlightText;
            _itemBackColorSelectedNoFocus =
                    Color.FromArgb(
                         (SystemColors.ControlDark.R * 160 / 255) + (Color.White.R * (255 - 160) / 255),
                         (SystemColors.ControlDark.G * 160 / 255) + (Color.White.G * (255 - 160) / 255),
                         (SystemColors.ControlDark.B * 160 / 255) + (Color.White.B * (255 - 160) / 255)
                         );

            this.UpdateColors();
        }

        private void SetItemInternal(Sound sound)
        {
            if (null != sound)
            {
                _previewPlayer.Item = sound;
            }
            else
            {
                _previewPlayer.Reset();
            }
        }

        private void Reset()
        {
            _previewPlayer.Reset();

            _targetSound = null;
            _targetSoundName = string.Empty;
            _name.Text = string.Empty;

            Invalidate(ImageRect);
            return;
        }

        private Sound GetValidSound(string soundName, Sound currentSound)
        {
            if (null == soundName) { throw new ArgumentNullException("soundName"); }

            if (!ProjectService.ComponentDictionary.Contains(soundName)) { return null; }

            Component[] components = ProjectService.ComponentDictionary[soundName];

            if (0 == components.Length) { return null; }

            foreach (Component component in components)
            {
                if (component == currentSound) { return currentSound; }
            }

            foreach (Component component in components)
            {
                if (component is Sound) { return component as Sound; }
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateName()
        {
            if (null == _previewPlayer.Item)
            {
                _targetSoundName = string.Empty;
                _name.Text = string.Empty;
                UpdateIcon();
                return;
            }

            // ラベルが変更されたら、Viewerに送信する
            if (_previewPlayer.Item.Name != _targetSoundName)
            {
                Debug.Assert(null != _previewPlayer);
                this.CommunicationService.SetSoundLabel(_previewPlayer);
            }

            _targetSoundName = _previewPlayer.Item.Name;
            _name.Text = _previewPlayer.Text;

            //
            UpdateIcon();
        }

        /// <summary>
        /// XMLから対象サウンドを取得します
        /// </summary>
        /// <param name="xmlDocument">検索対象のXML</param>
        /// <returns>対象サウンド。</returns>
        private string GetTargetSoundName(XmlDocument xmlDocument)
        {
            if (null == xmlDocument) { return null; }

            // Stream Sound
            XmlElement xmlStreamSound = xmlDocument.DocumentElement.SelectSingleNode("Items/StreamSound[1]") as XmlElement;

            if (null != xmlStreamSound)
            {
                return GetTargetSoundName(xmlStreamSound);
            }

            // Wave Sound
            XmlElement xmlWaveSound = xmlDocument.DocumentElement.SelectSingleNode("Items/WaveSound[1]") as XmlElement;

            if (null != xmlWaveSound)
            {
                return GetTargetSoundName(xmlWaveSound);
            }

            // Sequence Sound
            XmlElement xmlSequenceSound = xmlDocument.DocumentElement.SelectSingleNode("Items/SequenceSound[1]") as XmlElement;

            if (null != xmlSequenceSound)
            {
                return GetTargetSoundName(xmlSequenceSound);
            }

            return null;
        }

        /// <summary>
        /// XMLから対象サウンド名を取得します
        /// </summary>
        /// <param name="xmlDocument">検索対象のXML</param>
        /// <returns>対象サウンド名。</returns>
        private string GetTargetSoundName(XmlElement xmlSound)
        {
            if (null == xmlSound) { return null; }

            XmlElement xmlParameters = xmlSound.SelectSingleNode("Parameters") as XmlElement;
            if (null == xmlParameters) { return null; }

            foreach (XmlElement xmlParameter in xmlParameters.GetElementsByTagName("Parameter"))
            {

                string parameterName = xmlParameter.GetAttribute("Name");
                if (null == parameterName || parameterName != "Name") { continue; }

                string parameterValue = xmlParameter.GetAttribute("Value");
                if (null == parameterValue || 0 == parameterValue.Length) { continue; }

                if (!ProjectService.ComponentDictionary.Contains(parameterValue)) { return null; }

                return parameterValue;

            }

            return null;
        }

        private void UpdateState(bool newState)
        {
            if (_selected == newState) { return; }

            _selected = newState;

            if (null != SelectedChanged)
            {
                SelectedChanged(this, new EventArgs());
            }
        }

        private void UpdateButtons()
        {
            if (_previewPlayer.CanPlay)
            {
                _play.Enabled = true;
                _play.Image = ButtonImages.Images[(int)Images.ButtonID.Play];
            }
            else
            {
                _play.Enabled = false;
                _play.Image = ButtonImages.Images[(int)Images.ButtonID.Play_Disabled];
            }

            if (_previewPlayer.CanPause)
            {
                _pause.Enabled = true;
                _pause.Image = ButtonImages.Images[(int)Images.ButtonID.Pause];
            }
            else
            {
                _pause.Enabled = false;
                _pause.Image = ButtonImages.Images[(int)Images.ButtonID.Pause_Disabled];
            }

            if (_previewPlayer.CanStop)
            {
                _stop.Enabled = true;
                _stop.Image = ButtonImages.Images[(int)Images.ButtonID.Stop];
            }
            else
            {
                _stop.Enabled = false;
                _stop.Image = ButtonImages.Images[(int)Images.ButtonID.Stop_Disabled];
            }

            if (_previewPlayer.CanRepeat)
            {
                _repeat.Enabled = true;

                if (_repeat.Checked)
                {
                    _repeat.Image = ButtonImages.Images[(int)Images.ButtonID.Repeat_Selected];
                }
                else
                {
                    _repeat.Image = ButtonImages.Images[(int)Images.ButtonID.Repeat];
                }
            }
            else
            {
                _repeat.Enabled = false;
                _repeat.Image = ButtonImages.Images[(int)Images.ButtonID.Repeat_Disabled];
            }
        }

        #endregion

        #region ** 画像管理

        private static class Images
        {
            public enum IconID
            {
                None = -1,
                Stream = 0,
                WaveSound,
                WaveSoundCo,
                Seq,
                SeqSet,
                Bank,
                WaveArchive,
                Group,
                Player,
                SoundSet,
                Instrument
            }

            public enum ButtonID
            {
                None = -1,
                Play = 0,
                Play_Disabled,
                Pause,
                Pause_Disabled,
                Stop,
                Stop_Disabled,
                Repeat,
                Repeat_Selected,
                Repeat_Disabled
            }

            public enum StateID
            {
                None = -1,
                Disabled = 0,
                Stopped,
                Preparing,
                Playing,
                Paused
            }

            private static ImageList _iconImages;
            private static ImageList _buttonImages;
            private static ImageList _stateImages;

            static Images()
            {
                // アイコンイメージ
                _iconImages = new ImageList();
                _iconImages.ColorDepth = ColorDepth.Depth24Bit;
                _iconImages.ImageSize = new Size(16, 16);

                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStreamSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSequenceSound, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSequenceSoundSet, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconBank, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconWaveArchive, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconGroup, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPlayer, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconSoundSet, UIServiceBase.TransparentColor);
                _iconImages.Images.Add(FrameworkResources.ImageResource.BitmapIconInstrument, UIServiceBase.TransparentColor);

                // ボタンイメージ
                _buttonImages = new ImageList();
                _buttonImages.ColorDepth = ColorDepth.Depth24Bit;
                _buttonImages.ImageSize = new Size(14, 14);

                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPlaySmall, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPlaySmall_Disabled, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPauseSmall, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconPauseSmall_Disabled, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStopSmall, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStopSmall_Disabled, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconRepeatSmall, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconRepeatSmall_Selected, UIServiceBase.TransparentColor);
                _buttonImages.Images.Add(FrameworkResources.ImageResource.BitmapIconRepeatSmall_Disabled, UIServiceBase.TransparentColor);

                // StateImageList
                _stateImages = new ImageList();
                _stateImages.ColorDepth = ColorDepth.Depth24Bit;
                _stateImages.ImageSize = new Size(16, 16);

                _stateImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStateDisabled, UIServiceBase.TransparentColor);
                _stateImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStateStopped, UIServiceBase.TransparentColor);
                _stateImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStatePreparing, UIServiceBase.TransparentColor);
                _stateImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStatePlaying, UIServiceBase.TransparentColor);
                _stateImages.Images.Add(FrameworkResources.ImageResource.BitmapIconStatePaused, UIServiceBase.TransparentColor);
            }

            #region ** プロパティ

            public static ImageList IconImages
            {
                get { return _iconImages; }
            }

            public static ImageList ButtonImages
            {
                get { return _buttonImages; }
            }

            public static ImageList StateImages
            {
                get { return _stateImages; }
            }

            #endregion
        }

        #endregion
    }
}
