﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMakerPlugin;
    using SoundFoundation.FileFormats.Wave;
    using SoundFoundation.Resources;
    using System;
    using System.Collections.ObjectModel;
    using System.Linq;

    public class PreviewStreamSoundCommon : PreviewSound
    {
        private StreamSoundTrackBase _track = null;
        private Collection<IPreviewStreamChannel> _channels = new Collection<IPreviewStreamChannel>();
        private bool _forceNoLoop = false;

        public PreviewStreamSoundCommon(StreamSoundBase sound, OutputWaveFileRenderType samplingRate, bool forceNoLoop) : base(sound, samplingRate)
        {
            _forceNoLoop = forceNoLoop;
        }

        public override PreviewPlayerState State
        {
            get
            {
                if (0 >= _channels.Count) { return PreviewPlayerState.Stopped; }
                return _channels[0].State;
            }
        }

        protected new StreamSoundBase Sound
        {
            get { return base.Sound as StreamSoundBase; }
        }

        public override void OutputWaveFile(string filePath, int maxDuration, bool monaural)
        {
            try
            {
                PreviewStreamSound.AddPlayParameter(this.Sound);
                base.OutputWaveFile(filePath, maxDuration, monaural);
            }
            finally
            {
                PreviewStreamSound.RemovePlayParameter(this.Sound);
            }
        }

        public override float MeasureIntegratedLoudness(int? maxDuration, System.Threading.CancellationToken? token = null)
        {
            float loudness;

            try
            {
                PreviewStreamSound.AddPlayParameter(this.Sound);
                loudness = base.MeasureIntegratedLoudness(maxDuration, token);
            }
            finally
            {
                PreviewStreamSound.RemovePlayParameter(this.Sound);
            }

            return loudness;
        }

        public override void Prepare()
        {
            Stop();

            _channels.Clear();

            this.ValidateWaveFiles();

            SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.RuntimeSoundSystem_LockSoundThread();

            try
            {
                var baseTrack = Sound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;
                if (_track != null)
                {
                    _channels.Add(
                        SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreatePreviewStreamChannel(
                            _track,
                            baseTrack,
                            0,
                            Sound.Volume,
                            Sound.Pitch,
                            Sound.PanMode,
                            Sound.PanCurve,
                            Sound.Sends.MainSend,
                            this.RendererType,
                            _forceNoLoop));
                }
                else
                {
                    bool enableAac = baseTrack != null &&
                        AACUtil.IsAACFile(baseTrack.FilePath);

                    int channelCount = 0;

                    foreach (StreamSoundTrackBase track in Sound.Children.Where<Component>(item => item.IsEnabled))
                    {
                        if (track.Parameters.ContainsKey(PreviewStreamSound.PlayTrack) != false)
                        {
                            IPreviewStreamChannel channel = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.CreatePreviewStreamChannel
                                (track,
                                  baseTrack,
                                  channelCount,
                                  Sound.Volume,
                                  Sound.Pitch,
                                  Sound.PanMode,
                                  Sound.PanCurve,
                                  Sound.Sends.MainSend,
                                  this.RendererType,
                                  _forceNoLoop);

                            _channels.Add(channel);

                            if (enableAac)
                            {
                                channelCount += track.ChannelCount;
                            }

                            if (track.Parameters.ContainsKey(PreviewStreamSound.MuteTrack) != false)
                            {
                                channel.ChannelSetUserVolume(0.0F);
                            }
                        }
                    }
                }
            }
            finally
            {
                SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.RuntimeSoundSystem_UnlockSoundThread();
            }
        }

        /// <summary>
        /// 再生
        /// </summary>
        public override void Play()
        {
            foreach (IPreviewStreamChannel channel in _channels)
            {
                channel.Start();
            }
        }

        /// <summary>
        /// 一時停止
        /// </summary>
        public override void Pause(bool flag)
        {
            foreach (IPreviewStreamChannel channel in _channels)
            {
                channel.Pause(flag);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void MuteChannel(Component component, bool value)
        {
            if (component is StreamSoundTrackBase)
            {
                StreamSoundTrackBase targetTrack = component as StreamSoundTrackBase;
                foreach (IPreviewStreamChannel channel in _channels)
                {
                    if (channel.StreamSoundTrack == targetTrack)
                    {
                        if (value != false)
                        {
                            channel.ChannelSetUserVolume(0.0F);
                        }
                        else
                        {
                            StreamSoundBase stream = targetTrack.Parent as StreamSoundBase;
                            float volume = (((float)stream.Volume / 127.0F) *
                                            ((float)targetTrack.Volume / 127.0F));
                            channel.ChannelSetUserVolume(volume);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 停止
        /// </summary>
        public override void Stop()
        {
            foreach (IPreviewStreamChannel channel in _channels)
            {
                channel.Stop();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void SetParameter(string parameterNamme)
        {
            foreach (IPreviewStreamChannel channel in _channels)
            {
                float volume = (((float)Sound.Volume / 127.0f) *
                                (channel.StreamSoundTrack.Volume / 127.0f));
                channel.ChannelSetUserVolume(volume);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
            foreach (IPreviewStreamChannel channel in _channels)
            {
                channel.Dispose();
            }
        }

        private void ValidateWaveFiles()
        {
            // ダウンサンプルする場合は、サンプルレートチェックをスキップする
            if (this.Sound.IsResampleEnabled)
            {
                return;
            }

            // すべての波形ファイルが同じサンプルレートかどうか、チェックする（マルチトラックストリームの制約）
            int? sampleRate = null;

            foreach (StreamSoundTrackBase track in this.Sound.Children.Where<Component>(item => item.IsEnabled))
            {
                using (var reader = WaveFileReader.CreateInstance(track.FilePath))
                {
                    WaveFile waveFile = reader.Open(track.FilePath);

                    if (sampleRate.HasValue)
                    {
                        if (sampleRate != waveFile.SampleRate)
                        {
                            throw new ApplicationException(MessageResource.Message_StreamSoundTracksMustHaveTheSameSamplingRate);
                        }
                    }
                    else
                    {
                        sampleRate = waveFile.SampleRate;
                    }
                }
            }
        }
    }
}
