﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundMaker.Profile;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.SoundMakerPlugin;

    public enum OutputWaveFileRenderType
    {
        k32KHz = 0,
        k48KHz = 1,
    }

    public abstract class PreviewSound : IDisposable
    {
        private Sound _sound;
        private readonly RendererType rendererType = RendererType.k32KHz;

        private enum OutputState
        {
            Start,
            RequestStop,
            Stop
        }

        public PreviewSound(Sound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }
            _sound = sound;
        }

        public PreviewSound(Sound sound, OutputWaveFileRenderType samplingRate)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }
            this._sound = sound;

            switch (samplingRate)
            {
                default:
                case OutputWaveFileRenderType.k32KHz:
                    this.rendererType = RendererType.k32KHz;
                    break;

                case OutputWaveFileRenderType.k48KHz:
                    this.rendererType = RendererType.k48KHz;
                    break;
            }
        }

        #region ** プロパティ

        public virtual PreviewPlayerState State
        {
            get { return PreviewPlayerState.Disabled; }
        }

        public string Name
        {
            get { return _sound == null ? string.Empty : _sound.Name; }
        }

        public Sound Sound
        {
            get { return _sound; }
        }

        protected RendererType RendererType
        {
            get
            {
                return this.rendererType;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 波形ファイルに出力します。
        /// </summary>
        /// <param name="filePath">ファイルパス。</param>
        /// <param name="maxDuration">最大出力時間。</param>
        public virtual void OutputWaveFile(string filePath, int maxDuration, bool monaural)
        {
            int channelCount = 2;
            if (monaural == true) { channelCount = 1; }

            Stop();

            this.Plugin.RuntimeSoundSystem_StopAllVoices();
            this.Plugin.RuntimeSoundSystem_SoundThreadPause(true);

            try
            {
                var nSamplesPerSec = this.Plugin.RuntimeGlobal_DefaultSampleRate;

                if (this.Plugin.IsNwRendererEnabled)
                {
                    switch (this.RendererType)
                    {
                        case RendererType.k32KHz:
                            nSamplesPerSec = 32000;
                            break;

                        case RendererType.k48KHz:
                            nSamplesPerSec = 48000;
                            break;

                        default:
                            return;
                    }
                }

                Prepare();
                Play();

                using (BinaryWriter binWriter =
                new BinaryWriter(File.Open(filePath, FileMode.Create)))
                {
                    UInt16 nBlockAlign = (UInt16)(2 * channelCount);
                    UInt32 nAvgBytesPerSec = (uint)nSamplesPerSec * nBlockAlign;

                    binWriter.Write(new char[4] { 'R', 'I', 'F', 'F' });
                    binWriter.Write((UInt32)0); // chunk size
                    binWriter.Write(new char[4] { 'W', 'A', 'V', 'E' });
                    binWriter.Write(new char[4] { 'f', 'm', 't', ' ' });
                    binWriter.Write((UInt32)16); // chunk size
                    binWriter.Write((UInt16)1); // PCM
                    binWriter.Write((UInt16)channelCount); // monaural, stereo
                    binWriter.Write(nSamplesPerSec);
                    binWriter.Write(nAvgBytesPerSec);
                    binWriter.Write(nBlockAlign);
                    binWriter.Write((UInt16)16); // bits
                    binWriter.Write(new char[4] { 'd', 'a', 't', 'a' });
                    binWriter.Write((UInt32)0); // chunk size

                    int maxDurationMsec = maxDuration * 1000;

                    var sampleCountPerFrame = this.Plugin.RuntimeGlobal_CHANNEL_COUNT * this.Plugin.RuntimeGlobal_MSEC_PER_FRAME * nSamplesPerSec / 1000;
                    var maxSampleCount = sampleCountPerFrame;
                    Int16[] waveBuffer = new Int16[maxSampleCount];

                    // 2 フレーム分の無音をトリミングするために空回し
                    {
                        int renderedBytes = 0;
                        while (renderedBytes < sampleCountPerFrame * sizeof(short) * 2)
                        {
                            this.Plugin.RuntimeSoundSystem_SoundFrameProcess();
                            renderedBytes += this.Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);
                        }
                    }

                    int time = 0;
                    int dataChunkSize = 0;
                    var outputState = OutputState.Start;
                    while (outputState != OutputState.Stop)
                    {
                        if (State == PreviewPlayerState.Stopped && this.Plugin.RuntimeSoundSystem_GetActiveVoiceCount() <= 0 && outputState == OutputState.Start)
                        {
                            outputState = OutputState.RequestStop;
                        }

                        time = dataChunkSize / (sizeof(short) * channelCount * nSamplesPerSec / 1000);

                        if (time >= maxDurationMsec)
                        {
                            Stop();
                            this.Plugin.RuntimeSoundSystem_StopAllVoices();
                            // 時間指定時にタイムアウトした場合、フェードなしで打ち切る
                            outputState = OutputState.Stop;
                        }

                        this.Plugin.RuntimeSoundSystem_SoundFrameProcess();
                        var readLength = this.Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);

                        // ゼロ書き込みチェック用
                        bool isNonZeroWritten = false;

                        // 波形書き出し
                        if (this.Plugin.IsNwRendererEnabled)
                        {
                            if (monaural == true) // モノラル
                            {
                                for (int i = 0; i < readLength; i += 2)
                                {
                                    Int16 sampleL = waveBuffer[i];
                                    Int16 sampleR = waveBuffer[i + 1];
                                    Int16 sample = (Int16)(sampleL + sampleR);
                                    if ((0 < sampleL && 0 < sampleR) &&
                                        (sampleL > Int16.MaxValue - sampleR))
                                    {
                                        sample = Int16.MaxValue;
                                    }
                                    else if ((sampleL < 0 && sampleR < 0) &&
                                             (sampleL < Int16.MinValue - sampleR))
                                    {
                                        sample = Int16.MinValue;
                                    }
                                    binWriter.Write(sample);
                                    dataChunkSize += 2;

                                    if (sample != 0)
                                    {
                                        isNonZeroWritten = true;
                                    }
                                }
                            }
                            else // ステレオ
                            {
                                foreach (Int16 sample in waveBuffer.Take(readLength / sizeof(Int16)))
                                {
                                    binWriter.Write(sample);
                                    dataChunkSize += 2;

                                    if (sample != 0)
                                    {
                                        isNonZeroWritten = true;
                                    }

                                }
                            }
                        }
                        else
                        {
                            var sampleCountPerChannel = readLength / sizeof(short) / this.Plugin.RuntimeGlobal_CHANNEL_COUNT;
                            var frameCount = readLength / sampleCountPerFrame / sizeof(short);

                            foreach (var frameIndex in Enumerable.Range(0, frameCount))
                            {
                                foreach (var sampleIndex in Enumerable.Range(0, sampleCountPerChannel))
                                {
                                    // LR サンプルのみ出力して、それ以降の 4ch 分は読み捨てる
                                    Int16 sampleL = waveBuffer[frameIndex * sampleCountPerFrame + sampleIndex];
                                    Int16 sampleR = waveBuffer[frameIndex * sampleCountPerFrame + sampleCountPerChannel + sampleIndex];

                                    if (monaural == true) // モノラル
                                    {
                                        Int16 sample = (Int16)(sampleL + sampleR);
                                        if ((0 < sampleL && 0 < sampleR) &&
                                            (sampleL > Int16.MaxValue - sampleR))
                                        {
                                            sample = Int16.MaxValue;
                                        }
                                        else if ((sampleL < 0 && sampleR < 0) &&
                                                 (sampleL < Int16.MinValue - sampleR))
                                        {
                                            sample = Int16.MinValue;
                                        }
                                        binWriter.Write(sample);
                                        dataChunkSize += sizeof(Int16);

                                        if(sample != 0)
                                        {
                                            isNonZeroWritten = true;
                                        }
                                    }
                                    else
                                    {
                                        binWriter.Write(sampleL);
                                        binWriter.Write(sampleR);
                                        dataChunkSize += sizeof(Int16) * 2;

                                        if (sampleL != 0 || sampleR != 0)
                                        {
                                            isNonZeroWritten = true;
                                        }
                                    }
                                }
                            }
                        }

                        // Stop 時のゼロ書き込みチェック
                        if (!isNonZeroWritten && outputState == OutputState.RequestStop)
                        {
                            outputState = OutputState.Stop;
                        }
                    }

                    binWriter.Seek(4, SeekOrigin.Begin);
                    binWriter.Write((UInt32)(0x24 + dataChunkSize));
                    binWriter.Seek(0x28, SeekOrigin.Begin);
                    binWriter.Write((UInt32)dataChunkSize);
                }

            }
            finally
            {
                // 念のための停止処理
                Stop();
                this.Plugin.RuntimeSoundSystem_StopAllVoices();

                this.Plugin.RuntimeSoundSystem_SoundThreadPause(false);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void Load()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void MuteChannel(Component component, bool value)
        {
        }

        /// <summary>
        /// パラメータの設定
        /// </summary>
        public virtual void SetParameter(string parameterName)
        {
        }

        /// <summary>
        /// 平均ラウドネス値の計測
        /// </summary>
        /// <param name="maxDuration">最大出力時間</param>
        public virtual float MeasureIntegratedLoudness(int? maxDuration, System.Threading.CancellationToken? token = null)
        {
            float loudness;
            int channelCount = 2;

            var nSamplesPerSec = this.Plugin.RuntimeGlobal_DefaultSampleRate;
            if (this.Plugin.IsNwRendererEnabled == true)
            {
                switch (this.RendererType)
                {
                    case RendererType.k32KHz:
                        nSamplesPerSec = 32000;
                        break;

                    case RendererType.k48KHz:
                        nSamplesPerSec = 48000;
                        break;

                    default:
                        Debug.Assert(false, "Illegal Sample Rate");
                        break;
                }
            }

            var parameterSampleRate = LoudnessCalculator.Parameter48kHz;
            switch (nSamplesPerSec)
            {
                case 32000:
                    parameterSampleRate = LoudnessCalculator.Parameter32kHz;
                    break;

                case 48000:
                    parameterSampleRate = LoudnessCalculator.Parameter48kHz;
                    break;

                default:
                    Debug.Assert(false, "Illegal Sample Rate");
                    break;
            }

            Stop();

            this.Plugin.RuntimeSoundSystem_StopAllVoices();
            this.Plugin.RuntimeSoundSystem_SoundThreadPause(true);

            var loudnessCalculator = new LoudnessCalculator();
            loudnessCalculator.SetParameter(parameterSampleRate);

            try
            {
                Prepare();
                Play();

                var oneChannelSampleCountPerFrame = this.Plugin.RuntimeGlobal_MSEC_PER_FRAME * nSamplesPerSec / 1000;
                var sampleCountPerFrame = this.Plugin.RuntimeGlobal_CHANNEL_COUNT * oneChannelSampleCountPerFrame;

                Int16[] waveBuffer = new Int16[sampleCountPerFrame];
                short[][] samples = new short[LoudnessCalculator.ChannelCount][]
                    { new short[oneChannelSampleCountPerFrame], // FrontLeft   ステレオのみ使用
                      new short[oneChannelSampleCountPerFrame], // FrontRight
                      null,                                     // RearLeft    これ以下は使用しない
                      null,                                     // RearRight
                      null,                                     // FrontCenter
                      null,                                     // LFE
                    };

                // 2 フレーム分の無音をトリミングするために空回し
                {
                    int renderedBytes = 0;
                    while (renderedBytes < sampleCountPerFrame * sizeof(short) * 2)
                    {
                        this.Plugin.RuntimeSoundSystem_SoundFrameProcess();
                        renderedBytes += this.Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);
                    }
                }

                int time = 0;
                int dataSize = 0;
                while (State != PreviewPlayerState.Stopped || this.Plugin.RuntimeSoundSystem_GetActiveVoiceCount() > 0)
                {
                    if (token.HasValue == true)
                    {
                        // キャンセルされたら例外を発生させます。
                        token.Value.ThrowIfCancellationRequested();
                    }

                    if (maxDuration.HasValue == true) // シーケンスサウンド用
                    {
                        time = dataSize / (sizeof(short) * channelCount * nSamplesPerSec / 1000);

                        if (time >= maxDuration.Value * 1000)
                        {
                            Stop();
                            this.Plugin.RuntimeSoundSystem_StopAllVoices();
                        }
                    }

                    this.Plugin.RuntimeSoundSystem_SoundFrameProcess();
                    var readLength = this.Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);
                    var sampleCount = 0;

                    if (this.Plugin.IsNwRendererEnabled)
                    {
                        for (int i = 0; i < readLength / sizeof(Int16); i += 2)
                        {
                            samples[0][sampleCount] = waveBuffer[i];     // sampleL
                            samples[1][sampleCount] = waveBuffer[i + 1]; // sampleR
                            sampleCount++;
                            dataSize += 2;
                        }
                    }
                    else
                    {
                        var sampleCountPerChannel = readLength / sizeof(short) / this.Plugin.RuntimeGlobal_CHANNEL_COUNT;
                        var frameCount = readLength / sampleCountPerFrame / sizeof(short);

                        foreach (var frameIndex in Enumerable.Range(0, frameCount))
                        {
                            foreach (var sampleIndex in Enumerable.Range(0, sampleCountPerChannel))
                            {
                                // LR サンプルのみ出力して、それ以降の 4ch 分は読み捨てる
                                Int16 sampleL = waveBuffer[frameIndex * sampleCountPerFrame + sampleIndex];
                                Int16 sampleR = waveBuffer[frameIndex * sampleCountPerFrame + sampleCountPerChannel + sampleIndex];
                                samples[0][sampleCount] = sampleL;
                                samples[1][sampleCount] = sampleR;
                                sampleCount++;
                                dataSize += sizeof(Int16) * 2;
                            }
                        }
                    }

                    loudnessCalculator.Accumulate(samples, 0, sampleCount);
                }
            }
            finally
            {
                // 念のための停止処理
                Stop();
                this.Plugin.RuntimeSoundSystem_StopAllVoices();
                this.Plugin.RuntimeSoundSystem_SoundThreadPause(false);

                loudnessCalculator.Finish();
                loudness = loudnessCalculator.GatedLoudnessValue;
            }

            return loudness;
        }

        #endregion

        #region ** 抽象メソッド

        /// <summary>
        ///
        /// </summary>
        public abstract void Dispose();

        /// <summary>
        ///
        /// </summary>
        public abstract void Prepare();

        /// <summary>
        /// 再生
        /// </summary>
        public abstract void Play();

        /// <summary>
        /// 一時停止
        /// </summary>
        public abstract void Pause(bool flag);

        /// <summary>
        /// 停止
        /// </summary>
        public abstract void Stop();

        #endregion

        private ISoundMakerPlugin Plugin
        {
            get
            {
                return SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;
            }
        }
    }
}
