﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.FileFormats;
    using NintendoWare.SoundMaker.Profile;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.ToolDevelopmentKit;
    using SoundFoundation.Resources;
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public class PreviewSequenceSound : PreviewSound
    {
        #region ** 定数

        private const int BankCount = 4;
        private const int TrackCount = 16;
        private const int VariableCount = 16;

        #endregion

        private SoundProjectService _projectService;

        private SeqVariable[] _localVariables = new SeqVariable[VariableCount];
        private SeqVariable[,] _trackVariables = new SeqVariable[TrackCount, VariableCount];

        private ISequenceSoundPlayer _seqPlayer = Plugin.CreateSequenceSoundPlayer();

        public PreviewSequenceSound(SequenceSoundBase sound, SoundProjectService projectService, OutputWaveFileRenderType samplingRate)
            : base(sound, samplingRate)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            _projectService = projectService;

            for (int varNo = 0; varNo < VariableCount; varNo++)
            {
                _localVariables[varNo] = new SeqVariable((uint)varNo);
            }

            for (int trackNo = 0; trackNo < TrackCount; trackNo++)
            {

                for (int varNo = 0; varNo < VariableCount; varNo++)
                {
                    _trackVariables[trackNo, varNo] = new SeqVariable((uint)varNo);
                }

            }
        }

        public override PreviewPlayerState State
        {
            get
            {
                Debug.Assert(null != _seqPlayer, "unexpected error");

                if (!_seqPlayer.IsActive() || _seqPlayer.IsPlayFinished()) { return PreviewPlayerState.Stopped; }
                if (_seqPlayer.IsPause()) { return PreviewPlayerState.Paused; }

                return PreviewPlayerState.Playing;
            }
        }

        protected new SequenceSoundBase Sound
        {
            get { return base.Sound as SequenceSoundBase; }
        }

        private SeqFileManager.FileType FileType
        {
            get
            {
                switch (Sound.FileType)
                {
                    case SequenceSoundFileType.Text:
                        return SeqFileManager.FileType.Text;

                    case SequenceSoundFileType.Smf:
                        return SeqFileManager.FileType.Smf;
                }

                Debug.Fail("invalid file type");
                return SeqFileManager.FileType.Text;
            }
        }

        /// <summary>
        /// １つのトラックのみを再生したい場合にトラック番号（１以上）を指定します。
        /// </summary>
        public int? SingleTrack { get; set; }

        public static short GetGlobalVariable(int varNo)
        {
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }
            return Plugin.RuntimeSequenceSoundPlayer_GetGlobalVariable(varNo);
        }

        public static void SetGlobalVariable(int varNo, short var)
        {
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }
            Plugin.RuntimeSequenceSoundPlayer_SetGlobalVariable(varNo, var);
        }

        public override void Load()
        {
            this.ValidateSoundSetBanks();

            if (!File.Exists(Sound.FilePath))
            {
                throw new FileNotFoundException("file not found.", Sound.FilePath);
            }

            this.GetSeqFileReader();

            // NoteOn コールバックで即座にインストルメント波形を返せるように、事前に波形をコンバートしておく
            // そうしないと、発音が遅延する可能性がある
            foreach (var soundSetBank in this.Sound.SoundSetBanks)
            {
                if (soundSetBank == null || !File.Exists(soundSetBank.FilePath))
                {
                    continue;
                }

                var bankFileInfo = BankFileManager.Instance.LoadFile(soundSetBank.FilePath);

                foreach (var instrument in bankFileInfo.XmlBank.Body.Bank.Items
                    .Cast<XmlInstrument>()
                    .Where(instrument => instrument.IsInstrumentEnabled()))
                {
                    foreach (var velocityRegion in instrument.Items
                        .Cast<XmlKeyRegion>()
                        .SelectMany(keyRegion => keyRegion.Items.Cast<XmlVelocityRegion>()))
                    {
                        bool isResampleEnabled;
                        instrument.FindParameterValue(XmlParameterNames.IsResampleEnabled, out isResampleEnabled);
                        bool isDownMixEnabled;
                        instrument.FindParameterValue(XmlParameterNames.IsDownMixEnabled, out isDownMixEnabled);

                        int? sampleRate = null;
                        if (isResampleEnabled)
                        {
                            instrument.FindParameterValue(XmlParameterNames.SampleRate, out sampleRate);
                        }

                        int? channelCount = null;
                        if (isDownMixEnabled == true)
                        {
                            channelCount = 1; // モノラル化
                        }

                        string filePath;
                        if (!velocityRegion.FindParameterValue(XmlParameterNames.FilePath, out filePath) || string.IsNullOrEmpty(filePath))
                        {
                            continue;
                        }

                        try
                        {
                            WaveFileManager.Instance.LoadFile(
                                Path.Combine(Path.GetDirectoryName(soundSetBank.FilePath), filePath),
                                null,
                                null,
                                null,
                                sampleRate,
                                channelCount);
                        }
                        catch (ApplicationException exception)
                        {
                            // メッセージに、バンク、インスト名を付加する
                            throw new ApplicationException(string.Format(
                                MessageResourceCommon.Message_InstrumentWaveFileConversionError,
                                soundSetBank.Name,
                                instrument.Name,
                                exception.Message));
                        }
                    }
                }
            }
        }

        public override void Prepare()
        {
            Prepare(OnNoteOn, null);
        }

        public uint GetTrackMask()
        {
            if (!File.Exists(Sound.FilePath))
            {
                throw new FileNotFoundException("file not found.", Sound.FilePath);
            }

            using (ISeqFileReader reader = this.GetSeqFileReader())
            {
                uint startPosition = 0;

                if (null != Sound.StartPosition && Sound.StartPosition.Length > 0)
                {
                    if (!reader.ReadOffsetByLabel(Sound.StartPosition, out startPosition))
                    {
                        throw new ApplicationException(
                            string.Format(FrameworkResources.MessageResource.Message_SequenceStartLabelNotFound,
                                           Sound.StartPosition));
                    }
                }

                return reader.GetAllocateTrackFlags(startPosition);
            }
        }

        public void Prepare(RuntimeNoteOnCallback callback, object userData)
        {
            Stop();

            this.ValidateSoundSetBanks();

            if (!File.Exists(Sound.FilePath))
            {
                throw new FileNotFoundException("file not found.", Sound.FilePath);
            }

            using (ISeqFileReader reader = this.GetSeqFileReader())
            {
                uint startPosition = 0;

                if (null != Sound.StartPosition && Sound.StartPosition.Length > 0)
                {
                    if (!reader.ReadOffsetByLabel(Sound.StartPosition, out startPosition))
                    {
                        throw new ApplicationException(
                            string.Format(FrameworkResources.MessageResource.Message_SequenceStartLabelNotFound,
                                           Sound.StartPosition));
                    }
                }

                IntPtr fileAddress = reader.GetFileAddress();
                uint allocateTrackFlags = reader.GetAllocateTrackFlags(startPosition);

                _seqPlayer.Initialize();
                _seqPlayer.Setup(allocateTrackFlags, callback, userData);

                // AllocTrack コマンドがある場合は、コマンド分だけオフセットを進めておく
                if (allocateTrackFlags != 1)
                {
                    startPosition += 3;
                }

                if (this.SingleTrack.HasValue)
                {
                    // 使用するトラック以外を強制ミュートします。
                    _seqPlayer.ForceTrackMute(~(1u << (this.SingleTrack.Value - 1)));
                }

                // C# では NoteOn 時にファイルロードするので、このタイミングではロードしません。
                IntPtr[] banks = new IntPtr[BankCount];
                IntPtr[] warcs = new IntPtr[BankCount];
                bool[] warcIsIndividuals = new bool[BankCount];

                _seqPlayer.SetSeqData(fileAddress, banks, warcs, warcIsIndividuals, (int)startPosition);
                _seqPlayer.SetVolume(Sound.Volume / 127.0f);
                _seqPlayer.SetChannelPriority(Sound.ChannelPriority);

                ApplyVariables();
            }
        }

        /// <summary>
        /// 再生
        /// </summary>
        public override void Play()
        {
            _seqPlayer.Start();
        }

        /// <summary>
        /// 一時停止
        /// </summary>
        public override void Pause(bool flag)
        {
            _seqPlayer.Pause(flag);
        }

        /// <summary>
        /// 停止
        /// </summary>
        public override void Stop()
        {
            _seqPlayer.Stop();
        }

        /// <summary>
        ///
        /// </summary>
        public override void SetParameter(string parameterNamme)
        {
            float volume = (float)Sound.Volume / 127.0f;
            this._seqPlayer.SetVolume(volume);
        }

        /// <summary>
        ///
        /// </summary>
        public override void Dispose()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void MeasureCount(int maxDuration, ProjectProfileService.MeasureWaveReferenceCount.Data data, RuntimeNoteOnCallback noteOnCallback, Action<SequenceSoundBase, int> setMaximumVoiceCountCallback)
        {
#if !BUILD_WAVE_EXPORTER
            FormsApplicationCommon.Instance.LoudnessService.Stop();
#endif
            Stop();

            Plugin.RuntimeSoundSystem_StopAllVoices();
            Plugin.RuntimeSoundSystem_SoundThreadPause(true);

            try
            {
                if (setMaximumVoiceCountCallback != null)
                {
                    Prepare(noteOnCallback, null);
                }
                else
                {
                    Prepare(noteOnCallback, data);
                }
                Play();

                //
                UInt32 nSamplesPerSec = 32000;
                int maxDurationMsec = maxDuration * 1000;

                Int16[] waveBuffer =
                    new Int16[Plugin.RuntimeGlobal_CHANNEL_COUNT * Plugin.RuntimeGlobal_MSEC_PER_FRAME * nSamplesPerSec / 1000];

                // 2 フレーム分の無音をトリミングするために空回し
                for (int count = 0; count < 2; ++count)
                {
                    Plugin.RuntimeSoundSystem_SoundFrameProcess();
                    Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);
                }

                int time = 0;
                int voiceCount = 0;
                int maximumVoiceCount = 0;
                while (true)
                {
                    voiceCount = Plugin.RuntimeSoundSystem_GetActiveVoiceCount();
                    if (State == PreviewPlayerState.Stopped && voiceCount <= 0)
                    {
                        break;
                    }
                    if (maximumVoiceCount < voiceCount)
                    {
                        maximumVoiceCount = voiceCount;
                    }
                    time += Plugin.RuntimeGlobal_MSEC_PER_FRAME;
                    if (time >= maxDurationMsec)
                    {
                        Stop();
                        Plugin.RuntimeSoundSystem_StopAllVoices();
                    }

                    Plugin.RuntimeSoundSystem_SoundFrameProcess();
                    Plugin.RuntimeGlobal_AXSynthesize(waveBuffer, (Int32)nSamplesPerSec);
                }

                if (setMaximumVoiceCountCallback != null)
                {
                    setMaximumVoiceCountCallback(this.Sound, maximumVoiceCount);
                }
            }
            finally
            {
                // 念のための停止処理
                Stop();
                Plugin.RuntimeSoundSystem_StopAllVoices();

                Plugin.RuntimeSoundSystem_SoundThreadPause(false);
            }
        }

        public void MeasureWaveReferenceCount(int maxDuration, ProjectProfileService.MeasureWaveReferenceCount.Data data)
        {
            this.MeasureCount(maxDuration, data, this.OnNoteOnForWaveReferenceCount, null);
        }

        public void MeasureSequenceVoiceCount(int maxDuration)
        {
            this.MeasureCount(maxDuration, null, this.OnNoteOnForMaximumVoiceCount, this.SetMaximumVoiceCount);
        }

        /// <summary>
        ///
        /// </summary>
        public short GetLocalVariable(int varNo)
        {
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }
            return _seqPlayer.GetLocalVariable(varNo);
        }

        public void SetLocalVariable(int varNo, short var)
        {
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }

            _localVariables[varNo].Checked = true;
            _localVariables[varNo].Value = var;

            _seqPlayer.SetLocalVariable(varNo, var);
        }

        public short GetTrackVariable(int trackNo, int varNo)
        {
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }
            return _seqPlayer.GetTrackVariable(trackNo, varNo);
        }

        public void SetTrackVariable(int trackNo, int varNo, short var)
        {
            if (trackNo < 0 || TrackCount <= varNo) { throw new ArgumentOutOfRangeException("trackNo"); }
            if (varNo < 0 || VariableCount <= varNo) { throw new ArgumentOutOfRangeException("varNo"); }

            _trackVariables[trackNo, varNo].Checked = true;
            _trackVariables[trackNo, varNo].Value = var;

            _seqPlayer.SetTrackVariable(trackNo, varNo, var);
        }

        private ISeqFileReader GetSeqFileReader()
        {
            try
            {
                SeqFileManager.SeqFileInfo seqInfo = SeqFileManager.Instance.LoadFile(Sound.FilePath, FileType);
                if (null == seqInfo) { throw new Exception(); }

                return Plugin.CreateSeqFileReader(seqInfo.GetSeqDataPtr());
            }
            catch
            {
                throw new ApplicationException(FrameworkResources.MessageResource.Message_FailedToPlaySound);
            }
        }

        private void ValidateSoundSetBanks()
        {
            foreach (ComponentReference soundSetBankRef in this.Sound.SoundSetBankReferences)
            {
                if (soundSetBankRef.TargetName == String.Empty)
                {
                    continue;
                }

                if (!this._projectService.ComponentDictionary.Contains(soundSetBankRef.TargetName))
                {
                    throw new ApplicationException(
                        string.Format(FrameworkResources.MessageResource.Message_TargetBankNotFound,
                                       soundSetBankRef.TargetName));
                }

                Component[] items =
                    this._projectService.ComponentDictionary[soundSetBankRef.TargetName];
                if (1 != items.Length || !(items[0] is SoundSetBankBase))
                {
                    throw new ApplicationException(
                        string.Format(FrameworkResources.MessageResource.Message_TargetBankNotFound,
                                       soundSetBankRef.TargetName));
                }

                SoundSetBankBase soundSetBank = items[0] as SoundSetBankBase;

                if (!File.Exists(soundSetBank.FilePath))
                {
                    throw new FileNotFoundException("file not found.", soundSetBank.FilePath);
                }
            }
        }

        private SoundSetBankBase GetSoundSetBank(int index)
        {
            Assertion.Argument.True(index >= 0);
            return this.Sound.SoundSetBanks[index];
        }

        private void ApplyVariables()
        {
            Debug.Assert(null != _seqPlayer, "unexpected error");

            for (int varNo = 0; varNo < VariableCount; varNo++)
            {
                if (!_localVariables[varNo].Checked) { continue; }
                _seqPlayer.SetLocalVariable(varNo, (short)_localVariables[varNo].Value);
            }

            for (int trackNo = 0; trackNo < TrackCount; trackNo++)
            {

                for (int varNo = 0; varNo < VariableCount; varNo++)
                {
                    if (!_trackVariables[trackNo, varNo].Checked) { continue; }
                    _seqPlayer.SetTrackVariable(trackNo, varNo, (short)_trackVariables[trackNo, varNo].Value);
                }

            }
        }

        ///
        private object OnNoteOn(IntPtr seqPlayer, int bankNo, ref IRuntimeNoteOnInfo noteOnInfo, Object userData)
        {
            try
            {
                SoundSetBankBase bank = this.GetSoundSetBank(bankNo);
                if (null == bank) { return null; }

                IPreviewSequenceChannel newChannel = Plugin.CreatePreviewSequenceChannel(bank.FilePath, ref noteOnInfo, this.RendererType);
                if (null == newChannel) { throw new ApplicationException("invalid bank file path."); }

                newChannel.Start();

                return newChannel.Channel;
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private object OnNoteOnForWaveReferenceCount(IntPtr seqPlayer, int bankNo, ref IRuntimeNoteOnInfo noteOnInfo, Object userData)
        {
            ProjectProfileService.MeasureWaveReferenceCount.Data data =
                userData as ProjectProfileService.MeasureWaveReferenceCount.Data;
            if (data != null)
            {
                data.Count(bankNo, noteOnInfo.PrgNo, noteOnInfo.Key, noteOnInfo.Velocity);
            }

            return null;
        }

        private void SetMaximumVoiceCount(SequenceSoundBase sequenceSound, int maximumVoiceCount)
        {
            sequenceSound.MaximumVoiceCount = (uint)maximumVoiceCount;
        }

        private object OnNoteOnForMaximumVoiceCount(IntPtr seqPlayer, int bankNo, ref IRuntimeNoteOnInfo noteOnInfo, Object userData)
        {
            try
            {
                SoundSetBankBase bank = this.GetSoundSetBank(bankNo);
                if (null == bank) { return null; }

                IPreviewSequenceChannel newChannel = Plugin.CreatePreviewSequenceChannel(bank.FilePath, ref noteOnInfo, this.RendererType);
                if (null == newChannel) { throw new ApplicationException("invalid bank file path."); }

                newChannel.SetInstrumentVolume(0.0f);
                newChannel.Start();

                return newChannel.Channel;
            }
            catch
            {
                return null;
            }
        }

        private static ISoundMakerPlugin Plugin
        {
            get
            {
                return SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;
            }
        }
    }
}
