﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Text;
    using System.Threading;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Preview.MCS.Tool.Sound;
    using NintendoWare.SoundMaker.Windows.Forms;
    using Viewer = NintendoWare.SoundMaker.Framework.Preview.Communications.Viewer;
    using MCSViewer = NintendoWare.SoundMaker.Preview.MCS.Viewer;
    using PreviewCtrl = NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;

    #region ** PreviewPlayerWatcher

    public abstract class PlayerStateWatcher : PreviewWatcher
    {
        #region ** 定数

        // タイマー
        private const int DefaultSyncPeriod = 100;

        #endregion

        // パラメータ
        protected PreviewPlayerInformationCollectionCommon _informations = new PreviewPlayerInformationCollectionCommon();
        protected PreviewPlayerManager _parent = null;

        public PlayerStateWatcher(PreviewPlayerManager parent)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }
            _parent = parent;
        }

        #region ** メソッド

        public void Attach(PreviewPlayerInformation item)
        {
            if (_informations.Contains(item)) { return; }

            _informations.Add(item);
        }

        public void Detach(PreviewPlayerInformation item)
        {
            if (!_informations.Contains(item)) { return; }

            _informations.Remove(item);
        }


        #endregion

        #region ** BaseWatcher プロパティの実装

        protected override int Period
        {
            get { return DefaultSyncPeriod; }
        }

        #endregion
    }

    public class PlayerStateWatcherForPC : PlayerStateWatcher
    {
        public PlayerStateWatcherForPC(PreviewPlayerManager parent) : base(parent) { }

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return !_parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
            // プレビュープレイヤー情報を取得する
            foreach (PreviewPlayerInformation information in _informations)
            {

                if (null == information.Sound) { continue; }
                information.State = information.Sound.State;

            }
        }

        #endregion
    }

    public class PlayerStateWatcherForViewer : PlayerStateWatcher
    {
        public PlayerStateWatcherForViewer(PreviewPlayerManager parent)
            : base(parent)
        {
            PreviewCtrl.UpdateSoundHandleInfoPacket.UpdateSoundHandleCallbackDelegate = this.UpdateSoundHandleCallback;
        }

        #region ** メソッド

        public override void Dispose()
        {
            base.Dispose();

            if (PreviewCtrl.UpdateSoundHandleInfoPacket.UpdateSoundHandleCallbackDelegate != this.UpdateSoundHandleCallback) { return; }
            PreviewCtrl.UpdateSoundHandleInfoPacket.UpdateSoundHandleCallbackDelegate = null;
        }

        #endregion

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return _parent.IsSynchronizedViewer; }
        }

        #endregion

        protected override void OnTimer(object state)
        {
        }

        private void UpdateSoundHandleCallback(uint index, PreviewCtrl.SoundHandleState state)
        {
            if (_informations == null || !_informations.Contains(index)) { return; }

            _informations[index].State = ToPreviewPlayerState(state);
        }

        private PreviewPlayerState ToPreviewPlayerState(PreviewCtrl.SoundHandleState state)
        {
            switch (state)
            {
                case PreviewCtrl.SoundHandleState.Stopped:
                    return PreviewPlayerState.Stopped;

                case PreviewCtrl.SoundHandleState.Playing:
                    return PreviewPlayerState.Playing;

                case PreviewCtrl.SoundHandleState.Paused:
                    return PreviewPlayerState.Paused;

                case PreviewCtrl.SoundHandleState.Disabled:
                    return PreviewPlayerState.Disabled;
            }

            return PreviewPlayerState.Disabled;
        }
    }

    #endregion

    #region ** PreviewPlayerParameterWatcher

    public abstract class PreviewPlayerParameterWatcher : PreviewWatcher
    {
        #region ** 定数

        // タイマー
        private const int DefaultSyncPeriod = 100;

        #endregion

        // パラメータ
        protected PreviewPlayerManager _parent = null;

        public PreviewPlayerParameterWatcher(PreviewPlayerManager parent)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }
            _parent = parent;
        }

        #region ** BaseWatcher プロパティの実装

        protected override int Period
        {
            get { return DefaultSyncPeriod; }
        }

        #endregion
    }

    public class PreviewPlayerParameterWatcherForPC : PreviewPlayerParameterWatcher
    {
        public PreviewPlayerParameterWatcherForPC(PreviewPlayerManager parent) : base(parent) { }

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return !_parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
        }

        #endregion
    }

    public class PreviewPlayerParameterWatcherForViewer : PreviewPlayerParameterWatcher
    {
        private CommunicationService communicationService;

        public PreviewPlayerParameterWatcherForViewer(PreviewPlayerManager parent, CommunicationService communicationService)
            : base(parent)
        {
            this.communicationService = communicationService;
        }

        #region ** メソッド

        public override void Dispose()
        {
            base.Dispose();
        }

        #endregion

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return _parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
            if (MCSViewer.SetPreviewPlayerParameterPacket.Received == false) { return; }

            this.communicationService.SetPreviewPlayerParameter();
        }

        #endregion
    }

    #endregion

    #region ** SeqVariableWatcher

    public abstract class SeqVariableWatcher : PreviewWatcher
    {
        protected PreviewPlayerManager _parent = null;

        public SeqVariableWatcher(PreviewPlayerManager parent)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }
            _parent = parent;
        }

        #region ** BaseWatcher プロパティの実装

        protected override int Period
        {
            get { return (int)_parent.Settings.VariablesSyncType; }
        }

        #endregion
    }

    public class SeqVariableWatcherForPC : SeqVariableWatcher
    {
        private CommunicationService communicationService;

        public SeqVariableWatcherForPC(PreviewPlayerManager parent, CommunicationService communicationService)
            : base(parent)
        {
            this.communicationService = communicationService;
        }

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return !_parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
            Debug.Assert(null != _parent.Settings, "unexpected error");
            if (null == _parent.Settings.SyncTargetVariables) { return; }

            // 設定されたコンテナの変数値を更新する
            switch (_parent.Settings.SyncTargetVariables.ID.Type)
            {
                case SeqVariableType.Global:
                    _parent.UpdateGlobalVariables();
                    break;

                case SeqVariableType.Local:
                    {
                        PreviewPlayer targetPlayer = _parent.PreviewPlayers[(int)_parent.Settings.SyncTargetVariables.ID.PreviewPlayerIndex];
                        Debug.Assert(null != targetPlayer, "unexpected error");

                        targetPlayer.UpdateLocalVariables();
                    }
                    break;

                case SeqVariableType.Track:
                    {
                        PreviewPlayer targetPlayer = _parent.PreviewPlayers[(int)_parent.Settings.SyncTargetVariables.ID.PreviewPlayerIndex];
                        Debug.Assert(null != targetPlayer, "unexpected error");

                        targetPlayer.UpdateTrackVariables(_parent.Settings.SyncTargetVariables.ID.TrackNo);
                    }
                    break;
            }
        }

        #endregion
    }

    public class SeqVariableWatcherForViewer : SeqVariableWatcher
    {
        private SeqVariableContainerCollection _target = new SeqVariableContainerCollection();
        private CommunicationService communicationService;

        public SeqVariableWatcherForViewer(PreviewPlayerManager parent, CommunicationService communicationService)
            : base(parent)
        {
            MCSViewer.GetSeqVariablesPacket.Target = _target;
            this.communicationService = communicationService;
        }

        #region ** メソッド

        public override void Dispose()
        {
            base.Dispose();

            if (MCSViewer.GetSeqVariablesPacket.Target != _target) { return; }
            MCSViewer.GetSeqVariablesPacket.Target = null;
        }

        #endregion

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return _parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
            Debug.Assert(null != _parent.Settings, "unexpected error");

            // 前回の結果を取得できていない場合は処理しない
            // この１行は仮実装
            // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
            if (!MCSViewer.GetSeqVariablesPacket.Received) { return; }

            _target.Clear();
            if (null == _parent.Settings.SyncTargetVariables) { return; }

            // 設定されたコンテナの変数値を更新する
            _target.Add(_parent.Settings.SyncTargetVariables);
            this.communicationService.GetSeqVariables(_parent.Settings.SyncTargetVariables);
        }

        #endregion
    }

    #endregion

    #region ** EffectWatcher

    public abstract class EffectWatcher : PreviewWatcher
    {
        #region ** 定数

        // タイマー
        private const int DefaultSyncPeriod = 100;

        #endregion

        // パラメータ
        protected PreviewPlayerManager _parent = null;

        public EffectWatcher(PreviewPlayerManager parent)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }
            _parent = parent;
        }

        #region ** BaseWatcher プロパティの実装

        protected override int Period
        {
            get { return DefaultSyncPeriod; }
        }

        #endregion
    }

    public class EffectWatcherForPC : EffectWatcher
    {
        public EffectWatcherForPC(PreviewPlayerManager parent) : base(parent) { }

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return !_parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
        }

        #endregion
    }

    public class EffectWatcherForViewer : EffectWatcher
    {
        private CommunicationService communicationService;

        public EffectWatcherForViewer(PreviewPlayerManager parent, CommunicationService communicationService)
            : base(parent)
        {
            this.communicationService = communicationService;
        }

        #region ** メソッド

        public override void Dispose()
        {
            base.Dispose();
        }

        #endregion

        #region ** BaseWatcher プロパティの実装

        protected override bool IsSynchronized
        {
            get { return _parent.IsSynchronizedViewer; }
        }

        #endregion

        #region ** BaseWatcher イベントハンドラの実装

        protected override void OnTimer(object state)
        {
            if (MCSViewer.SetEffectParametersPacket.Received == false) { return; }

            this.communicationService.SetEffectParameters();
        }

        #endregion
    }

    #endregion
}
