﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Diagnostics;

namespace NintendoWare.SoundMaker.Preview
{
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Windows.Forms;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public class PreviewPlayerManager
    {
        #region ** 定数

        public const uint MaxPlayerCount = 32;
        public const uint DefaultPlayerCount = 4;

        // シーケンス変数
        public const uint SeqVariableCount = 16;
        public const uint TrackCount = 16;

        #endregion

        // パラメータ
        private PreviewPlayerCollection _previewPlayers = new PreviewPlayerCollection();          // プレビュープレイヤーコレクション
        private GlobalVariableContainerSet _globalVariableSet = new GlobalVariableContainerSet(); // グローバルシーケンス変数

        // 状態
        private bool _synchronizedViewer = false;               // Viewerとの接続状態
        private bool _synchronizedSndCtrl = false;              // sndctrlとの接続状態
        private PreviewPlayer _selectedPreviewPlayer = null; // 選択中のプレビュープレイヤー
        private PreviewPlayerSettings _settings = null;      // 設定

        public PreviewPlayerManager() { }

        //-----------------------------------------------------------------

        public event EventHandler ViewerSynchronousChanged;
        public event EventHandler PreviewPlayerSelectChanged;
        public event PreviewPlayerEventHandler SelectedPreviewPlayerStateChanged;

        //-----------------------------------------------------------------

        public bool IsDirty
        {
            get { return _settings.IsDirty; }
            set { _settings.IsDirty = value; }
        }

        public PreviewPlayerSettings Settings
        {
            get { return _settings; }
        }

        public PreviewPlayerCollection PreviewPlayers
        {
            get { return _previewPlayers; }
        }

        public PreviewPlayer SelectedPreviewPlayer
        {
            get { return _selectedPreviewPlayer; }
            set
            {
                if (value == _selectedPreviewPlayer) { return; }
                if (!_previewPlayers.Contains(value)) { return; }

                _selectedPreviewPlayer = value;

                OnPreviewPlayerSelectChangedEvent(new EventArgs());
            }
        }

        public PreviewPlayerType PreviewPlayerType
        {
            get { return _previewPlayers[0].Type; }
        }

        public uint SelectedPreviewPlayerIndex
        {
            get
            {
                if (null == _selectedPreviewPlayer) { return uint.MaxValue; }
                return _selectedPreviewPlayer.Index;
            }
            set
            {
                if (value == uint.MaxValue) { throw new ArgumentOutOfRangeException(); }
                if (value >= _previewPlayers.Count) { throw new ArgumentOutOfRangeException(); }

                SelectedPreviewPlayer = _previewPlayers[(int)value];
            }
        }

        public GlobalVariableContainerSet GlobalVariables
        {
            get { return _globalVariableSet; }
        }

        public bool IsSynchronizedViewer
        {
            get { return _synchronizedViewer; }
        }

        private CommunicationService CommunicationService
        {
            get
            {
#if BUILD_WAVE_EXPORTER
                return null;
#else
                return FormsApplicationCommon.Instance.CommunicationService;
#endif
            }
        }


        /// <summary>
        /// パッキングされた全シーケンス変数を取得します。
        /// </summary>
        private SeqVariableContainerCollection AllSeqVariables
        {
            get
            {
                SeqVariableContainerCollection work = new SeqVariableContainerCollection();

                // グローバル変数
                work.Add(_globalVariableSet.Edit);

                // プレビュープレイヤーが保持する変数
                foreach (PreviewPlayer player in _previewPlayers)
                {

                    // ローカル変数
                    work.Add(player.LocalVariables.Edit);

                    // トラック変数
                    foreach (TrackVariableContainerSet trackVariables in player.TrackVariables.Items)
                    {
                        work.Add(trackVariables.Edit);
                    }

                }

                return work;
            }
        }

        //-----------------------------------------------------------------

        public void Initialize()
        {
            Initialize(DefaultPlayerCount);
        }

        public void Initialize(uint playerCount)
        {
            if (_previewPlayers.Count == playerCount) { return; }
            if (MaxPlayerCount < playerCount) { throw new ArgumentOutOfRangeException(); }

            // プレビュープレイヤー設定クラスを作成する
            _settings = new PreviewPlayerSettings(this);


            //
            // プレビュープレイヤーを再構築する
            //
            _previewPlayers.Clear();

            for (int i = 0; i < playerCount; i++)
            {

                PreviewPlayer newPlayer = new PreviewPlayer(this, (uint)i);

                newPlayer.SoundChanging += OnPreviewPlayerSoundChanging;
                newPlayer.SoundChanged += OnPreviewPlayerSoundChanged;
                newPlayer.PlayerStateChanged += OnPreviewPlayerStateChanged;

                _previewPlayers.Add(newPlayer);

            }

            // プレイヤーを更新する
            UpdatePlayer();

            // プレイヤー0を選択する
            if (playerCount > 0)
            {
                SelectedPreviewPlayer = _previewPlayers[0];
            }

            this.CommunicationService.ViewerConnectionChanged += OnViewerConnectionChanged;
            this.CommunicationService.SndEditConnectionChanged += OnSreConnectionChanged;
        }

        public void Uninitialize()
        {
            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.Dispose();
            }

            _previewPlayers.Clear();

            if (null != _settings)
            {
                _settings.Dispose();
                _settings = null;
            }

            this.CommunicationService.ViewerConnectionChanged -= OnViewerConnectionChanged;
            this.CommunicationService.SndEditConnectionChanged -= OnSreConnectionChanged;
        }

        public void ResetAll()
        {
            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.Reset();
            }
        }

        public void StopAll()
        {
            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.Stop();
            }
        }

        public void SetAllSoundLabels()
        {
            if (!IsSynchronizedViewer) { return; }

            // 全てのサウンドラベルを送信する
            this.CommunicationService.SetSoundLabels(this);
        }

        public void SetAllPreviewPlayerParameters()
        {
            if (!IsSynchronizedViewer) { return; }

            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.SetParameter();
            }
        }

        /// <summary>
        /// 全てのシーケンス変数を送信します。
        /// </summary>
        public void SetAllVariables()
        {
            if (!IsSynchronizedViewer) { return; }

            // グローバル変数
            this.CommunicationService.SetSeqVariables(_globalVariableSet.Edit);

            // プレビュープレイヤーが保持する変数
            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.SetAllVariables();
            }
        }

        public void UpdateGlobalVariables()
        {
            _globalVariableSet.Current.LockVariableChangedEvent();

            foreach (SeqVariable target in _globalVariableSet.Current.Variables)
            {
                target.Value = PreviewSequenceSound.GetGlobalVariable((int)target.No);
            }

            // 変数の値に変更があった場合、ここでイベントが発行される
            _globalVariableSet.Current.UnlockVariableChangedEvent();
        }

        /// <summary>
        /// グローバルシーケンス変数を送信します。
        /// </summary>
        public void SetGlobalVariables()
        {
            if (IsSynchronizedViewer)
            {
                this.CommunicationService.SetSeqVariables(_globalVariableSet.Edit);
            }
            else
            {
                SetGlobalVariablesForPC(_globalVariableSet.Edit);
            }
        }

        //-----------------------------------------------------------------

        protected virtual void OnPreviewPlayerSelectChangedEvent(EventArgs e)
        {
            if (null != PreviewPlayerSelectChanged)
            {
                PreviewPlayerSelectChanged(this, e);
            }
        }

        protected virtual void OnViewerSynchronousChanged(EventArgs e)
        {
            // Viewer との接続状態を更新する
            _synchronizedViewer = (ConnectionState.Connected == this.CommunicationService.GetState());

            if (null != ViewerSynchronousChanged)
            {
                ViewerSynchronousChanged(this, e);
            }

            // プレイヤーを更新する
            UpdatePlayer();
        }

        //-----------------------------------------------------------------

        private void SetGlobalVariablesForPC(SeqVariableContainer variables)
        {
            if (null == variables) { throw new ArgumentException("invalid sequence variable type"); }

            // グローバル変数を更新する
            foreach (SeqVariable target in variables.Variables)
            {
                if (!target.Checked) { continue; }
                if (SeqVariable.InvalidValue == target.Value) { continue; }
                PreviewSequenceSound.SetGlobalVariable((int)target.No, (short)target.Value);
            }
        }

        private void UpdatePlayer()
        {
            PreviewPlayerType newPlayerType = PreviewPlayerType.PCPlayer;

            if (IsSynchronizedViewer != false || _synchronizedSndCtrl != false)
            {
                newPlayerType = PreviewPlayerType.ViewerPlayer;
            }


            // 全サウンド停止
            StopAll();

            // プレイヤーを切り替える
            SwitchPlayer(newPlayerType);


            //
            // PCEmurator / Viewer との同期
            //
            if (IsSynchronizedViewer)
            {

                // 全てのサウンドラベルを送信する（Viewerのみ）
                SetAllSoundLabels();

                // 全てのプレビュープレイヤーパラメータを送信する（Viewerのみ）
                SetAllPreviewPlayerParameters();

            }

            // 同期を開始または終了する
            Settings.SeqVariableWatcher.Reset();
            Settings.PlayerStateWatcher.Reset();
            Settings.PreviewPlayerParameterWatcher.Reset();
            Settings.EffectWatcher.Reset();
        }

        private void SwitchPlayer(PreviewPlayerType newPlayerType)
        {
            if (newPlayerType == this.PreviewPlayerType) { return; }

            // 設定を切り替える
            _settings.SwitchPlayer(newPlayerType, this.CommunicationService);

            // プレイヤーを切り替える
            foreach (PreviewPlayer player in _previewPlayers)
            {
                player.SwitchPlayer(newPlayerType);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// Viewer との接続状態が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの発行者</param>
        /// <param name="e">Viewer 接続状態イベント</param>
        private void OnViewerConnectionChanged(object sender, ConnectionChangedEventArgs e)
        {
            switch (e.State)
            {
                case ConnectionState.Connected:
                    // Viewer との接続状態を更新する
                    OnViewerSynchronousChanged(new EventArgs());
                    break;

                case ConnectionState.Disconnected:
                    // 切断されたら、Viewer との接続状態を更新する
                    OnViewerSynchronousChanged(new EventArgs());
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSreConnectionChanged(object sender, ConnectionChangedEventArgs e)
        {
            switch (e.State)
            {
                case ConnectionState.Connected:
                    // Viewer との接続状態を更新する
                    //OnViewerSynchronousChanged(new EventArgs());

                    _synchronizedSndCtrl = true;

                    // プレイヤーを更新する
                    UpdatePlayer();
                    break;

                case ConnectionState.Disconnected:
                    // 切断されたら、Viewer との接続状態を更新する
                    //OnViewerSynchronousChanged(new EventArgs());

                    _synchronizedSndCtrl = false;

                    // プレイヤーを更新する
                    UpdatePlayer();
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Sound GetSound(IParameterDictionary parameterDictionary)
        {
            foreach (PreviewPlayer previewPlayer in PreviewPlayers)
            {
                if (previewPlayer.Item != null)
                {
                    if (previewPlayer.Item.Parameters == parameterDictionary)
                    {
                        return previewPlayer.Item;
                    }

                    if (previewPlayer.Item is StreamSoundBase)
                    {
                        foreach (StreamSoundTrackBase track in previewPlayer.Item.Children)
                        {
                            if (track.Parameters == parameterDictionary)
                            {
                                return previewPlayer.Item;
                            }
                        }
                    }
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private void AddParameterChangedEvent(Component component)
        {
            component.Parameters.ParameterValueChanged += OnSoundParameterChanged;

            if (component is StreamSoundBase)
            {
                component.Children.CollectionChanged += OnCollectionChanged;

                foreach (StreamSoundTrackBase track in component.Children)
                {
                    track.Parameters.ParameterValueChanged += OnSoundParameterChanged;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void RemoveParameterChangedEvent(Component component)
        {
            component.Parameters.ParameterValueChanged -= OnSoundParameterChanged;

            if (component is StreamSoundBase)
            {
                component.Children.CollectionChanged -= OnCollectionChanged;

                foreach (StreamSoundTrackBase track in component.Children)
                {
                    track.Parameters.ParameterValueChanged -= OnSoundParameterChanged;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (Component component in e.NewItems)
                    {
                        component.Parameters.ParameterValueChanged += OnSoundParameterChanged;
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (Component component in e.OldItems)
                    {
                        component.Parameters.ParameterValueChanged -= OnSoundParameterChanged;
                    }
                    break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundParameterChanged(object sender, ParameterEventArgs e)
        {
            if (e.Key == ProjectParameterNames.Volume)
            {
                Sound sound = GetSound(sender as IParameterDictionary);
                Debug.Assert(sound != null, "Sound not regist on preview player.");
                PreviewPlayerOperator.SetParameter(e.Key, sound);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlayerSoundChanging(PreviewPlayer sender, EventArgs e)
        {
            if (sender.Item != null)
            {
                RemoveParameterChangedEvent(sender.Item);
                //sender.Item.Parameters.ParameterValueChanged -= OnSoundParameterChanged;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlayerSoundChanged(PreviewPlayer sender, EventArgs e)
        {
            if (sender.Item != null)
            {
                AddParameterChangedEvent(sender.Item);
                //sender.Item.Parameters.ParameterValueChanged += OnSoundParameterChanged;
            }

            if (!IsSynchronizedViewer) { return; }

            // サウンドラベルを送信する
            this.CommunicationService.SetSoundLabel(sender);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewPlayerStateChanged(PreviewPlayer sender, EventArgs e)
        {
            if (_selectedPreviewPlayer == sender)
            {

                if (null != SelectedPreviewPlayerStateChanged)
                {
                    SelectedPreviewPlayerStateChanged(sender, e);
                }
            }
        }
    }
}
