﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public partial class PreviewPlayer : IDisposable, IPreviewPlayer
    {
        #region ** 固定値

        public const uint InvalidIndex = uint.MaxValue; // インデックスの無効値

        #endregion

        #region ** デリゲート

        private delegate void EventDispatcher(EventArgs e);

        #endregion

        #region ** クラス

        public class VolumeParameter
        {
            #region ** 定数

            public const float VolumeMin = 0.00f;
            public const float VolumeMax = 4.00f;

            #endregion

            #region ** パラメータ

            private bool _enabled;
            private float _volume;

            #endregion

            public VolumeParameter(bool enabled, float volume)
            {
                Enabled = enabled;
                Volume = volume;
            }

            #region ** プロパティ

            public bool Enabled
            {
                get { return _enabled; }
                set
                {
                    if (value == _enabled) { return; }
                    _enabled = value;

                    OnParameterChanged(new EventArgs());
                }
            }

            public float Volume
            {
                get { return _volume; }
                set
                {
                    if (value < VolumeMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > VolumeMax) { throw new ArgumentOutOfRangeException("value"); }
                    _volume = value;

                    OnParameterChanged(new EventArgs());
                }
            }

            #endregion

            #region ** イベント

            public event EventHandler ParameterChanged;

            #endregion

            #region ** イベントハンドラ

            protected void OnParameterChanged(EventArgs e)
            {
                if (null == ParameterChanged) { return; }

                ParameterChanged(this, e);
            }

            #endregion
        }

        public class TrackParameter
        {
            private VolumeParameter volumeParameter;

            public TrackParameter(bool enable, float volume, bool mute, bool solo, bool couple)
            {
                this.volumeParameter = new VolumeParameter(enable, volume);
                this.Mute = mute;
                this.Solo = solo;
                this.Couple = couple;
            }

            public float Volume
            {
                get
                {
                    return this.volumeParameter.Volume;
                }
                set
                {
                    this.volumeParameter.Volume = value;
                }
            }
            public bool Enabled
            {
                get
                {
                    return this.volumeParameter.Enabled;
                }
                set
                {
                    this.volumeParameter.Enabled = value;
                }
            }
            public bool Mute { get; set; }
            public bool Solo { get; set; }
            public bool Couple { get; set; }

            public event EventHandler ParameterChanged
            {
                add
                {
                    if (this.volumeParameter != null)
                    {
                        this.volumeParameter.ParameterChanged += value;
                    }
                }
                remove
                {
                    if (this.volumeParameter != null)
                    {
                        this.volumeParameter.ParameterChanged -= value;
                    }
                }
            }
        }

        public partial class Parameters
        {
            #region ** 定数

            // デフォルト値
            public const float VolumeDefault = 1.00f;
            public const float PitchDefault = 1.00f;

            public const float PanDefault = 0.00f;
            public const float SurroundPanDefault = 0.00f;

            public const float MainPanDefault = 0.00f;
            public const float MainSurroundPanDefault = 0.00f;

            public const float DrcPanDefault = 0.00f;
            public const float DrcSurroundPanDefault = 0.00f;

            public const float LpfFrequencyDefault = 0.00f;
            public const BiquadType BiquadTypeDefault = BiquadType.None;
            public const float BiquadValueDefault = 0.00f;
            public const uint StartOffsetTimeDefault = 0;
            public const bool MainOutEnabledDefault = true;
            public const bool DrcOutEnabledDefault = true;
            public const bool Remote0OutEnabledDefault = false;
            public const bool Remote1OutEnabledDefault = false;
            public const bool Remote2OutEnabledDefault = false;
            public const bool Remote3OutEnabledDefault = false;
            public const float MainOutVolumeDefault = 1.00f;
            public const float DrcOutVolumeDefault = 1.00f;
            public const float Remote0OutVolumeDefault = 1.00f;
            public const float Remote1OutVolumeDefault = 1.00f;
            public const float Remote2OutVolumeDefault = 1.00f;
            public const float Remote3OutVolumeDefault = 1.00f;
            public const float MainSendDefault = 0.00f;
            public const float EffectAuxASendDefault = 0.00f;
            public const float EffectAuxBSendDefault = 0.00f;
            public const float EffectAuxCSendDefault = 0.00f;
            public const uint FadeInFramesDefault = 0;
            public const uint FadeOutFramesDefault = 0;
            public const uint PauseFadeFramesDefault = 0;
            public const uint TracksCountDefault = 16;
            public const float TrackVolumeDefault = 1.00f;
            public const bool TrackEnableDefault = true;
            public const bool TrackMuteDefault = false;
            public const bool TrackSoloDefault = false;
            public const bool TrackCoupleDefault = false;

            // 境界値
            private const float VolumeMin = VolumeParameter.VolumeMin;
            private const float VolumeMax = VolumeParameter.VolumeMax;
            private const float PitchMin = 0.01f;
            private const float PitchMax = 4.00f;

            private const float PanMin = -2.00f;
            private const float PanMax = 2.00f;
            private const float SurroundPanMin = -2.00f;
            private const float SurroundPanMax = 2.00f;

            private const float MainPanMin = -2.00f;
            private const float MainPanMax = 2.00f;
            private const float MainSurroundPanMin = -2.00f;
            private const float MainSurroundPanMax = 2.00f;

            private const float DrcPanMin = -2.00f;
            private const float DrcPanMax = 2.00f;
            private const float DrcSurroundPanMin = -2.00f;
            private const float DrcSurroundPanMax = 2.00f;

            private const float LpfFrequencyMin = -1.00f;
            private const float LpfFrequencyMax = 0.00f;
            private const float BiquadValueMin = 0.00f;
            private const float BiquadValueMax = 1.00f;
            private const float SendMin = -1.00f;
            private const float SendMax = 4.00f;
            private const uint FadeFramesMax = 600000;

            #endregion

            #region ** パラメータ

            PreviewPlayer _parent = null;

            // EventInvoker
            private LocableMethodInvoker _parameterChangedInvoker = new LocableMethodInvoker();

            // 基本パラメータ
            private float _volume = VolumeDefault;
            private float _pitch = PitchDefault;

            private float _pan = PanDefault;
            private float _surroundPan = SurroundPanDefault;

            private float mainPan = MainPanDefault;
            private float mainSurroundPan = MainSurroundPanDefault;

            private float drcPan = DrcPanDefault;
            private float drcSurroundPan = DrcSurroundPanDefault;

            private float _lpfFrequency = LpfFrequencyDefault;
            private BiquadType _biquadType = BiquadTypeDefault;
            private float _biquadValue = BiquadValueDefault;
            private uint _startOffsetTime = StartOffsetTimeDefault;

            // 出力
            private VolumeParameter _mainOut = new VolumeParameter(MainOutEnabledDefault, MainOutVolumeDefault);
            private VolumeParameter drcOut = new VolumeParameter(DrcOutEnabledDefault, DrcOutVolumeDefault);
            private VolumeParameter[] _remoteOut = new VolumeParameter[]
                                                    {
                                                        new VolumeParameter( Remote0OutEnabledDefault, Remote0OutVolumeDefault ),
                                                        new VolumeParameter( Remote1OutEnabledDefault, Remote1OutVolumeDefault ),
                                                        new VolumeParameter( Remote2OutEnabledDefault, Remote2OutVolumeDefault ),
                                                        new VolumeParameter( Remote3OutEnabledDefault, Remote3OutVolumeDefault )
                                                    };

            // センド
            private float _mainSend = MainSendDefault;
            private float _effectAuxASend = EffectAuxASendDefault;
            private float _effectAuxBSend = EffectAuxBSendDefault;
            private float _effectAuxCSend = EffectAuxCSendDefault;

            private float mainMainSend = MainSendDefault;
            private float mainEffectAuxASend = EffectAuxASendDefault;
            private float mainEffectAuxBSend = EffectAuxBSendDefault;
            private float mainEffectAuxCSend = EffectAuxCSendDefault;

            private float drcMainSend = MainSendDefault;
            private float drcEffectAuxASend = EffectAuxASendDefault;
            private float drcEffectAuxBSend = EffectAuxBSendDefault;
            private float drcEffectAuxCSend = EffectAuxCSendDefault;

            // フェード
            private uint _fadeInFrames = FadeInFramesDefault;
            private uint _fadeOutFrames = FadeOutFramesDefault;
            private uint _pauseFadeFrames = PauseFadeFramesDefault;

            // トラック
            private TrackParameter[] _tracksParameter;

            #endregion

            public Parameters(PreviewPlayer parent)
            {
                if (null == parent) { throw new ArgumentNullException("parent"); }
                _parent = parent;

                _mainOut.ParameterChanged += OnVolumeParameterChanged;
                drcOut.ParameterChanged += OnVolumeParameterChanged;
                _remoteOut[0].ParameterChanged += OnVolumeParameterChanged;
                _remoteOut[1].ParameterChanged += OnVolumeParameterChanged;
                _remoteOut[2].ParameterChanged += OnVolumeParameterChanged;
                _remoteOut[3].ParameterChanged += OnVolumeParameterChanged;

                _tracksParameter = new TrackParameter[TracksCountDefault];
                for (int i = 0; i < TracksCountDefault; i++)
                {
                    _tracksParameter[i] = new TrackParameter(TrackEnableDefault,
                                                             TrackVolumeDefault,
                                                             TrackMuteDefault,
                                                             TrackSoloDefault,
                                                             TrackCoupleDefault);
                    _tracksParameter[i].ParameterChanged += OnVolumeParameterChanged;
                }
            }

            #region ** プロパティ

            public PreviewPlayer Parent
            {
                get { return _parent; }
            }

            public float Volume
            {
                get { return _volume; }
                set
                {
                    if (value == _volume) { return; }
                    if (value < VolumeMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > VolumeMax) { throw new ArgumentOutOfRangeException("value"); }

                    _volume = value;
                    DispatchEvent();
                }
            }

            public float Pitch
            {
                get { return _pitch; }
                set
                {
                    if (value == _pitch) { return; }
                    if (value < PitchMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > PitchMax) { throw new ArgumentOutOfRangeException("value"); }

                    _pitch = value;
                    DispatchEvent();
                }
            }

            public float Pan
            {
                get { return _pan; }
                set
                {
                    if (value == _pan) { return; }
                    if (value < PanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > PanMax) { throw new ArgumentOutOfRangeException("value"); }

                    _pan = value;
                    DispatchEvent();
                }
            }

            public float SurroundPan
            {
                get { return _surroundPan; }
                set
                {
                    if (value == _surroundPan) { return; }
                    if (value < SurroundPanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SurroundPanMax) { throw new ArgumentOutOfRangeException("value"); }

                    _surroundPan = value;
                    DispatchEvent();
                }
            }

            public float MainPan
            {
                get { return mainPan; }
                set
                {
                    if (value == mainPan) { return; }
                    if (value < MainPanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > MainPanMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainPan = value;
                    DispatchEvent();
                }
            }

            public float MainSurroundPan
            {
                get { return mainSurroundPan; }
                set
                {
                    if (value == mainSurroundPan) { return; }
                    if (value < MainSurroundPanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > MainSurroundPanMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainSurroundPan = value;
                    DispatchEvent();
                }
            }

            public float DrcPan
            {
                get { return drcPan; }
                set
                {
                    if (value == drcPan) { return; }
                    if (value < DrcPanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > DrcPanMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcPan = value;
                    DispatchEvent();
                }
            }

            public float DrcSurroundPan
            {
                get { return drcSurroundPan; }
                set
                {
                    if (value == drcSurroundPan) { return; }
                    if (value < DrcSurroundPanMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > DrcSurroundPanMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcSurroundPan = value;
                    DispatchEvent();
                }
            }

            public float LpfFrequency
            {
                get { return _lpfFrequency; }
                set
                {
                    if (value == _lpfFrequency) { return; }
                    if (value < LpfFrequencyMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > LpfFrequencyMax) { throw new ArgumentOutOfRangeException("value"); }

                    _lpfFrequency = value;
                    DispatchEvent();
                }
            }

            public BiquadType BiquadType
            {
                get { return _biquadType; }
                set { _biquadType = value; }
            }

            public float BiquadValue
            {
                get { return _biquadValue; }
                set
                {
                    if (value == _biquadValue) { return; }
                    if (value < BiquadValueMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > BiquadValueMax) { throw new ArgumentOutOfRangeException("value"); }

                    _biquadValue = value;
                    DispatchEvent();
                }
            }

            public uint StartOffsetTime
            {
                get { return _startOffsetTime; }
                set
                {
                    if (value == _startOffsetTime) { return; }
                    _startOffsetTime = value;
                    DispatchEvent();
                }
            }

            public VolumeParameter MainOut
            {
                get { return _mainOut; }
                set
                {
                    if (value == _mainOut) { return; }

                    _mainOut = value;
                    DispatchEvent();
                }
            }

            public VolumeParameter DrcOut
            {
                get { return drcOut; }
                set
                {
                    if (value == drcOut) { return; }

                    drcOut = value;
                    DispatchEvent();
                }
            }

            public VolumeParameter[] RemoteOut
            {
                get { return _remoteOut; }
                set
                {
                    if (value == _remoteOut) { return; }

                    _remoteOut = value;
                    DispatchEvent();
                }
            }

            public float MainSend
            {
                get { return _mainSend; }
                set
                {
                    if (value == _mainSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    _mainSend = value;
                    DispatchEvent();
                }
            }

            public float EffectAuxASend
            {
                get { return _effectAuxASend; }
                set
                {
                    if (value == _effectAuxASend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    _effectAuxASend = value;
                    DispatchEvent();
                }
            }

            public float EffectAuxBSend
            {
                get { return _effectAuxBSend; }
                set
                {
                    if (value == _effectAuxBSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    _effectAuxBSend = value;
                    DispatchEvent();
                }
            }

            public float EffectAuxCSend
            {
                get { return _effectAuxCSend; }
                set
                {
                    if (value == _effectAuxCSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    _effectAuxCSend = value;
                    DispatchEvent();
                }
            }

            public float MainMainSend
            {
                get { return mainMainSend; }
                set
                {
                    if (value == mainMainSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainMainSend = value;
                    DispatchEvent();
                }
            }

            public float MainEffectAuxASend
            {
                get { return mainEffectAuxASend; }
                set
                {
                    if (value == mainEffectAuxASend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainEffectAuxASend = value;
                    DispatchEvent();
                }
            }

            public float MainEffectAuxBSend
            {
                get { return mainEffectAuxBSend; }
                set
                {
                    if (value == mainEffectAuxBSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainEffectAuxBSend = value;
                    DispatchEvent();
                }
            }

            public float MainEffectAuxCSend
            {
                get { return mainEffectAuxCSend; }
                set
                {
                    if (value == mainEffectAuxCSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    mainEffectAuxCSend = value;
                    DispatchEvent();
                }
            }

            public float DrcMainSend
            {
                get { return drcMainSend; }
                set
                {
                    if (value == drcMainSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcMainSend = value;
                    DispatchEvent();
                }
            }

            public float DrcEffectAuxASend
            {
                get { return drcEffectAuxASend; }
                set
                {
                    if (value == drcEffectAuxASend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcEffectAuxASend = value;
                    DispatchEvent();
                }
            }

            public float DrcEffectAuxBSend
            {
                get { return drcEffectAuxBSend; }
                set
                {
                    if (value == drcEffectAuxBSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcEffectAuxBSend = value;
                    DispatchEvent();
                }
            }

            public float DrcEffectAuxCSend
            {
                get { return drcEffectAuxCSend; }
                set
                {
                    if (value == drcEffectAuxCSend) { return; }
                    if (value < SendMin) { throw new ArgumentOutOfRangeException("value"); }
                    if (value > SendMax) { throw new ArgumentOutOfRangeException("value"); }

                    drcEffectAuxCSend = value;
                    DispatchEvent();
                }
            }

            public uint FadeInFrames
            {
                get { return _fadeInFrames; }
                set
                {
                    if (value == _fadeInFrames) { return; }
                    if (value > FadeFramesMax) { throw new ArgumentOutOfRangeException("value"); }

                    _fadeInFrames = value;
                    DispatchEvent();
                }
            }

            public uint FadeOutFrames
            {
                get { return _fadeOutFrames; }
                set
                {
                    if (value == _fadeOutFrames) { return; }
                    if (value > FadeFramesMax) { throw new ArgumentOutOfRangeException("value"); }

                    _fadeOutFrames = value;
                    DispatchEvent();
                }
            }

            public uint PauseFadeFrames
            {
                get { return _pauseFadeFrames; }
                set
                {
                    if (value == _pauseFadeFrames) { return; }
                    if (value > FadeFramesMax) { throw new ArgumentOutOfRangeException("value"); }

                    _pauseFadeFrames = value;
                    DispatchEvent();
                }
            }

            public TrackParameter[] TracksParameter
            {
                get { return _tracksParameter; }
                set
                {
                    if (value == _tracksParameter) { return; }

                    _tracksParameter = value;
                    DispatchEvent();
                }
            }

            #endregion

            #region ** イベント

            public event EventHandler ParameterChanged;

            #endregion

            #region ** メソッド

            public void LockEvent()
            {
                _parameterChangedInvoker.Lock();
            }

            public void UnlockEvent()
            {
                _parameterChangedInvoker.Unlock(ParameterChanged, this, new EventArgs());
            }

            public void ResetEvent()
            {
                _parameterChangedInvoker.Reset();
            }

            protected void OnVolumeParameterChanged(object sender, EventArgs e)
            {
                DispatchEvent();
            }

            private void DispatchEvent()
            {
                _parameterChangedInvoker.Invoke(ParameterChanged, this, new EventArgs());
            }

            #endregion
        }

        private class LocableMethodInvoker
        {
            // 状態
            private bool _locked = false;
            private bool _dirty = false;

            #region ** プロパティ

            public bool Locked
            {
                get { return _locked; }
            }
            public bool Dirty
            {
                get { return _dirty; }
            }

            #endregion

            #region ** メソッド

            public bool Invoke(Delegate e)
            {
                return Invoke(e, null);
            }

            public bool Invoke(Delegate method, params object[] args)
            {
                _dirty = true;
                if (_locked) { return false; }

                if (null != method)
                {
                    method.DynamicInvoke(args);
                }

                _dirty = false;
                return true;
            }

            public void Lock()
            {
                if (_locked) { return; }
                _locked = true;
            }

            public void Unlock(Delegate method)
            {
                Unlock(method, null);
            }

            public void Unlock(Delegate method, params object[] args)
            {
                _locked = false;

                if (!_dirty) { return; }

                Invoke(method, args);
            }

            public void Reset()
            {
                _dirty = false;
            }

            #endregion
        }

        #endregion

        #region ** パラメータ

        private PreviewPlayerManager _parent = null;

        // 状態
        private Sound _item = null;    // 対象サウンド
        private BasePlayer _player = null;    // 現在のプレイヤー

        // パラメータ
        private uint _index = 0;       // プレビュープレイヤーのインデックス
        private Parameters _paramters = null;    // パラメータ

        // シーケンス変数
        private LocalVariableContainerSet _localVariableSet;   // ローカル変数
        private TrackVariableContainerSets _trackVariableSets;  // トラック変数

        // EventInvoker
        private LocableMethodInvoker _stateChangedInvoker = new LocableMethodInvoker();

        #endregion

        public PreviewPlayer(PreviewPlayerManager parent, uint index)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }

            _parent = parent;
            _index = index;

            _localVariableSet = new LocalVariableContainerSet(this);
            _trackVariableSets = new TrackVariableContainerSets(this);

            // パラメータを初期化する
            _paramters = new Parameters(this);
        }

        #region ** プロパティ

        /// <summary>
        /// PreviewPlayerManager を取得します。
        /// </summary>
        public PreviewPlayerManager Parent
        {
            get { return _parent; }
        }

        /// <summary>
        /// プレビュープレイヤーのインデックスを取得します。
        /// </summary>
        public uint Index
        {
            get { return _index; }
        }

        public Sound Item
        {
            get { return _item; }
            set
            {
                if (value == _item) { return; }

                // ロック中の場合は変更しない
                if (ItemLocked) { return; }

                UpdateItem(value);
            }
        }

        /// 再生中のストリームサウンドトラックです。
        //private StreamSoundTrackBase _playTrack = null;

        /// <summary>
        /// プレイヤーに表示するテキストです。（サウンド名）
        /// </summary>
        public string Text
        {
            get
            {
                if (this._player == null)
                {
                    Debug.Assert(false, "Text: _player is null");
                    return String.Empty;
                }

                if (_player.Sound == null || _player.Sound.Sound == null)
                {
                    return string.Empty;
                }
                else if (_player.Sound.Sound is StreamSoundBase)
                {
                    StreamSoundBase stream = _player.Sound.Sound as StreamSoundBase;

#if false
                    if (_playTrack != null)
                    {
                        return stream.Name + " / " + Path.GetFileName(_playTrack.FilePath);
                    }
                    else
                    {
                        foreach( StreamSoundTrack track in stream.Children )
                        {
                            if( track.PreviewSoloPlay != false )
                            {
                                _playTrack = track;
                                return stream.Name + " / " + Path.GetFileName(track.FilePath);
                            }
                        }
                    }
#else
                    string name = stream.Name;
                    StreamSoundTrackBase[] tracks = GetSoloPlayTracks();
                    switch (tracks.Length)
                    {
                        case 0:
                            break;

                        case 1:
                            name += " / " + Path.GetFileName(tracks[0].FilePath);
                            break;

                        default:
                            name += " / *";
                            break;
                    }
                    return name;
#endif
                }

                return _player.Sound.Sound.Name;
            }
        }

#if true

        /// <summary>
        ///
        /// </summary>
        private Sound GetSound()
        {
            if (this._player == null ||
                this._player.Sound == null)
            {
                return null;
            }
            return this._player.Sound.Sound;
        }

        /// <summary>
        ///
        /// </summary>
        private StreamSoundTrackBase[] GetSoloPlayTracks()
        {
            List<StreamSoundTrackBase> list = new List<StreamSoundTrackBase>();
            StreamSoundBase stream = GetSound() as StreamSoundBase;
            if (stream != null)
            {
                foreach (StreamSoundTrackBase track in stream.Children)
                {
                    if (track.PreviewSoloPlay != false)
                    {
                        list.Add(track);
                    }
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// いずれかのストリームサウンドトラックでソロ再生が有効になっているのかを調べます。
        /// </summary>
        public bool ContainsSoloPlayTrack
        {
            get
            {
#if false
                if( this._playTrack != null )
                {
                    StreamSound stream = this._playTrack.Parent as StreamSound;
                    foreach( StreamSoundTrack track in stream.Children )
                    {
                        if( track.PreviewSoloPlay != false )
                        {
                            return true;
                        }
                    }
                }
#else

                StreamSoundBase stream = GetSound() as StreamSoundBase;
                if (stream != null)
                {
                    foreach (StreamSoundTrackBase track in stream.Children)
                    {
                        if (track.PreviewSoloPlay != false)
                        {
                            return true;
                        }
                    }
                }
#endif
                return false;
            }
        }

        /// <summary>
        /// いずれかのストリームサウンドトラックでミュートが有効になっているのかを調べます。
        /// </summary>
        public bool ContainsMuteTrack
        {
            get
            {
                foreach (StreamSoundTrackBase track in PlayStreamSoundTracks)
                {
                    if (track.PreviewMute != false)
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        ///
        private StreamSoundTrackBase[] PlayStreamSoundTracks
        {
            get
            {
                if (this.Item != null &&
                    this.Item is StreamSoundBase)
                {
                    StreamSoundBase stream = this.Item as StreamSoundBase;
                    return stream.Children
                        .Cast<StreamSoundTrackBase>()
                        .ToArray();
                }
                else
                {
                    return new StreamSoundTrackBase[0];
                }
            }
        }


#endif

        /// <summary>
        ///
        /// </summary>
        public bool ItemLocked
        {
            get
            {
                if (null == _item) { return false; }

                // 無効、停止中は、ロックしない
                switch (State)
                {
                    case PreviewPlayerState.Disabled:
                    case PreviewPlayerState.Stopped:
                        return false;
                }

                return true;
            }
        }

        /// <summary>
        /// プレイヤーの種類を取得します。
        /// </summary>
        public PreviewPlayerType Type
        {
            get
            {
                if (null == _player) { return PreviewPlayerType.Unknown; }
                return _player.Type;
            }
        }

        /// <summary>
        /// 再生状態を取得します。
        /// </summary>
        public PreviewPlayerState State
        {
            get
            {
                if (null == _player) { return PreviewPlayerState.Stopped; }
                return _player.State;
            }
        }

        /// <summary>
        /// リピート状態を取得または設定します。
        /// </summary>
        public bool Repeat
        {
            get { return _player.Repeat; }
            set { _player.Repeat = value; }
        }

        /// <summary>
        /// プレビュープレイヤーが停止中かどうかを返します。
        /// </summary>
        public bool Stopped
        {
            get
            {
                return (PreviewPlayerState.Disabled == State || PreviewPlayerState.Stopped == State);
            }
        }

        ///
        public bool Playing
        {
            get
            {
                return PreviewPlayerState.Playing == State ? true : false;
            }
        }

        /// <summary>
        /// 再生可能な状態かどうかを取得します。
        /// </summary>
        public bool CanPlay
        {
            get { return _player.CanPlay; }
        }

        /// <summary>
        /// 一時停止可能な状態かどうかを取得します。
        /// </summary>
        public bool CanPause
        {
            get { return _player.CanPause; }
        }

        /// <summary>
        /// 停止可能な状態かどうかを取得します。
        /// </summary>
        public bool CanStop
        {
            get { return _player.CanStop; }
        }

        /// <summary>
        /// リピート可能な状態かどうかを取得します。
        /// </summary>
        public bool CanRepeat
        {
            get { return _player.CanRepeat; }
        }

        public Parameters Paramters
        {
            get { return _paramters; }
        }

        /// <summary>
        /// ローカル変数セットを取得します。
        /// </summary>
        public LocalVariableContainerSet LocalVariables
        {
            get { return _localVariableSet; }
        }

        /// <summary>
        /// トラック変数セットを取得します。
        /// </summary>
        public TrackVariableContainerSets TrackVariables
        {
            get { return _trackVariableSets; }
        }

        /// <summary>
        /// パッキングされた全シーケンス変数を取得します。
        /// </summary>
        private SeqVariableContainerCollection AllSeqVariables
        {
            get
            {
                SeqVariableContainerCollection work = new SeqVariableContainerCollection();

                // ローカル変数
                work.Add(_localVariableSet.Edit);

                // トラック変数
                foreach (TrackVariableContainerSet trackVariableSet in _trackVariableSets.Items)
                {
                    work.Add(trackVariableSet.Edit);
                }

                return work;
            }
        }

        #endregion

        #region ** イベント

        public event PreviewPlayerEventHandler SoundChanging;
        public event PreviewPlayerEventHandler SoundChanged;
        public event PreviewPlayerEventHandler ItemLockedChanged;
        public event PreviewPlayerEventHandler PlayerStateChanged;
        public event PreviewPlayerEventHandler RepeatChanged;

        public event EventHandler MuteChanged;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnSoundChanging(EventArgs e)
        {
            if (SoundChanging != null)
            {
                SoundChanging(this, e);
            }
        }

        protected virtual void OnSoundChanged(EventArgs e)
        {
            if (null != SoundChanged)
            {
                SoundChanged(this, e);
            }

            if (null != _player.Sound && !_player.Enabled)
            {
                _player.Enabled = true;
                _player.UpdateState();
            }
        }

        protected virtual void OnItemLockedChanged(EventArgs e)
        {
            if (null != ItemLockedChanged)
            {
                ItemLockedChanged(this, e);
            }
        }

        protected virtual void OnPlayerStateChanged(EventArgs e)
        {
            if (null != PlayerStateChanged)
            {
                PlayerStateChanged(this, e);
            }

            OnItemLockedChanged(new EventArgs());
        }

        protected virtual void OnRepeatChanged(EventArgs e)
        {
            if (null != RepeatChanged)
            {
                RepeatChanged(this, e);
            }
        }

        private void OnInternalPlayerStateChanged(object sender, EventArgs e)
        {
            _stateChangedInvoker.Invoke(new EventDispatcher(OnPlayerStateChanged), new EventArgs());
        }

        private void OnInternalRepeatChanged(object sender, EventArgs e)
        {
            OnRepeatChanged(new EventArgs());
        }

        #endregion

        #region ** メソッド

        public static PreviewSound CreatePreviewSound(Sound sound)
        {
            return CreatePreviewSound(sound, OutputWaveFileRenderType.k32KHz);
        }

        public static PreviewSound CreatePreviewSound(Sound sound, OutputWaveFileRenderType samplingRate, bool forceNoLoop = false)
        {
            if (null == sound)
            {
                throw new ArgumentNullException("sound");
            }

            if (sound is StreamSoundBase)
            {
                return new PreviewStreamSoundCommon(sound as StreamSoundBase, samplingRate, forceNoLoop);
            }

            if (sound is WaveSoundBase)
            {
                return new PreviewWaveSound(sound as WaveSoundBase, samplingRate, forceNoLoop);
            }

            if (sound is SequenceSoundBase)
            {
                SequenceSoundBase sequenceSound = sound as SequenceSoundBase;

                if (!ApplicationBase.Instance.ProjectService.
                    ComponentDictionary.Contains(sequenceSound.SoundSetBankReferences[0].TargetName))
                {
                    throw new ApplicationException(
                                string.Format(FrameworkResources.MessageResource.Message_TargetBankNotFound,
                                               sequenceSound.SoundSetBankReferences));
                }

                return new PreviewSequenceSound
                    (sequenceSound, ApplicationBase.Instance.ProjectService, samplingRate);
            }

            return null;
        }

        ///
        public void SetParameter(string parameterName, Sound item)
        {
            if (this._player == null)
            {
                Debug.Assert(false, "SetParameter: _player is null");
                return;
            }

            if (_player.Sound == null || _player.Sound.Sound != item)
            {
                return;
            }

            _player.SetParameter(parameterName);
        }

        ///
        public void Play()
        {
            if (null == _player) { return; }
            _player.Play();
        }

        ///
        public void Play(Sound item)
        {
            if (null == item) { throw new ArgumentNullException("item"); }

            // 状態ロック
            LockState();

            try
            {

                // ロック中でも変更する
                UpdateItem(item);

                Play();

            }
            catch (Exception e)
            {
                Stop();
                throw e;
            }
            finally
            {
                UnlockState();
            }
        }

        public void Pause()
        {
            if (null == _player) { return; }
            _player.Pause();
        }

        /// <summary>
        ///
        /// </summary>
        public void MuteChannel(Component component, bool value)
        {
            this._player.MuteChannel(component, value);

            if (MuteChanged != null)
            {
                MuteChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void Stop()
        {
            if (null == _player) { return; }
            _player.Stop();
        }

        public void Reset()
        {
            UpdateItem(null);
        }

        public void SetParameter()
        {
            if (null == _player) { return; }
            _player.SetParameter();
        }

        public void UpdateLocalVariables()
        {
            if (null == _player) { return; }
            _player.GetVariables(_localVariableSet.Current);
        }

        public void SetLocalVariables()
        {
            if (null == _player) { return; }
            _player.SetVariables(_localVariableSet.Edit);
        }

        public void UpdateTrackVariables(uint trackNo)
        {
            if (null == _player) { return; }
            _player.GetVariables(_trackVariableSets.Items[(int)trackNo].Current);
        }

        public void SetTrackVariables(uint trackNo)
        {
            if (null == _player) { return; }
            _player.SetVariables(_trackVariableSets.Items[(int)trackNo].Edit);
        }

        public void SetAllVariables()
        {
            if (null == _player) { return; }
            _player.SetVariables(AllSeqVariables);
        }

        public void SwitchPlayer(PreviewPlayerType newPlayerType)
        {
            if (null != _player && newPlayerType == _player.Type) { return; }

            // 現在のプレイヤーを切り離す
            PreviewPlayerState oldState = PreviewPlayerState.Disabled;
            bool oldRepeat = false;

            if (null != _player)
            {

                // 現在の状態を初期状態としてプレイヤーを切り替える
                oldState = _player.State;
                oldRepeat = _player.Repeat;

            }

            DetachPlayer();


            //
            // 新しいプレイヤーを作成して、イベントハンドラを登録する
            //
            BasePlayer newPlayer = null;

            switch (newPlayerType)
            {
                case PreviewPlayerType.PCPlayer:
                    newPlayer = new PCPlayer(this, oldState);
                    newPlayer.Repeat = oldRepeat;
                    break;

                case PreviewPlayerType.ViewerPlayer:
                    newPlayer = new ViewerPlayer(this, oldState);
                    newPlayer.Repeat = oldRepeat;
                    break;
            }

            // イベントハンドラ登録 & PreviewPlayerWatcherへの関連付け
            newPlayer.PlayerStateChanged += OnInternalPlayerStateChanged;
            newPlayer.RepeatChanged += OnInternalRepeatChanged;
            _parent.Settings.PlayerStateWatcher.Attach(newPlayer.Information);


            if (oldState != PreviewPlayerState.Disabled)
            {
                newPlayer.Stop();
            }

            // プレイヤーを更新する
            _player = newPlayer;

            // PCPlayer の場合は、サウンド再生クラスのインスタンスを生成する
            // ViewerPlayer の場合は、サウンド名を設定する
            if (null != _item)
            {
                if (PreviewPlayerType.PCPlayer == newPlayerType)
                {
                    newPlayer.Sound = CreatePreviewSound(_item);
                }
                else
                {
                    newPlayer.SoundName = _item.Name;
                }
            }
        }

        private void DetachPlayer()
        {
            if (null == _player) { return; }

            _player.PlayerStateChanged -= OnInternalPlayerStateChanged;
            _player.RepeatChanged -= OnInternalRepeatChanged;
            _parent.Settings.PlayerStateWatcher.Detach(_player.Information);

            _player.Dispose();
            _player = null;
        }

        private void LockState()
        {
            _stateChangedInvoker.Lock();
        }

        private void UnlockState()
        {
            _stateChangedInvoker.Unlock(PlayerStateChanged, this, new EventArgs());
        }

        private void UpdateItem(Sound newItem)
        {
#if false
            if (newItem == _item) { return; }
#endif
            if (!Stopped)
            {
                Stop();
            }

            Debug.Assert(null != _player, "unexpected error");

            OnSoundChanging(new EventArgs());
            _item = newItem;
            //_playTrack = null;

            if (null != newItem)
            {

                try
                {
                    _player.Sound = CreatePreviewSound(newItem);
                }
                catch (Exception exception)
                {
                    Reset();
                    throw exception;
                }

                if (_player.Sound is PreviewSequenceSound)
                {
                    SetAllVariables();
                }

            }
            else
            {
                _player.Reset();
            }

            OnSoundChanged(new EventArgs());
        }
        #endregion

        #region ** IDisposable の実装

        public void Dispose()
        {
            Stop();
            DetachPlayer();
        }

        #endregion
    }

    public class PreviewPlayerCollection : Collection<PreviewPlayer> { }
}
