﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Projects;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace NintendoWare.SoundMaker.Preview.Service
{
    public enum LoudnessMeasurePriority : int
    {
        High = 0,
        Low,
    }

    public class LoudnessMeasureStack
    {
        private readonly Dictionary<Sound, LoudnessMeasureStackItem> _dictionary = new Dictionary<Sound, LoudnessMeasureStackItem>();
        private List<LoudnessMeasureStackItem>[] _measureStacks = new List<LoudnessMeasureStackItem>[]
        {
            new List<LoudnessMeasureStackItem>(), // High
            new List<LoudnessMeasureStackItem>(), // Low
        };

        public void Push(Sound sound, LoudnessMeasurePriority priority)
        {
            lock (_measureStacks)
            {
                LoudnessMeasureStackItem item;
                if (_dictionary.TryGetValue(sound, out item))
                {
                    if (item.LoudnessMeasurePriority < priority)
                    {
                        // 高優先度で登録済みなら、なにもしません。
                        return;
                    }
                    else
                    {
                        // 高優先度のスタック、またはスタックのトップに移動します。
                        _measureStacks[(int)item.LoudnessMeasurePriority].Remove(item);
                        item.LoudnessMeasurePriority = priority;
                        _measureStacks[(int)priority].Insert(0, item);
                    }
                }
                else
                {
                    // 新規登録します。
                    item = new LoudnessMeasureStackItem(sound, priority);
                    _measureStacks[(int)priority].Insert(0, item);
                    _dictionary.Add(sound, item);
                }
            }
        }

        public LoudnessMeasureStackItem Peek()
        {
            lock (_measureStacks)
            {
                LoudnessMeasureStackItem item = _measureStacks
                    .Where(s => s.Count > 0)
                    .Select(s => s[0])
                    .FirstOrDefault();

                return item;
            }
        }

        public void Remove(LoudnessMeasureStackItem item)
        {
            lock (_measureStacks)
            {
                _measureStacks[(int)item.LoudnessMeasurePriority].Remove(item);
                _dictionary.Remove(item.Sound);
            }
        }

        public void Clear()
        {
            lock (_measureStacks)
            {
                _measureStacks.ForEach(s => s.Clear());
                _dictionary.Clear();
            }
        }

        public void Remove(Sound sound)
        {
            lock (_measureStacks)
            {
                LoudnessMeasureStackItem item;
                if (_dictionary.TryGetValue(sound, out item))
                {
                    _measureStacks[(int)item.LoudnessMeasurePriority].Remove(item);
                    _dictionary.Remove(sound);
                }
            }
        }
    }

    public class LoudnessMeasureStackItem
    {
        public LoudnessMeasureStackItem(Sound sound, LoudnessMeasurePriority priority)
        {
            this.Sound = sound;
            this.LoudnessMeasurePriority = priority;
        }

        public Sound Sound
        {
            get;
        }

        public LoudnessMeasurePriority LoudnessMeasurePriority
        {
            get; internal set;
        }
    }
}
