﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Preview.MCS.Tool
{
    //=========================================================================
    #region MCSツールテストパケット
    /// <summary>
    /// MCSツールテストパケット
    /// <para>
    /// ツールからビューアに送る、テスト用パケットのベースクラスです。
    /// </para>
    /// </summary>
    public abstract class MCSToolTestPacket : MCSToolPacket
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected MCSToolTestPacket()
        {
            Debug.Assert( Category == Constants.CATEGORY_TEST );
        }
    }
    #endregion

    //=========================================================================
    #region 背景色
    /// <summary>
    /// 背景色
    /// <para>
    /// ビューアの背景色を変更するパケットです。パケット圧縮のサンプルになっており、
    /// defineにより圧縮の有無を切り替えられます。
    /// </para>
    /// </summary>
    public class MCSEfbColorPacket : MCSToolTestPacket
    {
        /// <summary>
        /// パケット送信
        /// </summary>
        public static void Send( byte r, byte g, byte b )
        {
            if ( !MCSManager.IsConnected ) { return; }
            SendPacket( new MCSEfbColorPacket( r, g, b ) );
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public override MCSPacketHeader Write( ProtocolSoundWriter writer )
        {
            MCSPacketHeader header = CreateHeader();
            header.Write( writer );
            ToolTestEfbColor packet = new ToolTestEfbColor();
            packet.r = _r;
            packet.g = _g;
            packet.b = _b;
            packet.a = 255;
            packet.Write( writer );
            return header;
        }

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress( MCSToolPacket packet )
        {
            // 同じ種類のパケットが続けて書き込まれた場合、
            // 値を上書きして後から追加されたパケットを破棄する。
#if true
            // 圧縮有り
            MCSEfbColorPacket target = packet as MCSEfbColorPacket;
            if ( target == null ) { return false; }
            _r = target._r;
            _g = target._g;
            _b = target._b;
            _compressCount++;
            return true;
#else
            // 圧縮無し
            return false;
#endif
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_TEST_EFB_COLOR; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolTestEfbColor.StructSize; }
        }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return base.ToString() +
                string.Format( " ( {0,3} {1,3} {2,3} ) CompressRate {3:f2}%",
                _r, _g, _b, 100f / _compressCount );
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected MCSEfbColorPacket( byte r, byte g, byte b )
        {
            _compressCount = 1;
            _r = r;
            _g = g;
            _b = b;
        }

        // 圧縮カウント
        private int _compressCount;
        // 赤
        private byte _r;
        // 緑
        private byte _g;
        // 青
        private byte _b;
    }
    #endregion

    //=========================================================================
}
